import React, { useState, useEffect } from 'react';
import { auctionAPI, AuctionSession, auctionUtils } from '../../api/auctions';
import { negotiationApi } from '../../api/negotiation.api';
import { Trophy, XCircle, TrendingUp, Calendar, DollarSign, Target } from 'lucide-react';

interface HistoryItem {
  id: string;
  type: 'auction' | 'negotiation';
  title: string;
  finalAmount: number;
  result: 'won' | 'lost' | 'accepted' | 'rejected';
  date: string;
  savings?: number;
}

export function BidsNegotiationsHistory() {
  const [historyItems, setHistoryItems] = useState<HistoryItem[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchHistory();
  }, []);

  const fetchHistory = async () => {
    try {
      setError(null);
      const items: HistoryItem[] = [];

      // Fetch completed auctions
      try {
        const auctionsResult = await auctionAPI.getActiveAuctions(1, 100);
        const completedAuctions = (auctionsResult.auctions || []).filter(
          (a: AuctionSession) => a.status === 'completed'
        );
        
        completedAuctions.forEach((auction: AuctionSession) => {
          items.push({
            id: auction.session_id,
            type: 'auction',
            title: auction.title,
            finalAmount: auction.current_bid || auction.starting_bid,
            result: 'lost', // Default to lost, would need API to check if user won
            date: auction.end_time,
          });
        });
      } catch (err) {
        console.error('Error fetching auction history:', err);
      }

      // Fetch completed negotiations
      try {
        const negotiations = await negotiationApi.listNegotiations();
        const completedNegotiations = (negotiations || []).filter(
          (n: any) => n.status === 'ACCEPTED' || n.status === 'REJECTED'
        );
        
        completedNegotiations.forEach((negotiation: any) => {
          const savings = negotiation.asking_price && negotiation.current_offer
            ? negotiation.asking_price - negotiation.current_offer
            : 0;
          
          items.push({
            id: negotiation.negotiation_id,
            type: 'negotiation',
            title: negotiation.property_title || `Property ${negotiation.property_id?.slice(0, 8)}`,
            finalAmount: negotiation.current_offer || 0,
            result: negotiation.status === 'ACCEPTED' ? 'accepted' : 'rejected',
            date: negotiation.last_activity || negotiation.created_at,
            savings: savings > 0 ? savings : undefined,
          });
        });
      } catch (err) {
        console.error('Error fetching negotiation history:', err);
      }

      // Sort by date (most recent first)
      items.sort((a, b) => new Date(b.date).getTime() - new Date(a.date).getTime());
      setHistoryItems(items);
    } catch (err) {
      console.error('Error fetching history:', err);
      setError('Failed to load history');
    } finally {
      setIsLoading(false);
    }
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-IN', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    });
  };

  const getResultIcon = (result: string) => {
    switch (result) {
      case 'won':
      case 'accepted':
        return <Trophy className="w-5 h-5 text-green-600" />;
      case 'lost':
      case 'rejected':
        return <XCircle className="w-5 h-5 text-red-600" />;
      default:
        return null;
    }
  };

  const getResultBadge = (result: string) => {
    switch (result) {
      case 'won':
        return <span className="px-2 py-1 bg-green-100 text-green-800 text-xs font-medium rounded-full">Won</span>;
      case 'lost':
        return <span className="px-2 py-1 bg-red-100 text-red-800 text-xs font-medium rounded-full">Lost</span>;
      case 'accepted':
        return <span className="px-2 py-1 bg-green-100 text-green-800 text-xs font-medium rounded-full">Accepted</span>;
      case 'rejected':
        return <span className="px-2 py-1 bg-red-100 text-red-800 text-xs font-medium rounded-full">Rejected</span>;
      default:
        return null;
    }
  };

  // Calculate analytics
  const analytics = {
    totalAuctions: historyItems.filter(i => i.type === 'auction').length,
    auctionsWon: historyItems.filter(i => i.type === 'auction' && i.result === 'won').length,
    totalNegotiations: historyItems.filter(i => i.type === 'negotiation').length,
    negotiationsAccepted: historyItems.filter(i => i.type === 'negotiation' && i.result === 'accepted').length,
    totalSavings: historyItems.reduce((sum, item) => sum + (item.savings || 0), 0),
    successRate: historyItems.length > 0
      ? Math.round(
          ((historyItems.filter(i => i.result === 'won' || i.result === 'accepted').length) / historyItems.length) * 100
        )
      : 0,
  };

  return (
    <div className="space-y-6">
      {/* Analytics Cards */}
      <div className="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-6 gap-4">
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex flex-col">
            <div className="flex items-center justify-between mb-2">
              <Calendar className="w-5 h-5 text-blue-500" />
            </div>
            <p className="text-sm text-gray-600">Total Auctions</p>
            <p className="text-2xl font-bold text-gray-900">{analytics.totalAuctions}</p>
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex flex-col">
            <div className="flex items-center justify-between mb-2">
              <Trophy className="w-5 h-5 text-green-500" />
            </div>
            <p className="text-sm text-gray-600">Auctions Won</p>
            <p className="text-2xl font-bold text-green-600">{analytics.auctionsWon}</p>
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex flex-col">
            <div className="flex items-center justify-between mb-2">
              <TrendingUp className="w-5 h-5 text-blue-500" />
            </div>
            <p className="text-sm text-gray-600">Negotiations</p>
            <p className="text-2xl font-bold text-gray-900">{analytics.totalNegotiations}</p>
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex flex-col">
            <div className="flex items-center justify-between mb-2">
              <Trophy className="w-5 h-5 text-green-500" />
            </div>
            <p className="text-sm text-gray-600">Accepted</p>
            <p className="text-2xl font-bold text-green-600">{analytics.negotiationsAccepted}</p>
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex flex-col">
            <div className="flex items-center justify-between mb-2">
              <DollarSign className="w-5 h-5 text-yellow-500" />
            </div>
            <p className="text-sm text-gray-600">Total Savings</p>
            <p className="text-xl font-bold text-yellow-600">{formatCurrency(analytics.totalSavings)}</p>
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex flex-col">
            <div className="flex items-center justify-between mb-2">
              <Target className="w-5 h-5 text-purple-500" />
            </div>
            <p className="text-sm text-gray-600">Success Rate</p>
            <p className="text-2xl font-bold text-purple-600">{analytics.successRate}%</p>
          </div>
        </div>
      </div>

      {/* Error State */}
      {error && (
        <div className="bg-red-50 border border-red-200 rounded-lg p-4">
          <div className="flex items-center gap-2 text-red-800">
            <XCircle className="w-5 h-5" />
            <p>{error}</p>
          </div>
        </div>
      )}

      {/* Loading State */}
      {isLoading && (
        <div className="flex justify-center items-center py-12">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
          <span className="ml-3 text-gray-600">Loading history...</span>
        </div>
      )}

      {/* History Timeline */}
      {!isLoading && !error && (
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Activity Timeline</h3>
          
          {historyItems.length > 0 ? (
            <div className="space-y-4">
              {historyItems.map((item) => (
                <div
                  key={item.id}
                  className="flex items-start gap-4 p-4 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors"
                >
                  <div className="flex-shrink-0 mt-1">
                    {getResultIcon(item.result)}
                  </div>
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-2 mb-1">
                      <h4 className="text-sm font-semibold text-gray-900 truncate">{item.title}</h4>
                      {getResultBadge(item.result)}
                      <span className="px-2 py-1 bg-blue-100 text-blue-800 text-xs font-medium rounded-full">
                        {item.type === 'auction' ? 'Auction' : 'Negotiation'}
                      </span>
                    </div>
                    <div className="flex items-center gap-4 text-sm text-gray-600">
                      <span>Final Amount: <span className="font-medium text-gray-900">{formatCurrency(item.finalAmount)}</span></span>
                      {item.savings && (
                        <span className="text-green-600">
                          Saved: <span className="font-medium">{formatCurrency(item.savings)}</span>
                        </span>
                      )}
                      <span className="text-gray-500">{formatDate(item.date)}</span>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          ) : (
            <div className="text-center py-12">
              <Calendar className="w-16 h-16 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No History Yet</h3>
              <p className="text-gray-600">Your completed auctions and negotiations will appear here.</p>
            </div>
          )}
        </div>
      )}
    </div>
  );
}
