import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { auctionAPI, AuctionSession, auctionUtils } from '../../api/auctions';
import { useAuction } from '../../hooks/useAuction';
import { Clock, TrendingUp, Trophy, XCircle, Eye, Gavel } from 'lucide-react';

type FilterType = 'all' | 'active' | 'won' | 'lost' | 'upcoming';

interface AuctionCardProps {
  auction: AuctionSession;
  onViewDetails: (sessionId: string) => void;
}

const AuctionBidCard: React.FC<AuctionCardProps> = ({ auction, onViewDetails }) => {
  const { currentBid, timeRemaining, isRegistered, session } = useAuction(auction.session_id);
  const displayBid = currentBid || auction.current_bid || auction.starting_bid;

  const getStatusBadge = () => {
    if (auction.status === 'active') {
      return <span className="px-2 py-1 bg-green-100 text-green-800 text-xs font-medium rounded-full flex items-center gap-1">
        <span className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></span>
        Live
      </span>;
    }
    if (auction.status === 'upcoming') {
      return <span className="px-2 py-1 bg-blue-100 text-blue-800 text-xs font-medium rounded-full">Upcoming</span>;
    }
    if (auction.status === 'completed') {
      return <span className="px-2 py-1 bg-gray-100 text-gray-800 text-xs font-medium rounded-full">Ended</span>;
    }
    return null;
  };

  return (
    <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-5 hover:shadow-md transition-shadow">
      <div className="flex justify-between items-start mb-3">
        <div className="flex-1">
          <h3 className="text-lg font-semibold text-gray-900 mb-1 line-clamp-1">{auction.title}</h3>
          {getStatusBadge()}
        </div>
      </div>

      <div className="space-y-2 mb-4">
        <div className="flex justify-between text-sm">
          <span className="text-gray-600">Starting Bid:</span>
          <span className="font-medium">{auctionUtils.formatCurrency(auction.starting_bid)}</span>
        </div>
        <div className="flex justify-between text-sm">
          <span className="text-gray-600">Current Bid:</span>
          <span className="font-semibold text-green-600">{auctionUtils.formatCurrency(displayBid)}</span>
        </div>
        {auction.status === 'active' && (
          <div className="flex justify-between text-sm">
            <span className="text-gray-600">Time Left:</span>
            <span className="font-medium text-red-600 flex items-center gap-1">
              <Clock className="w-3 h-3" />
              {timeRemaining || 'Calculating...'}
            </span>
          </div>
        )}
        <div className="flex justify-between text-sm">
          <span className="text-gray-600">Total Bids:</span>
          <span className="font-medium">{auction.total_bids || 0}</span>
        </div>
      </div>

      <div className="flex gap-2">
        <button
          onClick={() => onViewDetails(auction.session_id)}
          className="flex-1 bg-blue-600 text-white py-2 px-4 rounded-md hover:bg-blue-700 transition-colors text-sm font-medium flex items-center justify-center gap-2"
        >
          <Eye className="w-4 h-4" />
          View Details
        </button>
      </div>
    </div>
  );
};

export function BuyerBidsManagement() {
  const navigate = useNavigate();
  const [auctions, setAuctions] = useState<AuctionSession[]>([]);
  const [filteredAuctions, setFilteredAuctions] = useState<AuctionSession[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [activeFilter, setActiveFilter] = useState<FilterType>('all');

  useEffect(() => {
    fetchAuctions();
    const interval = setInterval(fetchAuctions, 30000); // Refresh every 30 seconds
    return () => clearInterval(interval);
  }, []);

  useEffect(() => {
    filterAuctions();
  }, [auctions, activeFilter]);

  const fetchAuctions = async () => {
    try {
      setError(null);
      const result = await auctionAPI.getActiveAuctions(1, 100);
      setAuctions(result.auctions || []);
    } catch (err) {
      console.error('Error fetching auctions:', err);
      setError('Failed to load auctions');
    } finally {
      setIsLoading(false);
    }
  };

  const filterAuctions = () => {
    let filtered = [...auctions];
    
    switch (activeFilter) {
      case 'active':
        filtered = filtered.filter(a => a.status === 'active');
        break;
      case 'upcoming':
        filtered = filtered.filter(a => a.status === 'upcoming');
        break;
      case 'won':
        filtered = filtered.filter(a => a.status === 'completed');
        break;
      case 'lost':
        filtered = filtered.filter(a => a.status === 'completed');
        break;
      default:
        break;
    }
    
    setFilteredAuctions(filtered);
  };

  const handleViewDetails = (sessionId: string) => {
    navigate(`/property-details/${sessionId}`);
  };

  const stats = {
    total: auctions.length,
    active: auctions.filter(a => a.status === 'active').length,
    upcoming: auctions.filter(a => a.status === 'upcoming').length,
    completed: auctions.filter(a => a.status === 'completed').length,
  };

  return (
    <div className="space-y-6">
      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Total Auctions</p>
              <p className="text-2xl font-bold text-gray-900">{stats.total}</p>
            </div>
            <Gavel className="w-8 h-8 text-blue-500" />
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Active Bids</p>
              <p className="text-2xl font-bold text-green-600">{stats.active}</p>
            </div>
            <TrendingUp className="w-8 h-8 text-green-500" />
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Upcoming</p>
              <p className="text-2xl font-bold text-blue-600">{stats.upcoming}</p>
            </div>
            <Clock className="w-8 h-8 text-blue-500" />
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Completed</p>
              <p className="text-2xl font-bold text-gray-600">{stats.completed}</p>
            </div>
            <Trophy className="w-8 h-8 text-yellow-500" />
          </div>
        </div>
      </div>

      {/* Filters */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
        <div className="flex flex-wrap gap-2">
          {(['all', 'active', 'upcoming', 'won', 'lost'] as FilterType[]).map((filter) => (
            <button
              key={filter}
              onClick={() => setActiveFilter(filter)}
              className={`px-4 py-2 rounded-md text-sm font-medium transition-colors ${
                activeFilter === filter
                  ? 'bg-[#0056D2] text-white shadow-md'
                  : 'bg-white text-gray-700 border border-gray-300 hover:bg-gray-50'
              }`}
            >
              {filter.charAt(0).toUpperCase() + filter.slice(1)}
            </button>
          ))}
        </div>
      </div>

      {/* Error State */}
      {error && (
        <div className="bg-red-50 border border-red-200 rounded-lg p-4">
          <div className="flex items-center gap-2 text-red-800">
            <XCircle className="w-5 h-5" />
            <p>{error}</p>
          </div>
        </div>
      )}

      {/* Loading State */}
      {isLoading && (
        <div className="flex justify-center items-center py-12">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
          <span className="ml-3 text-gray-600">Loading auctions...</span>
        </div>
      )}

      {/* Auctions Grid */}
      {!isLoading && !error && (
        <>
          {filteredAuctions.length > 0 ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {filteredAuctions.map((auction) => (
                <AuctionBidCard
                  key={auction.session_id}
                  auction={auction}
                  onViewDetails={handleViewDetails}
                />
              ))}
            </div>
          ) : (
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-12 text-center">
              <Gavel className="w-16 h-16 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No Auctions Found</h3>
              <p className="text-gray-600">
                {activeFilter === 'all' 
                  ? 'There are no auctions available at the moment.'
                  : `No ${activeFilter} auctions found.`}
              </p>
            </div>
          )}
        </>
      )}
    </div>
  );
}
