import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { 
  Plus, FileText, Clock, CheckCircle, AlertCircle, RefreshCw, Search, 
  Eye, Edit, Trash2, Download, Upload, FileSignature, Stamp, Shield,
  Calendar, User, Mail, Phone, MapPin, Home, DollarSign
} from 'lucide-react';
import { ContractList } from '../contracts/ContractList';
import { ContractForm } from '../contracts/ContractForm';
import { EStampCalculator } from '../contracts/EStampCalculator';

interface Contract {
  id: string;
  title: string;
  type: string;
  status: 'draft' | 'review' | 'approval' | 'active' | 'completed';
  property?: string;
  property_location?: string;
  property_type?: string;
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
  sale_price?: number;
  monthly_rent?: number;
  property_email?: string;
  property_owner?: string;
  parties: string[];
  party_emails?: string[];
  amount?: number;
  start_date?: string;
  end_date?: string;
  created_at: string;
  next_milestone?: string;
  progress: number;
}

export const BuyerContractManagement: React.FC = () => {
  const [contracts, setContracts] = useState<Contract[]>([]);
  const [filteredContracts, setFilteredContracts] = useState<Contract[]>([]);
  const [showForm, setShowForm] = useState(false);
  const [loading, setLoading] = useState(true);
  const [refreshing, setRefreshing] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [activeTab, setActiveTab] = useState<'all' | 'active' | 'pending' | 'completed'>('all');
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState<string>('all');

  useEffect(() => {
    fetchContracts();
  }, []);

  useEffect(() => {
    filterContracts();
  }, [contracts, activeTab, searchTerm, statusFilter]);

  const fetchContracts = async (isRefresh = false) => {
    try {
      if (isRefresh) {
        setRefreshing(true);
      } else {
        setLoading(true);
      }
      setError(null);
      
      const token = localStorage.getItem('access_token');
      if (!token) {
        setContracts([]);
        setLoading(false);
        setRefreshing(false);
        return;
      }

      // Get user ID from token
      const payload = JSON.parse(atob(token.split('.')[1]));
      const userId = payload.user_id;

      if (!userId) {
        setError('User ID not found in token');
        setContracts([]);
        setLoading(false);
        setRefreshing(false);
        return;
      }

      // Try user dashboard endpoint first
      try {
        const response = await fetch(`http://localhost:8090/api/user-dashboard/contracts?user_id=${userId}`, {
          headers: {
            'Content-Type': 'application/json'
          }
        });
        
        if (response.ok) {
          const contractsData = await response.json();
          // Ensure we always have an array
          const contractsArray = Array.isArray(contractsData) ? contractsData : 
                                Array.isArray(contractsData?.contracts) ? contractsData.contracts : [];
          setContracts(contractsArray);
        } else {
          // Fallback to contracts endpoint
          const fallbackResponse = await fetch(`http://localhost:8090/api/contracts/user/${userId}`, {
            headers: {
              'Content-Type': 'application/json'
            }
          });
          
          if (fallbackResponse.ok) {
            const fallbackData = await fallbackResponse.json();
            const contractsArray = Array.isArray(fallbackData) ? fallbackData : 
                                  Array.isArray(fallbackData?.contracts) ? fallbackData.contracts : [];
            setContracts(contractsArray);
          } else {
            setContracts([]);
          }
        }
      } catch (fetchError) {
        console.error('Error fetching contracts:', fetchError);
        setContracts([]);
        setError('Failed to load contracts. Please check your connection.');
      }
    } catch (error) {
      console.error('Error in fetchContracts:', error);
      setContracts([]);
      setError('Failed to load contracts. Please try again.');
    } finally {
      setLoading(false);
      setRefreshing(false);
    }
  };

  const filterContracts = () => {
    let filtered = contracts;

    // Filter by tab
    switch (activeTab) {
      case 'active':
        filtered = filtered.filter(c => c.status === 'active');
        break;
      case 'pending':
        filtered = filtered.filter(c => ['draft', 'review', 'approval', 'pending_signatures'].includes(c.status));
        break;
      case 'completed':
        filtered = filtered.filter(c => c.status === 'completed');
        break;
    }

    // Filter by status
    if (statusFilter !== 'all') {
      filtered = filtered.filter(c => c.status === statusFilter);
    }

    // Filter by search term
    if (searchTerm) {
      filtered = filtered.filter(c => 
        c.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
        c.type.toLowerCase().includes(searchTerm.toLowerCase()) ||
        c.property?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        c.parties.some(party => party.toLowerCase().includes(searchTerm.toLowerCase()))
      );
    }

    setFilteredContracts(filtered);
  };

  const stats = {
    total: contracts.length,
    active: contracts.filter(c => c.status === 'active').length,
    pending: contracts.filter(c => ['draft', 'review', 'approval', 'pending_signatures'].includes(c.status)).length,
    completed: contracts.filter(c => c.status === 'completed').length
  };

  if (showForm) {
    return <ContractForm onClose={() => setShowForm(false)} onSave={fetchContracts} />;
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <div>
          <h2 className="text-3xl font-bold text-gray-900">Contract Management</h2>
          <p className="text-gray-600 mt-2">Manage your property contracts, agreements, and digital signatures</p>
        </div>
        <div className="flex gap-3">
          <Button 
            variant="outline" 
            onClick={() => fetchContracts(true)}
            disabled={refreshing}
            className="flex items-center gap-2"
          >
            <RefreshCw className={`w-4 h-4 ${refreshing ? 'animate-spin' : ''}`} />
            {refreshing ? 'Refreshing...' : 'Refresh'}
          </Button>
          <Button 
            onClick={() => setShowForm(true)} 
            className="bg-blue-600 hover:bg-blue-700 text-black font-semibold px-6 py-3 shadow-lg border border-gray-300"
          >
            <Plus className="w-5 h-5 mr-2 text-black" />
            <span className="text-black">New Contract</span>
          </Button>
        </div>
      </div>

      {/* Error Message */}
      {error && (
        <Card className="border-red-200 bg-red-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <AlertCircle className="w-5 h-5 text-red-600" />
              <p className="text-red-800 font-medium">Error</p>
            </div>
            <p className="text-red-700 mt-1">{error}</p>
            <Button 
              variant="outline" 
              size="sm" 
              onClick={() => fetchContracts(true)}
              className="mt-2 text-red-600 border-red-300 hover:bg-red-50"
            >
              Try Again
            </Button>
          </CardContent>
        </Card>
      )}

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <Card className="hover:shadow-md transition-shadow">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Total Contracts</p>
                <p className="text-3xl font-bold text-gray-900">{stats.total}</p>
              </div>
              <div className="p-3 bg-blue-100 rounded-full">
                <FileText className="w-6 h-6 text-blue-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="hover:shadow-md transition-shadow">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Active</p>
                <p className="text-3xl font-bold text-green-600">{stats.active}</p>
              </div>
              <div className="p-3 bg-green-100 rounded-full">
                <CheckCircle className="w-6 h-6 text-green-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="hover:shadow-md transition-shadow">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Pending</p>
                <p className="text-3xl font-bold text-yellow-600">{stats.pending}</p>
              </div>
              <div className="p-3 bg-yellow-100 rounded-full">
                <Clock className="w-6 h-6 text-yellow-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="hover:shadow-md transition-shadow">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Completed</p>
                <p className="text-3xl font-bold text-purple-600">{stats.completed}</p>
              </div>
              <div className="p-3 bg-purple-100 rounded-full">
                <CheckCircle className="w-6 h-6 text-purple-600" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Filters and Search */}
      <Card>
        <CardContent className="p-6">
          <div className="flex flex-col md:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                <Input
                  placeholder="Search contracts by title, type, property, or parties..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10"
                />
              </div>
            </div>
            <select
              value={statusFilter}
              onChange={(e) => setStatusFilter(e.target.value)}
              className="w-full md:w-48 px-3 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500"
            >
              <option value="all">All Status</option>
              <option value="draft">Draft</option>
              <option value="pending_signatures">Pending Signatures</option>
              <option value="active">Active</option>
              <option value="completed">Completed</option>
              <option value="signed">Signed</option>
            </select>
          </div>
        </CardContent>
      </Card>

      {/* Tabs */}
      <div className="border-b border-gray-200">
        <nav className="-mb-px flex space-x-8">
          {[
            { key: 'all', label: `All (${stats.total})` },
            { key: 'active', label: `Active (${stats.active})` },
            { key: 'pending', label: `Pending (${stats.pending})` },
            { key: 'completed', label: `Completed (${stats.completed})` }
          ].map((tab) => (
            <button
              key={tab.key}
              onClick={() => setActiveTab(tab.key as any)}
              className={`py-2 px-1 border-b-2 font-medium text-sm ${
                activeTab === tab.key
                  ? 'border-blue-500 text-blue-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
              }`}
            >
              {tab.label}
            </button>
          ))}
        </nav>
      </div>

      {/* Contract List */}
      <Card>
        <CardHeader>
          <CardTitle className="text-xl font-semibold text-gray-900">
            {activeTab === 'all' && 'All Contracts'}
            {activeTab === 'active' && 'Active Contracts'}
            {activeTab === 'pending' && 'Pending Contracts'}
            {activeTab === 'completed' && 'Completed Contracts'}
          </CardTitle>
        </CardHeader>
        <CardContent className="p-6">
          {loading ? (
            <div className="text-center py-8">
              <RefreshCw className="w-8 h-8 animate-spin mx-auto mb-4 text-gray-400" />
              <p className="text-gray-500">Loading contracts...</p>
            </div>
          ) : filteredContracts.length === 0 ? (
            <div className="text-center py-12">
              <div className="mx-auto w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mb-4">
                <FileText className="w-12 h-12 text-gray-400" />
              </div>
              <h3 className="text-lg font-medium text-gray-900 mb-2">No contracts found</h3>
              <p className="text-gray-500 mb-6">
                {searchTerm || statusFilter !== 'all' 
                  ? 'No contracts match your current filters.' 
                  : 'Create your first contract to get started.'
                }
              </p>
              {!searchTerm && statusFilter === 'all' && (
                <Button 
                  onClick={() => setShowForm(true)}
                  className="bg-blue-600 hover:bg-blue-700 text-black font-semibold px-8 py-3 shadow-lg border border-gray-300"
                >
                  <Plus className="w-5 h-5 mr-2 text-black" />
                  <span className="text-black">Create Your First Contract</span>
                </Button>
              )}
            </div>
          ) : (
            <ContractList 
              contracts={filteredContracts} 
              onRefresh={() => fetchContracts(true)}
            />
          )}
        </CardContent>
      </Card>

      {/* Quick Actions */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <Card className="hover:shadow-md transition-shadow cursor-pointer">
          <CardContent className="p-6 text-center">
            <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <FileSignature className="w-6 h-6 text-blue-600" />
            </div>
            <h3 className="text-lg font-semibold text-gray-900 mb-2">E-Sign Documents</h3>
            <p className="text-gray-600 text-sm mb-4">Digitally sign contracts and agreements with legal validity</p>
            <Button variant="outline" className="w-full">
              Start E-Signing
            </Button>
          </CardContent>
        </Card>

        <Card className="hover:shadow-md transition-shadow cursor-pointer">
          <CardContent className="p-6 text-center">
            <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <Stamp className="w-6 h-6 text-green-600" />
            </div>
            <h3 className="text-lg font-semibold text-gray-900 mb-2">E-Stamp Calculator</h3>
            <p className="text-gray-600 text-sm mb-4">Calculate stamp duty and generate e-stamps for your documents</p>
            <Button variant="outline" className="w-full">
              Calculate Stamp Duty
            </Button>
          </CardContent>
        </Card>

        <Card className="hover:shadow-md transition-shadow cursor-pointer">
          <CardContent className="p-6 text-center">
            <div className="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <Shield className="w-6 h-6 text-purple-600" />
            </div>
            <h3 className="text-lg font-semibold text-gray-900 mb-2">Escrow Services</h3>
            <p className="text-gray-600 text-sm mb-4">Secure payment handling for property transactions</p>
            <Button variant="outline" className="w-full">
              Setup Escrow
            </Button>
          </CardContent>
        </Card>
      </div>

      {/* E-Stamp Calculator */}
      <EStampCalculator />
    </div>
  );
};