import React, { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { BuyerSidebar } from "./BuyerSidebar";
import { BuyerHeader } from "./BuyerHeader";
import { DashboardOverview } from "./DashboardOverview";
import { WishlistManagement } from "./WishlistManagement";
import { SavedProperties } from "./SavedProperties";
import { SearchHistory } from "./SearchHistory";
import { InquiryManagement } from "./InquiryManagement";
import { ScheduledVisits } from "./ScheduledVisits";
import PropertyComparison from "./PropertyComparison";
import ProfessionalPropertyComparison from "./ProfessionalPropertyComparison";
import { ProfileSettings } from "./ProfileSettings";
import { SupportCenter } from "./SupportCenter";
import { PropertyVisitManagement } from "./PropertyVisitManagement";
import { MyProperties } from "./MyProperties";
import { PropertyGenerator } from "./PropertyGenerator";
import { PaymentsTransactions } from "./PaymentsTransactions";

import ProfileCompletionModal from "../Profile/ProfileCompletionModal";
import { getProfile } from "../../api/profile";
import { BuyerInsightsPage } from "../../pages/future-insights/BuyerInsightsPage";
import { PricePrediction } from "../../pages/price-prediction/PricePrediction";
import { TransactionList } from "../transactions/TransactionList";
import { AgreementList } from "../agreements/AgreementList";
import { BuyerContractManagement } from "./BuyerContractManagement";
import { EscrowDashboard } from "../escrow/EscrowDashboard";
import { BehaviorAnalytics } from "./BehaviorAnalytics";
import { MyBadges } from "../../pages/MyBadges";
import { UpcomingVisits } from "../visits/UpcomingVisits";
import { MyItinerary } from "./MyItinerary";
import { MyFeatureFlags } from "./MyFeatureFlags";
import { ESignEStamp } from "../../pages/services/ESignEStamp";
import { ContractList } from "../contracts/ContractList";
import { EarnBadge } from "./EarnBadge";

import { TenantView } from "./TenantView";
import { RecommendedProperties } from "./RecommendedProperties";
import { UserCompliance } from "../compliance/UserCompliance";
import InvoicesPage from "./Finance/InvoicesPage";
import UsagePage from "./Finance/UsagePage";
import { RentalYieldDashboard } from "./RentalYieldDashboard";
import { BuyerRecommendations } from "./BuyerRecommendations";
import { BuyerBidsManagement } from "./BuyerBidsManagement";
import { BuyerNegotiations } from "./BuyerNegotiations";
import { BidsNegotiationsHistory } from "./BidsNegotiationsHistory";
import { BuyerSystemHealth } from "./BuyerSystemHealth";
import PrivacySettings from "../PrivacySettings";
import SystemHealthDashboard from '../../system-health/components/SystemHealthDashboard';
import { SupportTickets } from './SupportTickets';
import GrievancePage from '../../pages/GrievancePage';
import BuyerWallet from './BuyerWallet';
import { UserPricingWithCoupons } from './UserPricingWithCoupons';
import { UserScholarships } from './UserScholarships';
import UserEntitlementsDashboard from '../entitlements/UserEntitlementsDashboard';

export function BuyerDashboard() {
  const [buyerName, setBuyerName] = useState("");
  const [activeSection, setActiveSection] = useState("Dashboard");
  const [isSidebarCollapsed, setIsSidebarCollapsed] = useState(false);
  const [showProfileModal, setShowProfileModal] = useState(false);
  const [activeTab, setActiveTab] = useState<string>('');
  const navigate = useNavigate();

  const tabConfigs: Record<string, { id: string; label: string }[]> = {
    Properties: [
      { id: 'recommendations', label: 'AI Recommendations' },
      { id: 'search', label: 'Search History' },
      { id: 'saved', label: 'Saved Properties' },
      { id: 'wishlist', label: 'Wishlist' },
      { id: 'comparison', label: 'AI Comparison' }
    ],
    BidsNegotiations: [
      { id: 'bids', label: 'My Bids' },
      { id: 'negotiations', label: 'Negotiations' },
      { id: 'history', label: 'History' }
    ],
    Visits: [
      { id: 'upcoming', label: 'Upcoming Visits' },
      { id: 'requests', label: 'Visit Requests' },
      { id: 'itineraries', label: 'My Itineraries' },
      { id: 'enquiries', label: 'Enquiries' },
      { id: 'badges', label: 'My Badges' }
    ],
    Insights: [
      { id: 'price', label: 'Price Prediction' },
      { id: 'market', label: 'Market Insights' },
      { id: 'analytics', label: 'Behavior Analytics' },
      { id: 'generator', label: 'Reports' }
    ],
    Documents: [
      { id: 'contracts', label: 'Contracts & eSign' },
      { id: 'agreements', label: 'Digital Agreements' }
    ],
    Settings: [
      { id: 'profile', label: 'Profile Settings' },
      { id: 'support', label: 'Support Center' }
    ]
  };

  useEffect(() => {
    if (tabConfigs[activeSection]) {
      setActiveTab(tabConfigs[activeSection][0].id);
    }
  }, [activeSection]);

  useEffect(() => {
    const urlParams = new URLSearchParams(window.location.search);
    const section = urlParams.get('section');
    const tab = urlParams.get('tab');
    if (tab === 'badges') {
      setActiveSection('MyBadges');
    } else if (tab === 'itineraries') {
      setActiveSection('MyItineraries');
    } else if (section) {
      setActiveSection(section);
    }
  }, [window.location.search]);

  useEffect(() => {
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const token = localStorage.getItem('access_token');
    const isRoleSwitching = sessionStorage.getItem('original_login_role');

    // Allow access if role switching is active
    if (!token && !isRoleSwitching) {
      navigate("/login");
      return;
    }

    setBuyerName(user.first_name || "Buyer");
  }, [navigate]);

  // Profile completion modal logic
  useEffect(() => {
    const checkProfileCompletion = async () => {
      const accessToken = localStorage.getItem('access_token');
      const agentToken = localStorage.getItem('agentToken');
      const sellerToken = localStorage.getItem('sellerToken');
      const isRoleSwitching = sessionStorage.getItem('original_login_role');
      
      // Skip for agents/sellers or when role switching
      if (agentToken || sellerToken || isRoleSwitching) {
        return;
      }

      if (!accessToken || localStorage.getItem('profile_modal_shown_this_login')) {
        return;
      }

      const profileCompleted = localStorage.getItem('profile_completed') === 'true';
      if (profileCompleted) {
        return;
      }

      const userStr = localStorage.getItem('user') || '{}';
      let user = {};
      try {
        user = JSON.parse(userStr);
      } catch (e) {
        console.warn('Failed to parse user data:', e);
      }

      if (user && (user as any).is_new_user) {
        setShowProfileModal(true);
        return;
      }

      try {
        const data = await getProfile();
        if (data && data.profile_completion_percentage < 100) {
          setShowProfileModal(true);
        }
      } catch (err) {
        console.error('Profile check failed:', err);
      }
    };

    checkProfileCompletion();
  }, []);

  const handleLogout = () => {
    localStorage.removeItem('access_token');
    localStorage.removeItem('user');
    localStorage.removeItem('profile_modal_shown_this_login');
    navigate('/');
  };

  const renderTabs = () => {
    const tabs = tabConfigs[activeSection];
    if (!tabs) return null;

    return (
      <div className="mb-6">
        <div className="flex flex-wrap gap-3">
          {tabs.map(tab => (
            <button
              key={tab.id}
              onClick={() => setActiveTab(tab.id)}
              className={`px-6 py-3 rounded-md text-sm font-medium transition-colors ${
                activeTab === tab.id
                  ? 'bg-[#0056D2] text-white shadow-md'
                  : 'bg-white text-gray-700 border border-gray-300 hover:bg-gray-50'
              }`}
            >
              {tab.label}
            </button>
          ))}
        </div>
      </div>
    );
  };

  const renderActiveSection = () => {
    if (activeSection === "Dashboard") {
      return <DashboardOverview />;
    }

    if (activeSection === "Properties") {
      return (
        <div className="space-y-6">
          {renderTabs()}
          {activeTab === 'recommendations' && <BuyerRecommendations />}
          {activeTab === 'search' && <SearchHistory />}
          {activeTab === 'saved' && <SavedProperties />}
          {activeTab === 'wishlist' && <WishlistManagement />}
          {activeTab === 'comparison' && <ProfessionalPropertyComparison />}
        </div>
      );
    }

    if (activeSection === "BidsNegotiations") {
      return (
        <div className="space-y-6">
          {renderTabs()}
          {activeTab === 'bids' && <BuyerBidsManagement />}
          {activeTab === 'negotiations' && <BuyerNegotiations />}
          {activeTab === 'history' && <BidsNegotiationsHistory />}
        </div>
      );
    }

    if (activeSection === "Visits") {
      return (
        <div className="space-y-6">
          {renderTabs()}
          {activeTab === 'upcoming' && <UpcomingVisits />}
          {activeTab === 'requests' && <PropertyVisitManagement />}
          {activeTab === 'itineraries' && <MyItinerary />}
          {activeTab === 'enquiries' && <InquiryManagement />}
          {activeTab === 'badges' && <MyBadges />}
        </div>
      );
    }

    if (activeSection === "Insights") {
      return (
        <div className="space-y-6">
          {renderTabs()}
          {activeTab === 'price' && <PricePrediction />}
          {activeTab === 'market' && <BuyerInsightsPage />}
          {activeTab === 'analytics' && <BehaviorAnalytics />}
          {activeTab === 'generator' && <PropertyGenerator />}
        </div>
      );
    }

    if (activeSection === "Documents") {
      return (
        <div className="space-y-6">
          {renderTabs()}
          {activeTab === 'contracts' && <BuyerContractManagement />}
          {activeTab === 'agreements' && <AgreementList />}
        </div>
      );
    }

    if (activeSection === "Payments") {
      return <PaymentsTransactions />;
    }

    if (activeSection === "Settings") {
      return (
        <div className="space-y-6">
          {renderTabs()}
          {activeTab === 'profile' && <ProfileSettings />}
          {activeTab === 'support' && <SupportCenter />}
        </div>
      );
    }

    return <DashboardOverview />;
  };

  return (
    <div className="min-h-screen bg-gray-50">
      <BuyerSidebar
        activeSection={activeSection}
        setActiveSection={setActiveSection}
        isSidebarCollapsed={isSidebarCollapsed}
        setIsSidebarCollapsed={setIsSidebarCollapsed}
      />

      <div className={`min-h-screen bg-gray-50 transition-all duration-300 ${isSidebarCollapsed ? 'ml-20' : 'ml-64'
        }`} style={{ marginLeft: isSidebarCollapsed ? '80px' : '256px' }}>
        <BuyerHeader buyerName={buyerName} onLogout={handleLogout} />

        <main className="p-8">
          {renderActiveSection()}
        </main>
      </div>

      <ProfileCompletionModal
        isOpen={showProfileModal}
        onClose={() => setShowProfileModal(false)}
        onComplete={() => setShowProfileModal(false)}
      />
    </div>
  );
}