import React, { useState, useEffect, useRef } from "react";
import { Button } from "../ui/button";
import { Bell, ChevronDown, User, LogOut, X } from "lucide-react";
import { getUserNotifications, markNotificationAsRead, deleteNotification, Notification } from "../../api/notifications";
import { useNavigate } from "react-router-dom";
import { RoleSwitcher } from '../RoleSwitcher';

interface BuyerHeaderProps {
  buyerName: string;
  onLogout: () => void;
}

// HTML entity decoding function
const decodeHtmlEntities = (text: string): string => {
  if (!text) return '';
  
  const textarea = document.createElement('textarea');
  textarea.innerHTML = text;
  const decoded = textarea.value;
  
  return decoded
    .replace(/&#39;/g, "'")
    .replace(/&quot;/g, '"')
    .replace(/&amp;/g, '&')
    .replace(/&lt;/g, '<')
    .replace(/&gt;/g, '>')
    .replace(/&nbsp;/g, ' ');
};

function UserNotificationBell({ userId }: { userId: string }) {
  const [isOpen, setIsOpen] = useState(false);
  const [notifications, setNotifications] = useState<Notification[]>([]);
  const [unreadCount, setUnreadCount] = useState(0);
  const [isLoading, setIsLoading] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const navigate = useNavigate();

  const fetchNotifications = async () => {
    try {
      setIsLoading(true);
      const data = await getUserNotifications(userId);
      setNotifications(data.notifications);
      setUnreadCount(data.unread_count);
    } catch (error) {
      console.error('Error fetching notifications:', error);
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    fetchNotifications();
    const interval = setInterval(fetchNotifications, 60000);
    return () => clearInterval(interval);
  }, [userId]);

  useEffect(() => {
    if (isOpen) fetchNotifications();
  }, [isOpen]);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setIsOpen(false);
      }
    };
    if (isOpen) {
      document.addEventListener('mousedown', handleClickOutside);
      return () => document.removeEventListener('mousedown', handleClickOutside);
    }
  }, [isOpen]);

  return (
    <div className="relative" ref={dropdownRef}>
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="relative inline-flex items-center justify-center w-10 h-10 p-1 rounded-full hover:bg-gray-100 transition"
      >
        <Bell className="w-5 h-5 text-gray-700" />
        {unreadCount > 0 && (
          <span className="absolute top-0 right-0 flex items-center justify-center h-5 min-w-[20px] px-1.5 text-[8px] font-semibold text-white bg-red-600 rounded-full border-2 border-white">
            {unreadCount > 99 ? '99+' : unreadCount}
          </span>
        )}
      </button>

      {isOpen && (
        <div className="absolute right-0 mt-2 w-96 bg-white rounded-2xl shadow-xl border border-gray-100 z-50">
          <div className="px-6 py-4 bg-gradient-to-r from-blue-50 to-indigo-50 border-b border-gray-100">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-3">
                <div className="w-8 h-8 bg-blue-500 rounded-lg flex items-center justify-center">
                  <Bell className="w-4 h-4 text-white" />
                </div>
                <div>
                  <h3 className="text-lg font-semibold text-gray-900">Notifications</h3>
                  <p className="text-xs text-gray-600">
                    {unreadCount === 0 ? "All caught up!" : `${unreadCount} unread`}
                  </p>
                </div>
              </div>
              <div className="flex items-center gap-2">
                {notifications.length > 0 && (
                  <button
                    onClick={() => {
                      setNotifications([]);
                      setUnreadCount(0);
                    }}
                    className="text-xs text-blue-600 hover:text-blue-700 font-medium px-2 py-1 rounded hover:bg-white/50"
                  >
                    Clear All
                  </button>
                )}
                <button
                  onClick={() => setIsOpen(false)}
                  className="p-2 text-gray-400 hover:text-gray-600 hover:bg-white/50 rounded-lg transition-all"
                >
                  <X className="w-4 h-4" />
                </button>
              </div>
            </div>
          </div>

          <div className="max-h-80 overflow-y-auto">
            {isLoading ? (
              <div className="p-8 flex justify-center">
                <div className="animate-spin rounded-full h-6 w-6 border-2 border-blue-500 border-t-transparent"></div>
              </div>
            ) : notifications.length === 0 ? (
              <div className="p-8 text-center">
                <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-3">
                  <Bell className="w-6 h-6 text-gray-400" />
                </div>
                <p className="text-sm text-gray-500">No notifications yet</p>
                <p className="text-xs text-gray-400 mt-1">We'll notify you about property updates</p>
              </div>
            ) : (
              <div className="divide-y divide-gray-50">
                {notifications.map((notification) => {
                  let timeStr = 'Just now';
                  try {
                    if (notification.time && 
                        notification.time !== null && 
                        notification.time !== undefined && 
                        notification.time !== '' && 
                        notification.time !== 'null' && 
                        notification.time !== 'undefined') {
                      const date = new Date(notification.time);
                      if (!isNaN(date.getTime()) && date.getTime() > 0 && date.getFullYear() > 1970) {
                        timeStr = new Intl.DateTimeFormat('en-US', {
                          month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit'
                        }).format(date);
                      }
                    }
                  } catch (e) {
                    console.warn('Invalid date format for notification:', notification.time, e);
                    timeStr = 'Just now';
                  }

                  return (
                    <div 
                      key={notification.id} 
                      className="p-4 hover:bg-gray-50 transition-colors cursor-pointer" 
                      onClick={async (e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        try {
                          if (!notification.read) {
                            await markNotificationAsRead(notification.id);
                            setNotifications(prev => 
                              prev.map(n => n.id === notification.id ? { ...n, read: true } : n)
                            );
                            setUnreadCount(prev => Math.max(0, prev - 1));
                          }
                          
                          if (notification.data?.property_id) {
                            navigate(`/property/${notification.data.property_id}`);
                            setIsOpen(false);
                          }
                        } catch (error) {
                          console.error('Error handling notification click:', error);
                        }
                      }}
                    >
                      <div className="flex items-start justify-between gap-3">
                        <div className="flex-1 min-w-0">
                          <div className="flex items-center gap-2 mb-1">
                            <h4 className="text-sm font-semibold text-gray-900 break-words">
                              {notification.title ? notification.title.replace(/&#39;/g, "'").replace(/&quot;/g, '"') : 'Notification'}
                            </h4>
                            {!notification.read && (
                              <div className="w-2 h-2 bg-blue-500 rounded-full flex-shrink-0"></div>
                            )}
                          </div>
                          <p className="text-sm text-gray-600 mb-2 break-words whitespace-pre-wrap">
                            {notification.message ? notification.message.replace(/&#39;/g, "'").replace(/&quot;/g, '"') : 'No message'}
                          </p>
                          <div className="flex items-center gap-2">
                            <span className="text-xs text-gray-400">{timeStr}</span>
                            <span className="text-xs px-2 py-0.5 rounded-full font-medium bg-blue-100 text-blue-700">
                              {notification.type}
                            </span>
                          </div>
                        </div>
                        <button
                          onClick={async (e) => {
                            e.stopPropagation();
                            try {
                              setNotifications(prev => prev.filter(n => n.id !== notification.id));
                              if (!notification.read) {
                                setUnreadCount(prev => Math.max(0, prev - 1));
                              }
                            } catch (error) {
                              console.error('Failed to delete notification:', error);
                            }
                          }}
                          className="p-1 text-gray-400 hover:text-gray-600 rounded-full transition-all"
                        >
                          <X className="w-3 h-3" />
                        </button>
                      </div>
                    </div>
                  );
                })}
              </div>
            )}
          </div>
        </div>
      )}
    </div>
  );
}

export function BuyerHeader({ buyerName, onLogout }: BuyerHeaderProps) {
  const [showProfileDropdown, setShowProfileDropdown] = useState(false);
  const [actualUsername, setActualUsername] = useState("");
  const [userEmail, setUserEmail] = useState("");
  const [userRole, setUserRole] = useState("User");

  useEffect(() => {
    // Use original email if switching dashboards
    const originalEmail = sessionStorage.getItem('original_email');
    const email = originalEmail || localStorage.getItem('login_email') || localStorage.getItem('user_email') || '';
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const role = user.user_role === 'admin' ? 'Admin' : 'User';
    
    let username = 'User';
    if (user.first_name) {
      username = user.first_name;
    } else if (email) {
      const emailPart = email.split('@')[0];
      // Remove numbers and special characters to get clean username
      username = emailPart.replace(/[0-9]/g, '').replace(/[^a-zA-Z]/g, '') || emailPart;
    } else if (buyerName && buyerName !== 'Buyer') {
      username = buyerName;
    }
    
    setUserEmail(email);
    setUserRole(role);
    setActualUsername(username);
  }, [buyerName]);

  const getUserInitial = () => {
    return actualUsername.charAt(0).toUpperCase();
  };

  return (
    <header className="bg-white shadow-sm border-b px-8 py-4">
      <div className="flex items-center justify-between">
        <h1 className="text-xl font-semibold text-gray-900">Welcome back, {actualUsername}!</h1>
        <div className="flex items-center gap-4">
          <RoleSwitcher currentRole="buyer" />
          
          <UserNotificationBell userId={(() => {
            const user = JSON.parse(localStorage.getItem('user') || '{}');
            return user.id || user.user_id || user.email || localStorage.getItem('login_email') || 'user';
          })()} />
          
          <div className="relative">
            <button
              onClick={() => setShowProfileDropdown(!showProfileDropdown)}
              className="flex items-center gap-2 px-3 py-2 rounded-lg hover:bg-gray-50 transition-all duration-200"
            >
              <div className="w-8 h-8 bg-blue-500 rounded-full flex items-center justify-center shadow-sm">
                <span className="text-white text-sm font-medium">{getUserInitial()}</span>
              </div>
              <div className="flex flex-col items-start">
                <span className="text-sm font-medium text-gray-900">{userEmail}</span>
                <span className="text-xs text-gray-500">{userRole}</span>
              </div>
              <ChevronDown className="w-4 h-4 text-gray-400" />
            </button>
            
            {showProfileDropdown && (
              <div className="absolute top-full right-0 mt-2 w-64 bg-white border border-gray-200 rounded-xl shadow-xl z-50">
                <div className="p-4 border-b border-gray-100">
                  <div className="flex items-center gap-3">
                    <div className="w-10 h-10 bg-blue-500 rounded-full flex items-center justify-center shadow-md">
                      <span className="text-white font-semibold">{getUserInitial()}</span>
                    </div>
                    <div className="flex-1 min-w-0">
                      <h3 className="font-semibold text-gray-900 truncate">{userEmail || actualUsername}</h3>
                      <p className="text-sm text-gray-500 truncate">{userRole} Account</p>
                    </div>
                  </div>
                </div>
                
                <div className="p-2">
                  <button 
                    onClick={() => window.location.href = '/profile'}
                    className="w-full flex items-center gap-3 px-3 py-2 text-sm text-gray-700 hover:bg-gray-50 rounded-md transition-colors"
                  >
                    <User className="w-4 h-4" />
                    View Profile
                  </button>
                  <div className="border-t border-gray-100 mt-2 pt-2">
                    <button 
                      onClick={onLogout}
                      className="w-full flex items-center gap-3 px-3 py-2 text-sm text-gray-700 hover:bg-gray-50 rounded-md transition-colors"
                    >
                      <LogOut className="w-4 h-4" />
                      Logout
                    </button>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </header>
  );
}