import React, { useState, useEffect } from "react";
import {
  TrendingUp,
  TrendingDown,
  Minus,
  MapPin,
  CheckCircle,
  AlertCircle,
} from "lucide-react";

interface BuyerMarketSnapshotProps {
  propertyPrice: number;
  locality: string;
  city?: string;
  variant?: "full" | "summary"; // full for PDP, summary for dashboard
}

interface MarketData {
  localityAvgPrice: number;
  demandScore: "High" | "Medium" | "Low";
  trend: "Up" | "Stable" | "Down";
  nearbyHotAreas: Array<{
    name: string;
    level: "High" | "Medium" | "Low";
  }>;
  avgDaysOnMarket: number;
  totalActiveListings: number;
  priceAppreciation: number;
  investmentScore: number;
}

export const BuyerMarketSnapshot: React.FC<BuyerMarketSnapshotProps> = ({
  propertyPrice,
  locality,
  city = "Bangalore",
  variant = "full",
}) => {
  const [marketData, setMarketData] = useState<MarketData | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    console.log("🔍 BuyerMarketSnapshot mounted with props:", {
      propertyPrice,
      locality,
      city,
      variant,
    });
    fetchMarketData();
  }, [locality]);

  const fetchMarketData = async () => {
    try {
      console.log(
        "🔍 BuyerMarketSnapshot: Fetching data for locality:",
        locality,
      );

      // Fetch locality average price
      const avgPriceResponse = await fetch(
        `http://localhost:8090/api/properties?locality=${encodeURIComponent(locality)}&limit=50`,
      );
      console.log(
        "📊 Properties API response status:",
        avgPriceResponse.status,
      );
      const avgPriceData = await avgPriceResponse.json();
      console.log("📊 Properties API data:", avgPriceData);

      const properties = avgPriceData.properties || avgPriceData.results || [];
      const avgPrice =
        properties.length > 0
          ? properties.reduce(
              (sum: number, p: any) =>
                sum + parseFloat(p.sale_price || p.monthly_rent || "0"),
              0,
            ) / properties.length
          : propertyPrice;
      console.log(
        "💰 Calculated average price:",
        avgPrice,
        "from",
        properties.length,
        "properties",
      );

      // Fetch market volatility data for demand and trend
      const volatilityResponse = await fetch(
        `http://localhost:8090/api/buyer-market-volatility/safety-check?locality=${encodeURIComponent(locality)}`,
      );
      console.log(
        "📈 Volatility API response status:",
        volatilityResponse.status,
      );
      const volatilityData = await volatilityResponse.json();
      console.log("📈 Volatility API data:", volatilityData);

      // Fetch nearby hot areas from micro markets
      let hotAreasData = { success: false, data: { markets: [] } };
      try {
        const hotAreasResponse = await fetch(
          `http://localhost:8090/api/micro-markets/top-performing?limit=5`,
        );
        console.log(
          "🔥 Hot areas API response status:",
          hotAreasResponse.status,
        );
        if (hotAreasResponse.ok) {
          hotAreasData = await hotAreasResponse.json();
          console.log("🔥 Hot areas API data:", hotAreasData);
        } else {
          console.warn("🔥 Hot areas API failed, using fallback");
        }
      } catch (error) {
        console.warn("🔥 Hot areas API error:", error);
      }

      // Fetch time to sell data
      let timeToSellData = { success: false, data: { avg_days_to_sell: 45 } };
      try {
        const timeToSellResponse = await fetch(
          `http://localhost:8090/api/time-to-sell?locality=${encodeURIComponent(locality)}`,
        );
        if (timeToSellResponse.ok) {
          timeToSellData = await timeToSellResponse.json();
          console.log("⏱️ Time to sell data:", timeToSellData);
        }
      } catch (error) {
        console.warn("⏱️ Time to sell API error:", error);
      }

      // Fetch area analysis for investment score
      let areaAnalysisData = { success: false, data: { areas: [] } };
      try {
        const areaAnalysisResponse = await fetch(
          `http://localhost:8090/api/area-analysis`,
        );
        if (areaAnalysisResponse.ok) {
          areaAnalysisData = await areaAnalysisResponse.json();
          console.log("📊 Area analysis data:", areaAnalysisData);
        }
      } catch (error) {
        console.warn("📊 Area analysis API error:", error);
      }

      // Map API responses to component data
      const demandLevel =
        volatilityData.success &&
        volatilityData.data.volatility_status === "Stable"
          ? "High"
          : volatilityData.success &&
              volatilityData.data.volatility_status === "Moderate Fluctuation"
            ? "Medium"
            : "Low";

      const trendDirection =
        volatilityData.success &&
        volatilityData.data.volatility_status === "Stable"
          ? "Stable"
          : volatilityData.success &&
              volatilityData.data.volatility_status === "High Fluctuation"
            ? "Down"
            : "Up";

      const nearbyAreas =
        hotAreasData.success && hotAreasData.data.markets
          ? hotAreasData.data.markets.slice(0, 3).map((market: any) => ({
              name: market.name || market.locality || "Unknown Area",
              level:
                market.overall_score > 70
                  ? ("High" as const)
                  : market.overall_score > 40
                    ? ("Medium" as const)
                    : ("Low" as const),
            }))
          : [{ name: "No data available", level: "Medium" as const }];

      // Calculate additional metrics
      const avgDaysOnMarket = timeToSellData.success 
        ? timeToSellData.data.avg_days_to_sell 
        : 45;

      const totalActiveListings = properties.length;

      // Calculate price appreciation from area analysis
      const localityData = areaAnalysisData.success 
        ? areaAnalysisData.data.areas.find((a: any) => 
            a.locality?.toLowerCase() === locality.toLowerCase()
          )
        : null;
      
      const priceAppreciation = localityData?.price_trend || 0;
      
      // Calculate investment score (0-100)
      const investmentScore = Math.min(100, Math.round(
        (demandLevel === "High" ? 40 : demandLevel === "Medium" ? 25 : 10) +
        (trendDirection === "Up" ? 30 : trendDirection === "Stable" ? 20 : 5) +
        (avgDaysOnMarket < 30 ? 30 : avgDaysOnMarket < 60 ? 20 : 10)
      ));

      const marketData: MarketData = {
        localityAvgPrice: avgPrice,
        demandScore: demandLevel,
        trend: trendDirection,
        nearbyHotAreas: nearbyAreas,
        avgDaysOnMarket,
        totalActiveListings,
        priceAppreciation,
        investmentScore,
      };

      console.log("✅ Final market data:", marketData);
      setMarketData(marketData);
    } catch (error) {
      console.error("❌ Error fetching market data:", error);
      // Fallback data to ensure component still shows
      const fallbackData: MarketData = {
        localityAvgPrice: propertyPrice * 1.1,
        demandScore: "Medium",
        trend: "Stable",
        nearbyHotAreas: [{ name: "Area data unavailable", level: "Medium" }],
        avgDaysOnMarket: 45,
        totalActiveListings: 0,
        priceAppreciation: 0,
        investmentScore: 50,
      };
      console.log("🔄 Using fallback data:", fallbackData);
      setMarketData(fallbackData);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
        <div className="animate-pulse">
          <div className="h-4 bg-blue-200 rounded w-1/3 mb-2"></div>
          <div className="h-3 bg-blue-200 rounded w-2/3"></div>
        </div>
      </div>
    );
  }

  if (!marketData) return null;

  const priceDifference =
    ((propertyPrice - marketData.localityAvgPrice) /
      marketData.localityAvgPrice) *
    100;
  const isPriceFair = Math.abs(priceDifference) <= 10;
  const isPriceBelow = priceDifference < -5;

  // Helper functions for soft language presentation
  const getPriceFairnessText = (priceDifference: number) => {
    // Removed raw percentages for MVP - using soft qualitative language instead
    const absDiff = Math.abs(priceDifference);
    
    if (priceDifference < -15) {
      return "This property is priced lower than most homes in this area.";
    } else if (priceDifference < -5) {
      return "This property is priced below the area average.";
    } else if (absDiff <= 10) {
      return "This property is priced in line with the area average.";
    } else if (priceDifference <= 30) {
      return "This property is priced slightly above the average in this area.";
    } else {
      return "This property is priced significantly higher than most homes in this area.";
    }
  };

  const getDemandText = (level: string) => {
    switch (level) {
      case "High":
        return "High buyer interest in this area";
      case "Medium":
        return "Moderate buyer interest in this area";
      case "Low":
        return "Buyer activity in this area is currently lower";
      default:
        return "Moderate buyer interest in this area";
    }
  };

  const getTrendText = (trend: string) => {
    switch (trend) {
      case "Up":
        return "Prices in this area have increased recently.";
      case "Down":
        return "Prices in this area have softened recently.";
      case "Stable":
        return "Prices in this area have remained stable.";
      default:
        return "Prices in this area have remained stable.";
    }
  };

  const getCleanNearbyAreas = (areas: MarketData['nearbyHotAreas']) => {
    return areas
      .filter(area => 
        area.name && 
        area.name !== "No data available" && 
        area.name !== "Area data unavailable" &&
        area.name !== "Unknown Area" &&
        !area.name.toLowerCase().includes(city.toLowerCase()) // Exclude city-level names
      )
      .slice(0, 3)
      .map(area => ({
        ...area,
        name: area.name.split(' ').map(word => 
          word.charAt(0).toUpperCase() + word.slice(1).toLowerCase()
        ).join(' ') // Proper capitalization
      }));
  };

  const getDemandIcon = (level: string) => {
    switch (level) {
      case "High":
        return "🔥";
      case "Medium":
        return "🟡";
      case "Low":
        return "🔵";
      default:
        return "🟡";
    }
  };

  if (variant === "summary") {
    return (
      <div className="bg-white border border-gray-200 rounded-lg p-4">
        <h3 className="font-semibold text-gray-900 mb-3">Market Snapshot</h3>
        <div className="space-y-2 text-sm">
          <div className="flex items-center justify-between">
            <span>Price Check</span>
            <span
              className={`flex items-center gap-1 ${isPriceBelow ? "text-green-600" : isPriceFair ? "text-blue-600" : "text-orange-600"}`}
            >
              {isPriceBelow ? (
                <CheckCircle className="w-3 h-3" />
              ) : (
                <AlertCircle className="w-3 h-3" />
              )}
              {isPriceBelow
                ? "Below Average"
                : isPriceFair
                  ? "Around Average"
                  : "Above Average"}
            </span>
          </div>
          <div className="flex items-center justify-between">
            <span>Buyer Interest</span>
            <span className="flex items-center gap-1">
              {getDemandIcon(marketData.demandScore)} {marketData.demandScore}
            </span>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-lg p-6 mb-6">
      <h3 className="font-semibold text-gray-900 mb-4 flex items-center gap-2">
        <div className="w-6 h-6 bg-blue-500 rounded-full flex items-center justify-center">
          <MapPin className="w-3 h-3 text-white" />
        </div>
        Market Snapshot
      </h3>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {/* Price Fairness */}
        <div className="bg-white rounded-lg p-4 border border-gray-200">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-gray-700">
              Price Check
            </span>
            {isPriceBelow ? (
              <CheckCircle className="w-4 h-4 text-green-600" />
            ) : (
              <AlertCircle className="w-4 h-4 text-blue-600" />
            )}
          </div>
          <p className="text-sm text-gray-900">
            {getPriceFairnessText(priceDifference)}
            {isPriceBelow && " ✅"}
          </p>
        </div>

        {/* Area Demand */}
        <div className="bg-white rounded-lg p-4 border border-gray-200">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-gray-700">
              Buyer Interest
            </span>
          </div>
          <p className="text-sm text-gray-900">
            {getDemandText(marketData.demandScore)}
          </p>
        </div>

        {/* Price Trend */}
        <div className="bg-white rounded-lg p-4 border border-gray-200">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-gray-700">
              Price Trend
            </span>
            {marketData.trend === "Up" ? (
              <TrendingUp className="w-4 h-4 text-green-600" />
            ) : marketData.trend === "Down" ? (
              <TrendingDown className="w-4 h-4 text-red-600" />
            ) : (
              <Minus className="w-4 h-4 text-gray-600" />
            )}
          </div>
          <p className="text-sm text-gray-900">
            {getTrendText(marketData.trend)}
          </p>
        </div>

        {/* Average Days on Market - REMOVED */}

        {/* Active Listings - REMOVED */}

        {/* Investment Score */}
        <div className="bg-white rounded-lg p-4 border border-gray-200">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-gray-700">
              Investment Score
            </span>
            <span className={`text-lg font-bold ${
              marketData.investmentScore >= 70 ? "text-green-600" :
              marketData.investmentScore >= 50 ? "text-blue-600" : "text-orange-600"
            }`}>
              {marketData.investmentScore}/100
            </span>
          </div>
          <p className="text-sm text-gray-900">
            {marketData.investmentScore >= 70 ? "Strong" : marketData.investmentScore >= 50 ? "Moderate" : "Fair"} investment potential based on market conditions.
          </p>
        </div>

        {/* Popular Nearby Localities */}
        {(() => {
          const cleanAreas = getCleanNearbyAreas(marketData.nearbyHotAreas);
          if (cleanAreas.length === 0) return null;
          
          return (
            <div className="bg-white rounded-lg p-4 border border-gray-200">
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-medium text-gray-700">
                  Popular Nearby Localities
                </span>
                <MapPin className="w-4 h-4 text-gray-500" />
              </div>
              <div className="space-y-1">
                {cleanAreas.map((area, index) => (
                  <div key={index} className="flex items-center gap-2 text-sm">
                    <span>•</span>
                    <span className="text-gray-900">{area.name}</span>
                  </div>
                ))}
              </div>
            </div>
          );
        })()}
      </div>

      <div className="mt-4 text-xs text-gray-500 text-center">
        Market data helps you understand context • Not financial advice
      </div>
    </div>
  );
};
