import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { negotiationApi } from '../../api/negotiation.api';
import { MessageSquare, Clock, CheckCircle, XCircle, Eye, TrendingUp } from 'lucide-react';

type FilterType = 'all' | 'active' | 'accepted' | 'rejected';

interface Negotiation {
  negotiation_id: string;
  property_id: string;
  property_title?: string;
  property_price?: number;
  negotiation_status: string;
  created_at?: string;
  user_role?: string;
}

export function BuyerNegotiations() {
  const navigate = useNavigate();
  const [negotiations, setNegotiations] = useState<Negotiation[]>([]);
  const [filteredNegotiations, setFilteredNegotiations] = useState<Negotiation[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [activeFilter, setActiveFilter] = useState<FilterType>('all');

  useEffect(() => {
    fetchNegotiations();
    const interval = setInterval(fetchNegotiations, 30000); // Refresh every 30 seconds
    return () => clearInterval(interval);
  }, []);

  useEffect(() => {
    filterNegotiations();
  }, [negotiations, activeFilter]);

  const fetchNegotiations = async () => {
    try {
      setError(null);
      const result = await negotiationApi.listNegotiations();
      setNegotiations(result || []);
    } catch (err) {
      console.error('Error fetching negotiations:', err);
      setError('Failed to load negotiations');
    } finally {
      setIsLoading(false);
    }
  };

  const filterNegotiations = () => {
    let filtered = [...negotiations];
    
    switch (activeFilter) {
      case 'active':
        filtered = filtered.filter(n => n.negotiation_status === 'PENDING' || n.negotiation_status === 'COUNTER_OFFER');
        break;
      case 'accepted':
        filtered = filtered.filter(n => n.negotiation_status === 'ACCEPTED');
        break;
      case 'rejected':
        filtered = filtered.filter(n => n.negotiation_status === 'REJECTED');
        break;
      default:
        break;
    }
    
    setFilteredNegotiations(filtered);
  };

  const handleViewTimeline = (negotiationId: string) => {
    navigate(`/negotiations/${negotiationId}`);
  };

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'PENDING':
      case 'INITIATED':
        return (
          <span className="px-2 py-1 bg-yellow-100 text-yellow-800 text-xs font-medium rounded-full flex items-center gap-1">
            <Clock className="w-3 h-3" />
            Pending
          </span>
        );
      case 'COUNTER_OFFER':
        return (
          <span className="px-2 py-1 bg-blue-100 text-blue-800 text-xs font-medium rounded-full flex items-center gap-1">
            <TrendingUp className="w-3 h-3" />
            Counter Offer
          </span>
        );
      case 'ACCEPTED':
        return (
          <span className="px-2 py-1 bg-green-100 text-green-800 text-xs font-medium rounded-full flex items-center gap-1">
            <CheckCircle className="w-3 h-3" />
            Accepted
          </span>
        );
      case 'REJECTED':
        return (
          <span className="px-2 py-1 bg-red-100 text-red-800 text-xs font-medium rounded-full flex items-center gap-1">
            <XCircle className="w-3 h-3" />
            Rejected
          </span>
        );
      default:
        return (
          <span className="px-2 py-1 bg-gray-100 text-gray-800 text-xs font-medium rounded-full">
            {status}
          </span>
        );
    }
  };

  const formatCurrency = (amount?: number) => {
    if (!amount) return 'N/A';
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const formatDate = (dateString?: string) => {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    const now = new Date();
    const diffMs = now.getTime() - date.getTime();
    const diffMins = Math.floor(diffMs / 60000);
    const diffHours = Math.floor(diffMs / 3600000);
    const diffDays = Math.floor(diffMs / 86400000);

    if (diffMins < 60) return `${diffMins} minutes ago`;
    if (diffHours < 24) return `${diffHours} hours ago`;
    if (diffDays < 7) return `${diffDays} days ago`;
    return date.toLocaleDateString();
  };

  const stats = {
    total: negotiations.length,
    active: negotiations.filter(n => n.negotiation_status === 'PENDING' || n.negotiation_status === 'COUNTER_OFFER' || n.negotiation_status === 'INITIATED').length,
    accepted: negotiations.filter(n => n.negotiation_status === 'ACCEPTED').length,
    rejected: negotiations.filter(n => n.negotiation_status === 'REJECTED').length,
  };

  return (
    <div className="space-y-6">
      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Total Negotiations</p>
              <p className="text-2xl font-bold text-gray-900">{stats.total}</p>
            </div>
            <MessageSquare className="w-8 h-8 text-blue-500" />
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Active</p>
              <p className="text-2xl font-bold text-yellow-600">{stats.active}</p>
            </div>
            <Clock className="w-8 h-8 text-yellow-500" />
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Accepted</p>
              <p className="text-2xl font-bold text-green-600">{stats.accepted}</p>
            </div>
            <CheckCircle className="w-8 h-8 text-green-500" />
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Rejected</p>
              <p className="text-2xl font-bold text-red-600">{stats.rejected}</p>
            </div>
            <XCircle className="w-8 h-8 text-red-500" />
          </div>
        </div>
      </div>

      {/* Filters */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
        <div className="flex flex-wrap gap-2">
          {(['all', 'active', 'accepted', 'rejected'] as FilterType[]).map((filter) => (
            <button
              key={filter}
              onClick={() => setActiveFilter(filter)}
              className={`px-4 py-2 rounded-md text-sm font-medium transition-colors ${
                activeFilter === filter
                  ? 'bg-[#0056D2] text-white shadow-md'
                  : 'bg-white text-gray-700 border border-gray-300 hover:bg-gray-50'
              }`}
            >
              {filter.charAt(0).toUpperCase() + filter.slice(1)}
            </button>
          ))}
        </div>
      </div>

      {/* Error State */}
      {error && (
        <div className="bg-red-50 border border-red-200 rounded-lg p-4">
          <div className="flex items-center gap-2 text-red-800">
            <XCircle className="w-5 h-5" />
            <p>{error}</p>
          </div>
        </div>
      )}

      {/* Loading State */}
      {isLoading && (
        <div className="flex justify-center items-center py-12">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
          <span className="ml-3 text-gray-600">Loading negotiations...</span>
        </div>
      )}

      {/* Negotiations Grid */}
      {!isLoading && !error && (
        <>
          {filteredNegotiations.length > 0 ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {filteredNegotiations.map((negotiation) => (
                <div
                  key={negotiation.negotiation_id}
                  className="bg-white rounded-lg shadow-sm border border-gray-200 p-5 hover:shadow-md transition-shadow"
                >
                  <div className="flex justify-between items-start mb-3">
                    <div className="flex-1">
                      <h3 className="text-lg font-semibold text-gray-900 mb-1 line-clamp-1">
                        {negotiation.property_title || `Property ${negotiation.property_id?.toString().slice(0, 8)}`}
                      </h3>
                      {getStatusBadge(negotiation.negotiation_status)}
                    </div>
                  </div>

                  <div className="space-y-2 mb-4">
                    <div className="flex justify-between text-sm">
                      <span className="text-gray-600">Property Price:</span>
                      <span className="font-medium">{formatCurrency(negotiation.property_price)}</span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span className="text-gray-600">Status:</span>
                      <span className="font-semibold text-blue-600">{negotiation.negotiation_status}</span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span className="text-gray-600">Started:</span>
                      <span className="font-medium text-gray-700">
                        {formatDate(negotiation.created_at)}
                      </span>
                    </div>
                  </div>

                  <button
                    onClick={() => handleViewTimeline(negotiation.negotiation_id)}
                    className="w-full bg-blue-600 text-white py-2 px-4 rounded-md hover:bg-blue-700 transition-colors text-sm font-medium flex items-center justify-center gap-2"
                  >
                    <Eye className="w-4 h-4" />
                    View Timeline
                  </button>
                </div>
              ))}
            </div>
          ) : (
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-12 text-center">
              <MessageSquare className="w-16 h-16 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No Negotiations Found</h3>
              <p className="text-gray-600">
                {activeFilter === 'all'
                  ? 'You have not started any negotiations yet.'
                  : `No ${activeFilter} negotiations found.`}
              </p>
            </div>
          )}
        </>
      )}
    </div>
  );
}
