import React, { useState, useEffect } from "react";
import { fetchBuyerRecommendations, BuyerRecommendationResponse } from "../../api/buyerRecommendations";
import { userPreferencesAPI } from "../../api/user/userPreferences";
import { FacilitiesMap } from "./FacilitiesMap";
import { PropertyCard } from "../PropertyCard";

export function BuyerRecommendations() {
  const [budget, setBudget] = useState("");
  const [locality, setLocality] = useState("");
  const [minSize, setMinSize] = useState("");
  const [maxSize, setMaxSize] = useState("");
  const [radiusKm, setRadiusKm] = useState(5);
  
  // New filter states
  const [propertyTypes, setPropertyTypes] = useState<string[]>([]);
  const [bedrooms, setBedrooms] = useState("");
  const [bathrooms, setBathrooms] = useState("");
  const [furnishing, setFurnishing] = useState("");
  const [ageOfProperty, setAgeOfProperty] = useState("");
  const [possession, setPossession] = useState("");
  const [facing, setFacing] = useState("");
  const [parking, setParking] = useState("");
  const [gatedCommunity, setGatedCommunity] = useState("");
  const [amenities, setAmenities] = useState<string[]>([]);
  const [showAdvanced, setShowAdvanced] = useState(false);
  
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [result, setResult] = useState<BuyerRecommendationResponse | null>(null);
  const [preferencesLoaded, setPreferencesLoaded] = useState(false);
  const [preferencesMessage, setPreferencesMessage] = useState<string | null>(null);

  const handlePropertyTypeToggle = (type: string) => {
    setPropertyTypes(prev => 
      prev.includes(type) ? prev.filter(t => t !== type) : [...prev, type]
    );
  };

  const handleAmenityToggle = (amenity: string) => {
    setAmenities(prev => 
      prev.includes(amenity) ? prev.filter(a => a !== amenity) : [...prev, amenity]
    );
  };

  // Load user preferences on mount
  useEffect(() => {
    const loadPreferences = async () => {
      try {
        // Get user_id from localStorage or auth context
        const userStr = localStorage.getItem('user');
        if (!userStr) return;
        
        const user = JSON.parse(userStr);
        const userId = user.user_id || user.id;
        
        if (!userId) return;
        
        const prefs = await userPreferencesAPI.get(userId);
        
        if (prefs) {
          // Pre-fill form with preferences
          if (prefs.budget_max) setBudget(prefs.budget_max.toString());
          if (prefs.location && prefs.location.length > 0) setLocality(prefs.location[0]);
          if (prefs.built_up_area_min) setMinSize(prefs.built_up_area_min.toString());
          if (prefs.built_up_area_max) setMaxSize(prefs.built_up_area_max.toString());
          if (prefs.property_types && prefs.property_types.length > 0) {
            setPropertyTypes(prefs.property_types.map((t: string) => t.toUpperCase()));
          }
          if (prefs.bedrooms) setBedrooms(prefs.bedrooms);
          if (prefs.bathrooms) setBathrooms(prefs.bathrooms);
          if (prefs.furnishing) setFurnishing(prefs.furnishing);
          if (prefs.age_of_property) setAgeOfProperty(prefs.age_of_property);
          if (prefs.possession) setPossession(prefs.possession);
          if (prefs.facing) setFacing(prefs.facing);
          if (prefs.parking) setParking(prefs.parking);
          if (prefs.amenities && prefs.amenities.length > 0) setAmenities(prefs.amenities);
          
          setPreferencesLoaded(true);
          setPreferencesMessage("✨ We've loaded your saved preferences! Review and click 'Find Properties' to search.");
          
          // Auto-hide message after 8 seconds
          setTimeout(() => setPreferencesMessage(null), 8000);
        }
      } catch (err) {
        console.log('No preferences found or error loading:', err);
      }
    };
    
    loadPreferences();
  }, []);

  // Prevent mouse wheel from changing number input values
  const handleWheel = (e: React.WheelEvent<HTMLInputElement>) => {
    e.currentTarget.blur();
  };

  const onSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setError(null);
    setResult(null);

    try {
      const payload = {
        budget: Number(budget) || 1000000,
        city: "Bangalore",
        locality: locality.trim() || "Bangalore",
        min_size: Number(minSize) || 600,
        max_size: Number(maxSize) || 2000,
        radius_km: Number(radiusKm) || 5,
        property_types: propertyTypes.length > 0 ? propertyTypes : undefined,
        bedrooms: bedrooms || undefined,
        bathrooms: bathrooms || undefined,
        furnishing: furnishing || undefined,
        age_of_property: ageOfProperty || undefined,
        possession: possession || undefined,
        facing: facing || undefined,
        parking: parking || undefined,
        gated_community: gatedCommunity || undefined,
        amenities: amenities.length > 0 ? amenities : undefined,
        include_facilities: true,
        include_locality_intel: true
      };

      console.log('Sending payload:', payload);
      const data = await fetchBuyerRecommendations(payload);
      console.log('Buyer Recommendations Response:', data);
      console.log('Facilities:', data.facilities);
      
      // Log property details to debug
      if (data.exact_results?.groups) {
        data.exact_results.groups.forEach(group => {
          console.log(`Exact ${group.property_type} properties:`, group.properties.map(p => ({
            id: p.id,
            title: p.title,
            bedrooms: p.bedrooms,
            bathrooms: p.bathrooms
          })));
        });
      }
      if (data.nearby_results?.groups) {
        data.nearby_results.groups.forEach(group => {
          console.log(`Nearby ${group.property_type} properties:`, group.properties.map(p => ({
            id: p.id,
            title: p.title,
            bedrooms: p.bedrooms,
            bathrooms: p.bathrooms
          })));
        });
      }
      
      setResult(data);
    } catch (err: any) {
      console.error('Search error:', err);
      setError(err?.message || "Recommendation failed");
    } finally {
      setLoading(false);
    }
  };

  const renderGroups = (title: string, groups: any[]) => {
    if (!groups || groups.length === 0) return null;
    return (
      <div className="bg-white rounded-xl border border-gray-200 p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">{title}</h3>
        <div className="space-y-6">
          {groups.map((group) => (
            <div key={group.property_type}>
              <div className="flex items-center justify-between mb-3">
                <h4 className="text-base font-semibold text-gray-800">{group.property_type}</h4>
                <span className="text-sm text-gray-500">{group.count} results</span>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {group.properties.slice(0, 6).map((p: any) => {
                  console.log('Property data before card:', {
                    id: p.id,
                    title: p.title,
                    bedrooms: p.bedrooms,
                    bathrooms: p.bathrooms,
                    size_sqft: p.size_sqft
                  });
                  return (
                    <PropertyCard
                      key={`${group.property_type}-${p.id}`}
                      id={p.id || 0}
                      title={p.title || 'Property'}
                      location={`${p.locality}, ${p.city}`}
                      beds={p.bedrooms}
                      baths={p.bathrooms}
                      area={p.size_sqft ? `${p.size_sqft} sqft` : 'N/A'}
                      price={p.price ? `₹${(p.price / 100000).toFixed(2)} L` : 'N/A'}
                      amenities={p.amenities || []}
                    />
                  );
                })}
              </div>
            </div>
          ))}
        </div>
      </div>
    );
  };

  return (
    <div className="space-y-6">
      <div className="bg-white rounded-xl border border-gray-200 p-6">
        <h3 className="text-lg font-bold text-gray-900 mb-4">Find Your Perfect Property</h3>
        
        {/* Preferences Loaded Message */}
        {preferencesMessage && (
          <div className="mb-4 p-4 bg-gradient-to-r from-green-50 to-blue-50 border border-green-200 rounded-lg flex items-start gap-3">
            <span className="text-2xl">✨</span>
            <div className="flex-1">
              <p className="text-sm font-semibold text-green-800">{preferencesMessage}</p>
              <p className="text-xs text-green-600 mt-1">You can modify any field before searching.</p>
            </div>
            <button
              type="button"
              onClick={() => setPreferencesMessage(null)}
              className="text-green-600 hover:text-green-800 text-xl leading-none"
            >
              ×
            </button>
          </div>
        )}
        
        <form onSubmit={onSubmit} className="space-y-4">
          {/* Basic Search Fields */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Budget (₹)</label>
              <input
                type="number"
                value={budget}
                onChange={(e) => setBudget(e.target.value)}
                onWheel={handleWheel}
                placeholder="1000000"
                className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                min={100000}
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Locality</label>
              <input
                type="text"
                value={locality}
                onChange={(e) => setLocality(e.target.value)}
                className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="Yelahanka New Town"
                required
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Bedrooms</label>
              <select
                value={bedrooms}
                onChange={(e) => setBedrooms(e.target.value)}
                className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
              >
                <option value=""></option>
                <option value="1">1</option>
                <option value="2">2</option>
                <option value="3">3</option>
                <option value="4">4</option>
                <option value="5+">5+</option>
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Min Size (sqft)</label>
              <input
                type="number"
                value={minSize}
                onChange={(e) => setMinSize(e.target.value)}
                onWheel={handleWheel}
                placeholder="600"
                className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Max Size (sqft)</label>
              <input
                type="number"
                value={maxSize}
                onChange={(e) => setMaxSize(e.target.value)}
                onWheel={handleWheel}
                placeholder="2000"
                className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Radius (KM)</label>
              <input
                type="number"
                value={radiusKm}
                onChange={(e) => setRadiusKm(Number(e.target.value))}
                onWheel={handleWheel}
                className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                min={1}
                max={15}
              />
            </div>
          </div>

          {/* Property Types */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Property Type</label>
            <div className="flex flex-wrap gap-2">
              {['APARTMENT', 'VILLA', 'HOUSE', 'PLOT', 'STUDIO', 'PENTHOUSE'].map(type => (
                <label key={type} className="flex items-center gap-2 bg-white border border-gray-200 rounded-lg px-3 py-2 cursor-pointer hover:bg-gray-50">
                  <input
                    type="checkbox"
                    checked={propertyTypes.includes(type)}
                    onChange={() => handlePropertyTypeToggle(type)}
                    className="rounded"
                  />
                  <span className="text-sm">{type}</span>
                </label>
              ))}
            </div>
          </div>

          {/* Advanced Filters Toggle */}
          <div>
            <button
              type="button"
              onClick={() => setShowAdvanced(!showAdvanced)}
              className="text-sm font-medium text-blue-600 hover:text-blue-700 flex items-center gap-1"
            >
              {showAdvanced ? '▼' : '▶'} Advanced Filters
            </button>
          </div>

          {/* Advanced Filters Section */}
          {showAdvanced && (
            <div className="border border-gray-200 rounded-lg p-4 space-y-4 bg-gray-50">
              {/* Bathrooms and other filters */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Bathrooms</label>
                  <select
                    value={bathrooms}
                    onChange={(e) => setBathrooms(e.target.value)}
                    className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    <option value=""></option>
                    <option value="1">1</option>
                    <option value="2">2</option>
                    <option value="3">3</option>
                    <option value="4+">4+</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Furnishing</label>
                  <select
                    value={furnishing}
                    onChange={(e) => setFurnishing(e.target.value)}
                    className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    <option value=""></option>
                    <option value="unfurnished">Unfurnished</option>
                    <option value="semi-furnished">Semi-Furnished</option>
                    <option value="fully-furnished">Fully-Furnished</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Parking</label>
                  <select
                    value={parking}
                    onChange={(e) => setParking(e.target.value)}
                    className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    <option value=""></option>
                    <option value="none">None</option>
                    <option value="1">1</option>
                    <option value="2">2</option>
                    <option value="3+">3+</option>
                  </select>
                </div>
              </div>

              {/* Age, Possession, Facing */}
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Age of Property</label>
                  <select
                    value={ageOfProperty}
                    onChange={(e) => setAgeOfProperty(e.target.value)}
                    className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    <option value=""></option>
                    <option value="new">New</option>
                    <option value="0-1">0-1 years</option>
                    <option value="1-5">1-5 years</option>
                    <option value="5-10">5-10 years</option>
                    <option value="10+">10+ years</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Possession</label>
                  <select
                    value={possession}
                    onChange={(e) => setPossession(e.target.value)}
                    className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    <option value=""></option>
                    <option value="ready-to-move">Ready to Move</option>
                    <option value="under-construction">Under Construction</option>
                    <option value="new-launch">New Launch</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Facing</label>
                  <select
                    value={facing}
                    onChange={(e) => setFacing(e.target.value)}
                    className="w-full border border-gray-200 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    <option value=""></option>
                    <option value="north">North</option>
                    <option value="south">South</option>
                    <option value="east">East</option>
                    <option value="west">West</option>
                    <option value="northeast">Northeast</option>
                    <option value="northwest">Northwest</option>
                  </select>
                </div>
              </div>

              {/* Gated Community */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Gated Community</label>
                <div className="flex gap-4">
                  {['', 'yes', 'no'].map(value => (
                    <label key={value} className="flex items-center gap-2 cursor-pointer">
                      <input
                        type="radio"
                        name="gatedCommunity"
                        value={value}
                        checked={gatedCommunity === value}
                        onChange={(e) => setGatedCommunity(e.target.value)}
                        className="rounded-full"
                      />
                      <span className="text-sm">{value === '' ? 'None' : value === 'yes' ? 'Yes' : 'No'}</span>
                    </label>
                  ))}
                </div>
              </div>

              {/* Amenities */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Amenities</label>
                <div className="flex flex-wrap gap-2">
                  {['Swimming Pool', 'Gym', 'Parking', 'Security', 'Power Backup', 'Lift', 'Club House', 'Garden', "Children's Play Area"].map(amenity => (
                    <label key={amenity} className="flex items-center gap-2 bg-white border border-gray-200 rounded-lg px-3 py-2 cursor-pointer hover:bg-gray-50">
                      <input
                        type="checkbox"
                        checked={amenities.includes(amenity)}
                        onChange={() => handleAmenityToggle(amenity)}
                        className="rounded"
                      />
                      <span className="text-sm">{amenity}</span>
                    </label>
                  ))}
                </div>
              </div>
            </div>
          )}

          {/* Submit Button */}
          <div className="flex items-center gap-3">
            <button
              type="submit"
              className="bg-[#0056D2] text-white rounded-lg px-6 py-2 font-semibold hover:bg-[#004bb5] transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
              disabled={loading}
            >
              {loading ? "Finding..." : "Find Properties"}
            </button>
            <button
              type="button"
              onClick={() => {
                setBudget("");
                setLocality("");
                setMinSize("");
                setMaxSize("");
                setRadiusKm(5);
                setPropertyTypes([]);
                setBedrooms("");
                setBathrooms("");
                setFurnishing("");
                setAgeOfProperty("");
                setPossession("");
                setFacing("");
                setParking("");
                setGatedCommunity("");
                setAmenities([]);
                setResult(null);
                setError(null);
              }}
              className="text-sm text-gray-600 hover:text-gray-800 underline"
            >
              Clear All Filters
            </button>
          </div>
        </form>
        {error && <div className="text-red-600 text-sm mt-3">{error}</div>}
      </div>

      {result && (
        <div className="space-y-6">
          {/* Current Market Affordability - Only show if PSF exists */}
          {result.affordability_current.avg_psf && (
            <div className="bg-white rounded-xl border border-gray-200 p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-3">Current Market Affordability</h3>
              <div className="grid grid-cols-1 md:grid-cols-4 gap-4 text-sm">
                <div>
                  <div className="text-gray-500">Affordable Now</div>
                  <div className={`font-semibold text-lg ${result.affordability_current.is_affordable ? "text-green-600" : "text-red-600"}`}>
                    {result.affordability_current.is_affordable ? "✅ Yes" : "❌ No"}
                  </div>
                </div>
                <div>
                  <div className="text-gray-500">Current Market PSF</div>
                  <div className="font-semibold">₹{result.affordability_current.avg_psf?.toLocaleString() || "-"}</div>
                </div>
                <div>
                  <div className="text-gray-500">Max Affordable Area</div>
                  <div className="font-semibold">{result.affordability_current.max_affordable_area?.toLocaleString() || "-"} sqft</div>
                </div>
                {!result.affordability_current.is_affordable && result.affordability_current.budget_gap && result.affordability_current.budget_gap > 0 && (
                  <div>
                    <div className="text-gray-500">Budget Gap</div>
                    <div className="font-semibold text-red-600">₹{result.affordability_current.budget_gap.toLocaleString()}</div>
                  </div>
                )}
              </div>
            </div>
          )}

          {/* Market Forecast - Only show if data is meaningful */}
          {result.market_forecast.predicted_future_psf && result.market_forecast.forecast_status === "growth_potential" && result.market_forecast.appreciation_rate && result.market_forecast.appreciation_rate < 200 && (
            <div className="bg-gradient-to-r from-blue-50 to-purple-50 rounded-xl border border-blue-200 p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-3">📈 Market Forecast & Investment Insights</h3>
              <div className="grid grid-cols-1 md:grid-cols-4 gap-4 text-sm">
                <div>
                  <div className="text-gray-500">Current Market PSF</div>
                  <div className="font-semibold text-lg">₹{result.market_forecast.current_psf?.toLocaleString() || "-"}</div>
                </div>
                <div>
                  <div className="text-gray-500">Predicted Future PSF</div>
                  <div className="font-semibold text-lg text-blue-600">₹{result.market_forecast.predicted_future_psf?.toLocaleString() || "-"}</div>
                </div>
                <div>
                  <div className="text-gray-500">Expected Appreciation</div>
                  <div className={`font-semibold text-lg ${(result.market_forecast.appreciation_rate || 0) > 0 ? "text-green-600" : "text-red-600"}`}>
                    {result.market_forecast.appreciation_rate ? `${result.market_forecast.appreciation_rate > 0 ? "+" : ""}${result.market_forecast.appreciation_rate}%` : "-"}
                  </div>
                </div>
                <div>
                  <div className="text-gray-500">Investment Grade</div>
                  <div className="font-semibold text-lg">
                    {(result.market_forecast.appreciation_rate || 0) > 50 ? "A+ 🚀" : (result.market_forecast.appreciation_rate || 0) > 20 ? "A" : "B"}
                  </div>
                </div>
              </div>
              {result.affordability_current.is_affordable && !result.affordability_future.is_affordable && (
                <div className="mt-4 p-3 bg-yellow-100 border border-yellow-300 rounded-lg text-sm text-yellow-800">
                  <strong>⚠️ Act Fast:</strong> Properties are affordable NOW but may become expensive in the future. Predicted {result.market_forecast.appreciation_rate}% price increase expected.
                </div>
              )}
              <div className="mt-3 text-xs text-gray-600">
                ⓘ Predictions based on historical market trends and analysis. Actual future prices may vary.
              </div>
            </div>
          )}

          {result.reason_if_no_match && (
            <div className="bg-red-50 border border-red-200 rounded-xl p-6">
              <h3 className="text-lg font-semibold text-red-900 mb-2">⚠️ No Properties Found</h3>
              <p className="text-sm text-red-800 mb-4">{result.reason_if_no_match}</p>
              
              {result.matched_locality.match_type === 'fallback' && (
                <div className="bg-white rounded-lg p-4 border border-red-200">
                  <p className="text-sm font-semibold text-gray-900 mb-2">Suggestions:</p>
                  <ul className="text-sm text-gray-700 space-y-1 list-disc list-inside">
                    <li>Check the spelling of the locality name</li>
                    <li>Try searching for nearby areas (e.g., Koramangala, Indiranagar, HSR Layout)</li>
                    <li>Use a more general area name (e.g., "Bangalore" instead of specific sub-locality)</li>
                  </ul>
                </div>
              )}
              
              {result.matched_locality.match_type !== 'fallback' && !result.affordability_current.is_affordable && result.affordability_current.min_budget_needed && (
                <div className="bg-white rounded-lg p-4 border border-red-200">
                  <p className="text-sm font-semibold text-gray-900 mb-2">Try these options:</p>
                  <ul className="text-sm text-gray-700 space-y-1 list-disc list-inside">
                    <li>Increase your budget to ₹{result.affordability_current.min_budget_needed?.toLocaleString()}</li>
                    <li>Reduce minimum size requirement (currently {result.input.min_size} sqft)</li>
                    <li>Increase search radius (currently {result.input.radius_km} km)</li>
                    <li>Search in nearby localities with lower prices</li>
                  </ul>
                </div>
              )}
              
              {(!result.affordability_current.min_budget_needed || result.affordability_current.is_affordable) && (
                <div className="bg-white rounded-lg p-4 border border-red-200">
                  <p className="text-sm font-semibold text-gray-900 mb-2">Try these options:</p>
                  <ul className="text-sm text-gray-700 space-y-1 list-disc list-inside">
                    <li>Increase search radius beyond {result.input.radius_km} km</li>
                    <li>Try nearby popular areas (Yelahanka, Hebbal, Koramangala)</li>
                    <li>Check if properties are listed under different locality names</li>
                  </ul>
                </div>
              )}
            </div>
          )}

          {renderGroups("Exact Locality Results", result.exact_results.groups)}
          
          {/* Nearby Results with Progressive Radius Info */}
          {result.nearby_results.groups && result.nearby_results.groups.length > 0 ? (
            <div className="bg-white rounded-xl border border-gray-200 p-6">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold text-gray-900">
                  Nearby Results (Within {result.nearby_results.radius_used || radiusKm} km)
                </h3>
                {result.nearby_results.search_expanded === true && (
                  <span className="text-xs bg-blue-100 text-blue-800 px-3 py-1 rounded-full font-medium">
                    🔍 Search Expanded
                  </span>
                )}
              </div>
              
              {result.nearby_results.expansion_message && (
                <div className="mb-4 p-3 bg-blue-50 border border-blue-200 rounded-lg text-sm text-blue-800">
                  <strong>ℹ️ Auto-Expanded Search:</strong> {result.nearby_results.expansion_message}
                </div>
              )}
              
              <div className="space-y-6">
                {result.nearby_results.groups.map((group) => (
                  <div key={group.property_type}>
                    <div className="flex items-center justify-between mb-3">
                      <h4 className="text-base font-semibold text-gray-800">{group.property_type}</h4>
                      <span className="text-sm text-gray-500">{group.count} results</span>
                    </div>
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                      {group.properties.slice(0, 6).map((p: any) => {
                        console.log('Nearby property data before card:', {
                          id: p.id,
                          title: p.title,
                          bedrooms: p.bedrooms,
                          bathrooms: p.bathrooms,
                          size_sqft: p.size_sqft
                        });
                        return (
                          <PropertyCard
                            key={`${group.property_type}-${p.id}`}
                            id={p.id || 0}
                            title={p.title || 'Property'}
                            location={`${p.locality}, ${p.city}`}
                            beds={p.bedrooms}
                            baths={p.bathrooms}
                            area={p.size_sqft ? `${p.size_sqft} sqft` : 'N/A'}
                            price={p.price ? `₹${(p.price / 100000).toFixed(2)} L` : 'N/A'}
                            amenities={p.amenities || []}
                          />
                        );
                      })}
                    </div>
                  </div>
                ))}
              </div>
            </div>
          ) : null}

          {/* Locality Intelligence - Only show if data exists */}
          {result.locality_intel && result.locality_intel.total_properties && result.locality_intel.total_properties > 0 && (
            <div className="bg-white rounded-xl border border-gray-200 p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-3">Locality Intelligence</h3>
              <div className="grid grid-cols-1 md:grid-cols-5 gap-4 text-sm">
                <div>
                  <div className="text-gray-500">Avg PSF</div>
                  <div className="font-semibold">₹{result.locality_intel.avg_price_per_sqft?.toLocaleString() || "-"}</div>
                </div>
                <div>
                  <div className="text-gray-500">Price Range</div>
                  <div className="font-semibold text-xs">
                    ₹{result.locality_intel.min_price_per_sqft?.toLocaleString() || "-"} - ₹{result.locality_intel.max_price_per_sqft?.toLocaleString() || "-"}
                  </div>
                </div>
                <div>
                  <div className="text-gray-500">Demand</div>
                  <div className={`font-semibold ${
                    result.locality_intel.demand_level === "High" ? "text-red-600" :
                    result.locality_intel.demand_level === "Medium" ? "text-yellow-600" : "text-green-600"
                  }`}>
                    {result.locality_intel.demand_level || "-"}
                  </div>
                </div>
                <div>
                  <div className="text-gray-500">Supply</div>
                  <div className={`font-semibold ${
                    result.locality_intel.supply_level === "High" ? "text-green-600" :
                    result.locality_intel.supply_level === "Medium" ? "text-yellow-600" : "text-red-600"
                  }`}>
                    {result.locality_intel.supply_level || "-"} ({result.locality_intel.total_properties || 0} properties)
                  </div>
                </div>
                <div>
                  <div className="text-gray-500">Rental Yield</div>
                  <div className="font-semibold">{result.locality_intel.rental_yield ? `${result.locality_intel.rental_yield}%` : "-"}</div>
                </div>
              </div>
            </div>
          )}

          {/* Nearby Facilities - Map View */}
          {result.matched_locality.latitude && result.matched_locality.longitude ? (
            <div className="bg-white rounded-xl border border-gray-200 p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Nearby Facilities (Within {radiusKm} km)</h3>
              
              {result.facilities && result.facilities.length > 0 ? (
                <FacilitiesMap
                  centerLat={result.matched_locality.latitude}
                  centerLon={result.matched_locality.longitude}
                  facilities={result.facilities}
                  radiusKm={radiusKm}
                  tileConfig={result.map_tile_config}
                />
              ) : (
                <div className="space-y-3">
                  <div className="text-sm text-gray-600 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                    <p className="font-medium text-yellow-800 mb-2">⚠️ Facilities data temporarily unavailable</p>
                    <p className="text-yellow-700 text-xs">
                      OpenStreetMap API is currently rate-limited or experiencing high traffic. 
                      The map will still display your search location. Facilities data will be available when you search again later.
                    </p>
                  </div>
                  
                  {/* Show map anyway without facilities */}
                  <FacilitiesMap
                    centerLat={result.matched_locality.latitude}
                    centerLon={result.matched_locality.longitude}
                    facilities={[]}
                    radiusKm={radiusKm}
                    tileConfig={result.map_tile_config}
                  />
                </div>
              )}
            </div>
          ) : (
            <div className="bg-white rounded-xl border border-gray-200 p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-3">Nearby Facilities (Within {radiusKm} km)</h3>
              <div className="text-sm text-gray-500 p-4 bg-gray-50 rounded-lg">
                No facilities data available. This may be due to limited OpenStreetMap data for this area.
              </div>
            </div>
          )}

          {result.rag_summary && result.rag_summary.length > 0 ? null : null}
        </div>
      )}
    </div>
  );
}
