import React, { useState } from 'react';
import { QueryClient, QueryClientProvider } from '@tanstack/react-query';
import { useWalletBalance, useCreditPackages, usePurchaseCredits, useTransactionHistory, useServices, useServiceUsage } from '../../hooks/useWallet';
import { Wallet, History, ShoppingBag, TrendingUp } from 'lucide-react';

const queryClient = new QueryClient();

function BuyerWalletContent() {
  const [activeTab, setActiveTab] = useState<'overview' | 'buy' | 'services' | 'history'>('overview');
  const userId = 1; // Get from auth context
  const userType = 'owner';

  const { data: balance } = useWalletBalance(userId, userType);
  const { data: packages } = useCreditPackages(userType);
  const { data: transactions } = useTransactionHistory(userId, 50);
  const { data: services } = useServices();
  const purchaseMutation = usePurchaseCredits(userId, userType);
  const usageMutation = useServiceUsage(userId, userType);

  const handlePurchase = async (packageId: number) => {
    try {
      const result = await purchaseMutation.mutateAsync(packageId);
      alert(result.message);
    } catch (error: any) {
      alert(error.response?.data?.detail || 'Purchase failed');
    }
  };

  const handleUseService = async (serviceCode: string) => {
    try {
      const result = await usageMutation.mutateAsync({ serviceCode });
      alert(result.message);
    } catch (error: any) {
      alert(error.response?.data?.detail || 'Service usage failed');
    }
  };

  return (
    <div className="space-y-6">
      {/* Header with Balance */}
      <div className="bg-gradient-to-r from-blue-600 to-purple-600 rounded-xl p-6 text-white">
        <div className="flex items-center justify-between">
          <div>
            <p className="text-sm opacity-90">Available Balance</p>
            <h2 className="text-4xl font-bold mt-1">{balance?.balance || 0} Credits</h2>
            <p className="text-sm mt-2 opacity-80">≈ ₹{((balance?.balance || 0) * 10).toLocaleString()}</p>
          </div>
          <Wallet className="w-16 h-16 opacity-20" />
        </div>
        
        <div className="grid grid-cols-3 gap-4 mt-6 pt-6 border-t border-white/20">
          <div>
            <p className="text-xs opacity-80">Total Earned</p>
            <p className="text-lg font-semibold">{balance?.total_earned || 0}</p>
          </div>
          <div>
            <p className="text-xs opacity-80">Total Spent</p>
            <p className="text-lg font-semibold">{balance?.total_spent || 0}</p>
          </div>
          <div>
            <p className="text-xs opacity-80">Total Purchased</p>
            <p className="text-lg font-semibold">{balance?.total_purchased || 0}</p>
          </div>
        </div>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-lg shadow-sm border">
        <div className="border-b">
          <div className="flex space-x-1 p-2">
            {[
              { id: 'overview', label: 'Overview', icon: TrendingUp },
              { id: 'buy', label: 'Buy Credits', icon: ShoppingBag },
              { id: 'services', label: 'Services', icon: Wallet },
              { id: 'history', label: 'History', icon: History }
            ].map(tab => (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id as any)}
                className={`flex items-center gap-2 px-4 py-2 rounded-lg transition-colors ${
                  activeTab === tab.id
                    ? 'bg-blue-50 text-blue-600 font-medium'
                    : 'text-gray-600 hover:bg-gray-50'
                }`}
              >
                <tab.icon className="w-4 h-4" />
                {tab.label}
              </button>
            ))}
          </div>
        </div>

        <div className="p-6">
          {/* Overview Tab */}
          {activeTab === 'overview' && (
            <div className="space-y-4">
              <h3 className="text-lg font-semibold">Quick Actions</h3>
              <div className="grid grid-cols-2 gap-4">
                <button
                  onClick={() => setActiveTab('buy')}
                  className="p-4 border-2 border-dashed border-gray-300 rounded-lg hover:border-blue-500 hover:bg-blue-50 transition-colors"
                >
                  <ShoppingBag className="w-8 h-8 text-blue-600 mx-auto mb-2" />
                  <p className="font-medium">Buy More Credits</p>
                </button>
                <button
                  onClick={() => setActiveTab('services')}
                  className="p-4 border-2 border-dashed border-gray-300 rounded-lg hover:border-purple-500 hover:bg-purple-50 transition-colors"
                >
                  <Wallet className="w-8 h-8 text-purple-600 mx-auto mb-2" />
                  <p className="font-medium">Browse Services</p>
                </button>
              </div>
            </div>
          )}

          {/* Buy Credits Tab */}
          {activeTab === 'buy' && (
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              {packages?.map(pkg => (
                <div key={pkg.id} className="border rounded-lg p-6 hover:shadow-lg transition-shadow">
                  {pkg.discount_percent > 0 && (
                    <span className="bg-red-500 text-white text-xs px-2 py-1 rounded-full">
                      Save {pkg.discount_percent}%
                    </span>
                  )}
                  <h4 className="text-xl font-bold mt-2">{pkg.name}</h4>
                  <div className="my-4">
                    <p className="text-3xl font-bold text-blue-600">{pkg.credits}</p>
                    {pkg.bonus_credits > 0 && (
                      <p className="text-sm text-green-600">+{pkg.bonus_credits} Bonus</p>
                    )}
                  </div>
                  <p className="text-2xl font-bold mb-4">₹{pkg.price.toLocaleString()}</p>
                  <button
                    onClick={() => handlePurchase(pkg.id)}
                    disabled={purchaseMutation.isPending}
                    className="w-full bg-blue-600 text-white py-2 rounded-lg hover:bg-blue-700 disabled:opacity-50"
                  >
                    {purchaseMutation.isPending ? 'Processing...' : 'Buy Now'}
                  </button>
                </div>
              ))}
            </div>
          )}

          {/* Services Tab */}
          {activeTab === 'services' && (
            <div className="space-y-4">
              {services?.map(service => (
                <div key={service.id} className="flex items-center justify-between p-4 border rounded-lg hover:bg-gray-50">
                  <div>
                    <h4 className="font-semibold">{service.service_name}</h4>
                    <p className="text-sm text-gray-600">{service.description}</p>
                    <span className="text-xs bg-gray-100 px-2 py-1 rounded mt-1 inline-block">
                      {service.service_category}
                    </span>
                  </div>
                  <div className="text-right">
                    <p className="text-2xl font-bold text-blue-600">{service.credits_required}</p>
                    <p className="text-xs text-gray-500">credits</p>
                    <button
                      onClick={() => handleUseService(service.service_code)}
                      disabled={(balance?.balance || 0) < service.credits_required || usageMutation.isPending}
                      className="mt-2 px-4 py-1 bg-blue-600 text-white text-sm rounded hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      {usageMutation.isPending ? 'Processing...' : 'Use Service'}
                    </button>
                  </div>
                </div>
              ))}
            </div>
          )}

          {/* History Tab */}
          {activeTab === 'history' && (
            <div className="space-y-2">
              {transactions?.map(txn => (
                <div key={txn.id} className="flex items-center justify-between p-4 border-b hover:bg-gray-50">
                  <div>
                    <p className="font-medium">{txn.description}</p>
                    <p className="text-xs text-gray-500">
                      {new Date(txn.created_at).toLocaleString()}
                    </p>
                  </div>
                  <div className="text-right">
                    <p className={`text-lg font-bold ${
                      txn.transaction_type === 'debit' ? 'text-red-600' : 'text-green-600'
                    }`}>
                      {txn.transaction_type === 'debit' ? '-' : '+'}{txn.amount}
                    </p>
                    <p className="text-xs text-gray-500">Balance: {txn.balance_after}</p>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

const BuyerWallet: React.FC = () => {
  return (
    <QueryClientProvider client={queryClient}>
      <BuyerWalletContent />
    </QueryClientProvider>
  );
};

export default BuyerWallet;
