import React, { useState, useEffect } from "react";
import { Heart, Eye, Bookmark, MessageSquare, Calendar, TrendingUp, TrendingDown, Minus, FileText, Download, Search, Settings, X, BarChart3, PieChart, Activity } from "lucide-react";
import { LineChart, Line, BarChart, Bar, PieChart as RechartsPie, Pie, Cell, ResponsiveContainer, XAxis, YAxis, Tooltip, Legend, CartesianGrid } from 'recharts';
import { toast } from 'sonner';
import { userDashboardAPI, DashboardOverview as DashboardData } from '../../api/user/userDashboard';
import { searchHistoryAPI } from '../../api/searchHistory';
import { visitScheduleAPI, VisitSchedule } from '../../api/user/visitScheduleApi';
import { userPreferencesAPI, UserPreferences } from '../../api/user/userPreferences';
import { getUserVisitSchedules } from '../../api/visitSchedules';
import { getUserId } from '../../utils/userUtils';
import { useNavigate } from 'react-router-dom';
import { reportService } from '../../services/reportService';
import { FeatureFlagsSection } from '../FeatureFlagsSection';
import { behaviorAnalysisApi } from '../../api/behaviorAnalysis';
import { PropertyCard } from '../PropertyCard';

export function DashboardOverview() {
  const navigate = useNavigate();
  const [dashboardData, setDashboardData] = useState<DashboardData | null>(null);
  const [upcomingVisits, setUpcomingVisits] = useState<VisitSchedule[]>([]);
  const [searchHistory, setSearchHistory] = useState<any[]>([]);
  const [totalSearchCount, setTotalSearchCount] = useState(0);
  const [loading, setLoading] = useState(true);
  const [reportLoading, setReportLoading] = useState(false);
  const [showPreferencesModal, setShowPreferencesModal] = useState(false);
  const [hasPreferences, setHasPreferences] = useState(false);
  const [preferencesLoading, setPreferencesLoading] = useState(false);
  
  // New state for analytics
  const [statsWithTrends, setStatsWithTrends] = useState<any>(null);
  const [activityBreakdown, setActivityBreakdown] = useState<any[]>([]);
  const [searchTrends, setSearchTrends] = useState<any[]>([]);
  const [propertyTypeDistribution, setPropertyTypeDistribution] = useState<any[]>([]);
  
  // Real-time market stats
  const [marketStats, setMarketStats] = useState<any>(null);
  
  // Matched properties
  const [matchedProperties, setMatchedProperties] = useState<any[]>([]);
  const [totalMatches, setTotalMatches] = useState(0);
  const [matchedLoading, setMatchedLoading] = useState(false);
  
  // Engagement score calculation
  const calculateEngagementScore = () => {
    if (!dashboardData) return { total: 0, breakdown: {}, level: 'Bronze', message: 'Just getting started!', icon: '🥉' };
    
    const profileScore = Math.min(25, (dashboardData.user_stats.profile_completion || 0) / 4);
    const searchScore = Math.min(25, Math.floor(totalSearchCount / 2));
    const viewScore = Math.min(25, Math.floor((dashboardData.user_stats.viewed_properties || 0) / 1.6));
    const interactionScore = Math.min(25, Math.floor(((dashboardData.user_stats.inquiries_sent || 0) + (dashboardData.user_stats.wishlisted_properties || 0)) * 1.67));
    
    const total = Math.round(profileScore + searchScore + viewScore + interactionScore);
    
    let level = 'Bronze';
    let message = 'Just getting started!';
    let icon = '🥉';
    
    if (total >= 76) {
      level = 'Platinum';
      message = 'Outstanding!';
      icon = '💎';
    } else if (total >= 51) {
      level = 'Gold';
      message = 'Great engagement!';
      icon = '🏆';
    } else if (total >= 26) {
      level = 'Silver';
      message = 'Making progress!';
      icon = '🥈';
    }
    
    return {
      total,
      breakdown: {
        profile: Math.round(profileScore),
        search: Math.round(searchScore),
        view: Math.round(viewScore),
        interaction: Math.round(interactionScore)
      },
      level,
      message,
      icon
    };
  };
  
  // Budget utilization calculation
  const calculateBudgetUtilization = () => {
    if (!dashboardData) return 0;
    
    const searchWeight = Math.min(20, (totalSearchCount / 10) * 20);
    const viewWeight = Math.min(30, ((dashboardData.user_stats.viewed_properties || 0) / 20) * 30);
    const wishlistWeight = Math.min(25, ((dashboardData.user_stats.wishlisted_properties || 0) / 10) * 25);
    const inquiryWeight = Math.min(25, ((dashboardData.user_stats.inquiries_sent || 0) / 5) * 25);
    
    return Math.round(searchWeight + viewWeight + wishlistWeight + inquiryWeight);
  };
  
  const engagementScore = calculateEngagementScore();
  const budgetUtilization = calculateBudgetUtilization();
  
  // Fetch market stats when preferences change
  const fetchMarketStats = async () => {
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id;
      if (!userId || preferences.budget.max === 0) return;
      
      const stats = await userDashboardAPI.getMarketStats(
        userId,
        preferences.budget.min,
        preferences.budget.max,
        preferences.location
      );
      
      if (stats) {
        setMarketStats(stats);
        console.log('📊 Market stats loaded:', stats);
      }
    } catch (error) {
      console.error('Error fetching market stats:', error);
      setMarketStats(null);
    }
  };
  
  const fetchMatchedProperties = async () => {
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id;
      if (!userId) return;
      
      setMatchedLoading(true);
      const data = await userDashboardAPI.getMatchedProperties(userId, 10);
      
      setMatchedProperties(data.matched_properties || []);
      setTotalMatches(data.total_matches || 0);
      console.log('🎯 Matched properties loaded:', data.matched_properties?.length || 0);
    } catch (error) {
      console.error('Error fetching matched properties:', error);
      setMatchedProperties([]);
      setTotalMatches(0);
    } finally {
      setMatchedLoading(false);
    }
  };
  
  const [preferences, setPreferences] = useState({
    id: undefined,
    budget: { min: 0, max: 10000000 },
    location: [],
    propertyType: [],
    listingType: '',
    bedrooms: '',
    bathrooms: '',
    builtUpArea: { min: 0, max: 5000 },
    furnishing: '',
    facing: '',
    propertyAge: '',
    possessionStatus: '',
    gatedCommunity: '',
    amenities: [],
    features: [],
    specifications: [],
    highlights: [],
    timeline: ''
  });

  const handleGenerateReport = async () => {
    try {
      setReportLoading(true);
      const filename = await reportService.downloadDashboardReport();
      console.log('Report downloaded:', filename);
      toast.success(`Report downloaded: ${filename}`, {
        description: 'Your dashboard report has been generated successfully.'
      });
    } catch (error) {
      console.error('Failed to generate report:', error);
      toast.error('Failed to generate report. Please try again.');
    } finally {
      setReportLoading(false);
    }
  };

  const fetchDashboardData = async () => {
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id;
      if (!userId) return;
      
      // Fetch dashboard API and behavior events
      const [data, behaviorEvents] = await Promise.all([
        userDashboardAPI.getOverview(userId),
        behaviorAnalysisApi.getUserEvents(200, 30).catch(() => [])
      ]);
      
      console.log('📊 Dashboard API stats:', data.user_stats);
      console.log('📊 Behavior events fetched:', behaviorEvents.length);
      
      // Use API for wishlisted, enquiries, searches
      // Use behavior events ONLY for viewed
      const viewedCount = behaviorEvents.filter((e: any) => {
        const type = e.event_type.toLowerCase();
        return type === 'property_view' || type === 'property_click';
      }).length;
      
      console.log('📊 Viewed from behavior events:', viewedCount);
      
      // Override only viewed count
      data.user_stats.viewed_properties = viewedCount;
      
      setDashboardData(data);
      
      // Generate trends
      const generateTrends = (currentValue: number, eventTypes: string[], useEvents: boolean) => {
        let history = [];
        
        if (useEvents) {
          // Use behavior events for viewed
          const last14Days = behaviorEvents.filter((e: any) => {
            const eventDate = new Date(e.created_at);
            const daysAgo = Math.floor((Date.now() - eventDate.getTime()) / (1000 * 60 * 60 * 24));
            const type = e.event_type.toLowerCase();
            return daysAgo <= 14 && eventTypes.some(t => type === t.toLowerCase());
          });
          
          const dayGroups: any = {};
          for (let i = 6; i >= 0; i--) {
            const date = new Date();
            date.setDate(date.getDate() - i);
            const dayKey = date.toISOString().split('T')[0];
            dayGroups[dayKey] = 0;
          }
          
          last14Days.forEach((event: any) => {
            const dayKey = new Date(event.created_at).toISOString().split('T')[0];
            if (dayGroups[dayKey] !== undefined) {
              dayGroups[dayKey]++;
            }
          });
          
          history = Object.values(dayGroups).map((v, i) => ({ day: i, value: v as number }));
        } else {
          // For non-event metrics, create flat history with current value
          history = Array(7).fill(0).map((_, i) => ({ day: i, value: 0 }));
        }
        
        return {
          current: currentValue,
          change: 0,
          trend: 'stable' as const,
          history
        };
      };
      
      const trends = {
        wishlisted: generateTrends(data.user_stats.wishlisted_properties || 0, ['wishlist_add', 'save_property'], true),
        viewed: generateTrends(viewedCount, ['property_view', 'property_click'], true),
        enquiries: generateTrends(data.user_stats.inquiries_sent || 0, ['inquiry_sent', 'contact_click'], true),
        searches: generateTrends(totalSearchCount, ['search', 'filter_apply'], true)
      };
      
      console.log('📊 Final trends:', trends);
      setStatsWithTrends(trends);
      
      // Activity breakdown from behavior events
      const activityCounts: any = {};
      behaviorEvents.forEach((event: any) => {
        const type = event.event_type.replace(/_/g, ' ');
        activityCounts[type] = (activityCounts[type] || 0) + 1;
      });
      
      const breakdown = Object.entries(activityCounts)
        .map(([name, value]) => ({
          name: name.charAt(0).toUpperCase() + name.slice(1),
          value
        }))
        .sort((a: any, b: any) => b.value - a.value)
        .slice(0, 5);
      setActivityBreakdown(breakdown);
      
    } catch (error) {
      console.error('Error fetching dashboard data:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchSearchHistory = async () => {
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id || user.id;
      if (!userId) {
        setSearchHistory([]);
        setTotalSearchCount(0);
        return;
      }
      
      // Fetch last 5 searches for display
      const history = await searchHistoryAPI.getUserHistory(userId, 5);
      setSearchHistory(Array.isArray(history) ? history : []);
      
      // Fetch total count for stats card
      const count = await searchHistoryAPI.getUserSearchCount(userId);
      setTotalSearchCount(count);
      
      console.log('📊 Search history loaded:', history.length, 'Total count:', count);
      
      // Fetch real behavior events for last 7 days
      const behaviorEvents = await behaviorAnalysisApi.getUserEvents(200, 7).catch(() => []);
      
      console.log('📊 Behavior Events Fetched:', behaviorEvents.length);
      console.log('📊 Event Types:', [...new Set(behaviorEvents.map((e: any) => e.event_type))]);
      
      // Create date map for last 7 days using YYYY-MM-DD format for reliable matching
      const today = new Date();
      today.setHours(0, 0, 0, 0);
      const dateMap: any = {};
      const displayDates: any = {};
      
      for (let i = 6; i >= 0; i--) {
        const date = new Date(today);
        date.setDate(date.getDate() - i);
        const dateKey = date.toISOString().split('T')[0]; // YYYY-MM-DD format
        const displayKey = date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
        dateMap[dateKey] = { date: displayKey, searches: 0, views: 0, wishlists: 0 };
        displayDates[dateKey] = displayKey;
      }
      
      console.log('📊 Date Map Keys:', Object.keys(dateMap));
      
      // Count events by type and day using normalized date
      behaviorEvents.forEach((event: any) => {
        try {
          const eventDate = new Date(event.created_at);
          const dateKey = eventDate.toISOString().split('T')[0]; // YYYY-MM-DD format
          
          console.log('📊 Event date:', event.created_at, '-> normalized:', dateKey, 'type:', event.event_type);
          
          if (dateMap[dateKey]) {
            const eventType = event.event_type.toLowerCase();
            
            if (eventType === 'search' || eventType === 'filter_apply') {
              dateMap[dateKey].searches++;
            }
            if (eventType === 'property_view' || eventType === 'property_click') {
              dateMap[dateKey].views++;
            }
            if (eventType === 'wishlist_add' || eventType === 'save_property') {
              dateMap[dateKey].wishlists++;
            }
          }
        } catch (err) {
          console.error('📊 Error processing event:', err, event);
        }
      });
      
      const trends = Object.values(dateMap);
      console.log('📊 Final Search Trends Data:', trends);
      console.log('📊 Total searches:', trends.reduce((sum: number, day: any) => sum + day.searches, 0));
      console.log('📊 Total views:', trends.reduce((sum: number, day: any) => sum + day.views, 0));
      console.log('📊 Total wishlists:', trends.reduce((sum: number, day: any) => sum + day.wishlists, 0));
      
      setSearchTrends(trends);
      
    } catch (error) {
      console.error('Error fetching search history:', error);
      setSearchHistory([]);
      setSearchTrends([]);
    }
  };

  const fetchNextVisit = async () => {
    try {
      const userId = getUserId();
      if (!userId) {
        console.log('DashboardOverview: No user ID found');
        return;
      }
      
      console.log('DashboardOverview: Fetching visits for user:', userId);
      const response = await getUserVisitSchedules(userId, 1, 50);
      console.log('DashboardOverview: API response:', response);
      
      if (response.visits.length > 0) {
        console.log('DashboardOverview: All visits:', response.visits);
        const filteredVisits = response.visits.filter(visit => {
          const isApproved = visit.status === 'seller_approved';
          const visitDate = new Date(visit.preferred_date);
          const today = new Date();
          today.setHours(0, 0, 0, 0); // Reset time to start of day
          const isFuture = visitDate >= today;
          console.log(`DashboardOverview: Visit ${visit.id} - status: ${visit.status}, date: ${visit.preferred_date}, visitDate: ${visitDate}, today: ${today}, isApproved: ${isApproved}, isFuture: ${isFuture}`);
          return isApproved && isFuture;
        });
        
        console.log('DashboardOverview: Filtered visits:', filteredVisits);
        
        filteredVisits.sort((a, b) => {
          const dateA = new Date(`${a.preferred_date} ${a.preferred_time}`);
          const dateB = new Date(`${b.preferred_date} ${b.preferred_time}`);
          return dateA.getTime() - dateB.getTime();
        });
        
        setUpcomingVisits(filteredVisits);
      }
    } catch (error) {
      console.error('Error fetching next visit:', error);
    }
  };

  const fetchUserPreferences = async () => {
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id;
      if (!userId) return;
      
      const userPrefs = await userPreferencesAPI.get(userId);
      if (userPrefs) {
        setHasPreferences(true);
        setPreferences({
          id: userPrefs.id,
          budget: { 
            min: userPrefs.budget_min || 0, 
            max: userPrefs.budget_max || 10000000 
          },
          location: userPrefs.location || [],
          propertyType: userPrefs.property_type || [],
          listingType: userPrefs.purpose || '',
          bedrooms: userPrefs.bedrooms || '',
          bathrooms: userPrefs.bathrooms || '',
          builtUpArea: { min: 0, max: 5000 },
          furnishing: userPrefs.furnishing || '',
          facing: userPrefs.facing || '',
          propertyAge: userPrefs.age_of_property || '',
          possessionStatus: userPrefs.possession || '',
          gatedCommunity: '',
          amenities: userPrefs.amenities || [],
          features: [],
          specifications: [],
          highlights: [],
          timeline: userPrefs.timeline || ''
        });
      } else {
        setHasPreferences(false);
      }
    } catch (error) {
      console.error('Error fetching user preferences:', error);
      setHasPreferences(false);
    }
  };

  const handleSavePreferences = async () => {
    try {
      setPreferencesLoading(true);
      
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id;
      
      if (!userId) {
        toast.error('User ID not found');
        return;
      }

      const prefsData: any = {
        user_id: userId.toString()
      };
      
      // Only add fields that have values - match backend JSON format
      if (preferences.budget.min > 0) prefsData.budget_min = Number(preferences.budget.min);
      if (preferences.budget.max > 0) prefsData.budget_max = Number(preferences.budget.max);
      if (preferences.location.length > 0) prefsData.location = preferences.location;
      if (preferences.listingType) prefsData.listing_type = preferences.listingType;
      if (preferences.propertyType.length > 0) prefsData.property_types = preferences.propertyType; // property_types not property_type
      if (preferences.bedrooms) prefsData.bedrooms = preferences.bedrooms;
      if (preferences.bathrooms) prefsData.bathrooms = preferences.bathrooms;
      if (preferences.builtUpArea.min > 0) prefsData.built_up_area_min = Number(preferences.builtUpArea.min);
      if (preferences.builtUpArea.max > 0) prefsData.built_up_area_max = Number(preferences.builtUpArea.max);
      if (preferences.furnishing) prefsData.furnishing = preferences.furnishing;
      if (preferences.facing) prefsData.facing = preferences.facing;
      if (preferences.propertyAge) prefsData.age_of_property = preferences.propertyAge;
      if (preferences.possessionStatus) prefsData.possession = preferences.possessionStatus;
      if (preferences.gatedCommunity) prefsData.gated_community = preferences.gatedCommunity;
      if (preferences.amenities.length > 0) prefsData.specifications = preferences.amenities; // Preferred Amenities -> specifications
      if (preferences.features.length > 0) prefsData.features = preferences.features;
      if (preferences.specifications.length > 0) prefsData.amenities = preferences.specifications; // Keep original amenities mapping
      if (preferences.highlights.length > 0) prefsData.highlights = preferences.highlights;
      if (preferences.timeline) prefsData.timeline = preferences.timeline;
      prefsData.purpose = "residence"; // Fixed value as per your example

      console.log('Saving preferences:', prefsData);

      if (hasPreferences) {
        // Add ID for update
        prefsData.id = preferences.id;
        const result = await userPreferencesAPI.update(prefsData);
        console.log('Update result:', result);
        toast.success('Preferences updated successfully!');
      } else {
        const result = await userPreferencesAPI.create(prefsData);
        console.log('Create result:', result);
        toast.success('Preferences saved successfully!');
        setHasPreferences(true);
      }
      
      setShowPreferencesModal(false);
      
      // Fetch market stats and matched properties with new preferences
      await fetchMarketStats();
      await fetchMatchedProperties();
    } catch (error: any) {
      console.error('Error saving preferences:', error);
      
      // Try to get detailed error from response
      if (error.response?.data) {
        console.error('Backend validation errors:', JSON.stringify(error.response.data, null, 2));
      }
      
      let errorMessage = 'Failed to save preferences';
      
      if (error.response?.data?.detail) {
        if (Array.isArray(error.response.data.detail)) {
          errorMessage = error.response.data.detail.map((err: any) => {
            if (typeof err === 'string') return err;
            return err.msg || JSON.stringify(err);
          }).join(', ');
        } else if (typeof error.response.data.detail === 'string') {
          errorMessage = error.response.data.detail;
        } else {
          errorMessage = JSON.stringify(error.response.data.detail);
        }
      } else if (error.response?.data?.message) {
        errorMessage = typeof error.response.data.message === 'string' ? 
                     error.response.data.message : 
                     JSON.stringify(error.response.data.message);
      } else if (error.message) {
        errorMessage = error.message;
      }
      
      toast.error(String(errorMessage));
    } finally {
      setPreferencesLoading(false);
    }
  };

  useEffect(() => {
    fetchDashboardData();
    fetchNextVisit();
    fetchSearchHistory();
    fetchUserPreferences();
    fetchMatchedProperties();

    const handleRefresh = () => {
      fetchDashboardData();
      fetchNextVisit();
      fetchSearchHistory();
      fetchUserPreferences();
      fetchMatchedProperties();
    };

    window.addEventListener('refreshDashboard', handleRefresh);
    return () => window.removeEventListener('refreshDashboard', handleRefresh);
  }, []);
  
  // Fetch market stats when preferences change
  useEffect(() => {
    if (preferences.budget.max > 0) {
      fetchMarketStats();
    }
  }, [preferences.budget.min, preferences.budget.max, preferences.location]);
  
  // Real-time polling for dashboard updates (every 30 seconds)
  useEffect(() => {
    const pollInterval = setInterval(() => {
      if (!document.hidden) {
        fetchDashboardData();
        fetchMatchedProperties();
        if (preferences.budget.max > 0) {
          fetchMarketStats();
        }
      }
    }, 30000);
    
    // Stop polling when page is hidden
    const handleVisibilityChange = () => {
      if (document.hidden) {
        clearInterval(pollInterval);
      }
    };
    
    document.addEventListener('visibilitychange', handleVisibilityChange);
    
    return () => {
      clearInterval(pollInterval);
      document.removeEventListener('visibilitychange', handleVisibilityChange);
    };
  }, [preferences.budget.min, preferences.budget.max, preferences.location]);
  
  // Generate property type distribution when dashboard data loads
  useEffect(() => {
    if (dashboardData?.recent_activities) {
      // Fetch real behavior events to get property types
      behaviorAnalysisApi.getUserEvents(100, 30).then((events) => {
        const propertyTypes: any = {};
        events.forEach((event: any) => {
          if (event.event_data?.property_type) {
            const type = event.event_data.property_type;
            propertyTypes[type] = (propertyTypes[type] || 0) + 1;
          }
        });
        
        const distribution = Object.entries(propertyTypes).map(([name, value]) => ({
          name: name.charAt(0).toUpperCase() + name.slice(1),
          value
        }));
        setPropertyTypeDistribution(distribution);
      }).catch(() => {
        // Fallback to activity data if behavior events fail
        const propertyTypes: any = {};
        dashboardData.recent_activities.forEach((activity: any) => {
          if (activity.details?.property_type) {
            const type = activity.details.property_type;
            propertyTypes[type] = (propertyTypes[type] || 0) + 1;
          }
        });
        
        const distribution = Object.entries(propertyTypes).map(([name, value]) => ({
          name: name.charAt(0).toUpperCase() + name.slice(1),
          value
        }));
        setPropertyTypeDistribution(distribution);
      });
    }
  }, [dashboardData]);
  return (
    <div className="space-y-6">
      {/* Modern Hero Header */}
      <div className="bg-[#0056D2] rounded-2xl shadow-xl p-8">
        <div className="flex items-center justify-between">
          <div>
            <h2 className="text-3xl font-bold text-white flex items-center gap-3">
              Dashboard Overview
              {!loading && (
                <span className="inline-flex items-center gap-2 px-3 py-1.5 text-xs font-semibold bg-white/20 text-white rounded-full backdrop-blur-md border border-white/30 animate-pulse">
                  <span className="w-2 h-2 bg-blue-400 rounded-full shadow-lg shadow-blue-400/50"></span>
                  LIVE
                </span>
              )}
            </h2>
            <p className="text-white/90 mt-2 text-lg">Your property search activity at a glance</p>
          </div>
          <div className="flex gap-4">
            <button 
              onClick={() => setShowPreferencesModal(true)}
              className="flex items-center gap-2 px-6 py-3 bg-white text-[#0056D2] rounded-xl hover:bg-white/90 transition-all shadow-lg hover:shadow-xl transform hover:-translate-y-1 hover:scale-105 duration-200"
            >
              <Settings className="w-5 h-5" />
              <span className="font-semibold">{hasPreferences ? 'Update Preferences' : 'Add Preferences'}</span>
            </button>
          </div>
        </div>
      </div>

      {/* Quick Action Toolbar */}
      <div className="bg-[#0056D2] rounded-2xl shadow-xl p-6">
        <div className="flex flex-wrap gap-4 justify-center">
          <button 
            onClick={() => navigate('/properties')}
            className="flex items-center gap-2 px-6 py-3 bg-white text-[#0056D2] rounded-xl hover:bg-white/90 transition-all shadow-lg hover:shadow-xl transform hover:-translate-y-1 hover:scale-105 duration-200"
          >
            <Search className="w-5 h-5" />
            <span className="font-semibold">Search Properties</span>
          </button>
          <button 
            onClick={() => navigate('/user/dashboard?section=Visits')}
            className="flex items-center gap-2 px-6 py-3 bg-white text-[#0056D2] rounded-xl hover:bg-white/90 transition-all shadow-lg hover:shadow-xl transform hover:-translate-y-1 hover:scale-105 duration-200"
          >
            <Calendar className="w-5 h-5" />
            <span className="font-semibold">Schedule Visit</span>
          </button>
          <button 
            onClick={() => navigate('/user/dashboard?section=Properties&tab=recommendations')}
            className="flex items-center gap-2 px-6 py-3 bg-white text-[#0056D2] rounded-xl hover:bg-white/90 transition-all shadow-lg hover:shadow-xl transform hover:-translate-y-1 hover:scale-105 duration-200"
          >
            <TrendingUp className="w-5 h-5" />
            <span className="font-semibold">View Recommendations</span>
          </button>
          <button 
            onClick={() => navigate('/budget-calculator')}
            className="flex items-center gap-2 px-6 py-3 bg-white text-[#0056D2] rounded-xl hover:bg-white/90 transition-all shadow-lg hover:shadow-xl transform hover:-translate-y-1 hover:scale-105 duration-200"
          >
            <Settings className="w-5 h-5" />
            <span className="font-semibold">Budget Calculator</span>
          </button>
        </div>
      </div>

      {/* Enhanced Stats Cards with Trends */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {/* Wishlisted Card */}
        <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200 hover:shadow-md transition-shadow">
          <div className="flex items-center justify-between mb-3">
            <div className="flex-1">
              <p className="text-sm font-medium text-gray-600">Wishlisted</p>
              <div className="flex items-baseline gap-2 mt-1">
                <p className="text-3xl font-bold text-gray-900">
                  {loading ? '...' : dashboardData?.user_stats.wishlisted_properties || 0}
                </p>
              </div>
            </div>
            <div className="w-12 h-12 bg-red-50 rounded-lg flex items-center justify-center">
              <Heart className="w-6 h-6 text-red-600" />
            </div>
          </div>
          {!loading && statsWithTrends?.wishlisted?.history && (
            <ResponsiveContainer width="100%" height={40}>
              <LineChart data={statsWithTrends.wishlisted.history}>
                <Line type="monotone" dataKey="value" stroke="#EF4444" strokeWidth={2} dot={false} />
              </LineChart>
            </ResponsiveContainer>
          )}
        </div>

        {/* Viewed Card */}
        <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200 hover:shadow-md transition-shadow">
          <div className="flex items-center justify-between mb-3">
            <div className="flex-1">
              <p className="text-sm font-medium text-gray-600">Viewed</p>
              <div className="flex items-baseline gap-2 mt-1">
                <p className="text-3xl font-bold text-gray-900">
                  {loading ? '...' : statsWithTrends?.viewed?.current || dashboardData?.user_stats.viewed_properties || 0}
                </p>
              </div>
            </div>
            <div className="w-12 h-12 bg-blue-50 rounded-lg flex items-center justify-center">
              <Eye className="w-6 h-6 text-blue-600" />
            </div>
          </div>
          {!loading && statsWithTrends?.viewed?.history && (
            <ResponsiveContainer width="100%" height={40}>
              <LineChart data={statsWithTrends.viewed.history}>
                <Line type="monotone" dataKey="value" stroke="#3B82F6" strokeWidth={2} dot={false} />
              </LineChart>
            </ResponsiveContainer>
          )}
        </div>

        {/* Enquiries Card */}
        <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200 hover:shadow-md transition-shadow">
          <div className="flex items-center justify-between mb-3">
            <div className="flex-1">
              <p className="text-sm font-medium text-gray-600">Enquiries</p>
              <div className="flex items-baseline gap-2 mt-1">
                <p className="text-3xl font-bold text-gray-900">
                  {loading ? '...' : dashboardData?.user_stats.inquiries_sent || 0}
                </p>
              </div>
            </div>
            <div className="w-12 h-12 bg-green-50 rounded-lg flex items-center justify-center">
              <MessageSquare className="w-6 h-6 text-green-600" />
            </div>
          </div>
          {!loading && statsWithTrends?.enquiries?.history && (
            <ResponsiveContainer width="100%" height={40}>
              <LineChart data={statsWithTrends.enquiries.history}>
                <Line type="monotone" dataKey="value" stroke="#10B981" strokeWidth={2} dot={false} />
              </LineChart>
            </ResponsiveContainer>
          )}
        </div>

        {/* Searches Card */}
        <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200 hover:shadow-md transition-shadow">
          <div className="flex items-center justify-between mb-3">
            <div className="flex-1">
              <p className="text-sm font-medium text-gray-600">Searches</p>
              <div className="flex items-baseline gap-2 mt-1">
                <p className="text-3xl font-bold text-gray-900">{totalSearchCount}</p>
              </div>
            </div>
            <div className="w-12 h-12 bg-purple-50 rounded-lg flex items-center justify-center">
              <Search className="w-6 h-6 text-purple-600" />
            </div>
          </div>
          {!loading && statsWithTrends?.searches?.history && (
            <ResponsiveContainer width="100%" height={40}>
              <LineChart data={statsWithTrends.searches.history}>
                <Line type="monotone" dataKey="value" stroke="#8B5CF6" strokeWidth={2} dot={false} />
              </LineChart>
            </ResponsiveContainer>
          )}
        </div>
      </div>

      {/* Activity Breakdown Bar Chart & Search Trends */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Activity Breakdown */}
        {!loading && activityBreakdown.length > 0 && (
          <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200">
            <div className="flex items-center gap-2 mb-4">
              <BarChart3 className="w-5 h-5 text-blue-600" />
              <h3 className="text-lg font-semibold text-gray-900">Activity Breakdown</h3>
            </div>
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={activityBreakdown} layout="vertical">
                <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
                <XAxis type="number" />
                <YAxis dataKey="name" type="category" width={120} />
                <Tooltip 
                  contentStyle={{ backgroundColor: '#fff', border: '1px solid #e5e7eb', borderRadius: '8px' }}
                  cursor={{ fill: 'rgba(59, 130, 246, 0.1)' }}
                />
                <Bar dataKey="value" radius={[0, 8, 8, 0]}>
                  {activityBreakdown.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={[
                      '#3B82F6', '#10B981', '#F59E0B', '#EF4444', '#8B5CF6'
                    ][index % 5]} />
                  ))}
                </Bar>
              </BarChart>
            </ResponsiveContainer>
          </div>
        )}

        {/* Property Type Distribution Pie Chart */}
        {!loading && (
          <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200">
            <div className="flex items-center gap-2 mb-4">
              <PieChart className="w-5 h-5 text-purple-600" />
              <h3 className="text-lg font-semibold text-gray-900">Property Type Interest</h3>
            </div>
            {propertyTypeDistribution.length > 0 ? (
              <ResponsiveContainer width="100%" height={300}>
                <RechartsPie>
                  <Pie
                    data={propertyTypeDistribution}
                    cx="50%"
                    cy="50%"
                    labelLine={false}
                    label={({ name, percent }) => `${name} ${(percent * 100).toFixed(0)}%`}
                    outerRadius={100}
                    fill="#8884d8"
                    dataKey="value"
                  >
                    {propertyTypeDistribution.map((entry, index) => (
                      <Cell key={`cell-${index}`} fill={[
                        '#3B82F6', '#10B981', '#F59E0B', '#EF4444', '#8B5CF6', '#EC4899'
                      ][index % 6]} />
                    ))}
                  </Pie>
                  <Tooltip />
                  <Legend />
                </RechartsPie>
              </ResponsiveContainer>
            ) : (
              <div className="flex flex-col items-center justify-center h-[300px] text-center">
                <PieChart className="w-16 h-16 text-gray-300 mb-4" />
                <p className="text-gray-600 font-medium mb-2">No property data yet</p>
                <p className="text-sm text-gray-500 mb-4">Start exploring properties to see your interests</p>
                <button
                  onClick={() => setShowPreferencesModal(true)}
                  className="px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors text-sm font-medium"
                >
                  Add Your Preferences
                </button>
              </div>
            )}
          </div>
        )}
      </div>



      {/* Financial Dashboard & Engagement Score */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Financial Overview */}
        {!loading && preferences.budget.max > 0 && (
          <div className="bg-gradient-to-br from-green-50 to-blue-50 rounded-lg shadow-sm p-6 border border-green-200">
            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
              <TrendingUp className="w-5 h-5 text-green-600" />
              Financial Overview
            </h3>
            <div className="grid grid-cols-2 gap-4">
              <div className="bg-white rounded-lg p-4 shadow-sm">
                <p className="text-sm text-gray-600 mb-1">Budget Range</p>
                <p className="text-lg font-bold text-gray-900">
                  ₹{(preferences.budget.min / 100000).toFixed(1)}L - ₹{(preferences.budget.max / 100000).toFixed(1)}L
                </p>
              </div>
              <div className="bg-white rounded-lg p-4 shadow-sm">
                <p className="text-sm text-gray-600 mb-1">
                  {marketStats ? 'Market Avg Price' : 'Budget Midpoint'}
                  {!marketStats && (
                    <span className="ml-1 text-xs text-gray-400" title="Average of your min and max budget">ⓘ</span>
                  )}
                </p>
                <p className="text-lg font-bold text-blue-600">
                  ₹{marketStats 
                    ? (marketStats.avg_market_price / 100000).toFixed(1)
                    : ((preferences.budget.min + preferences.budget.max) / 2 / 100000).toFixed(1)
                  }L
                </p>
                {marketStats && (
                  <p className="text-xs text-gray-500 mt-1">Real market data</p>
                )}
              </div>
              <div className="bg-white rounded-lg p-4 shadow-sm">
                <p className="text-sm text-gray-600 mb-1">
                  {marketStats ? 'Available in Budget' : 'Properties Viewed'}
                </p>
                <p className="text-lg font-bold text-green-600">
                  {marketStats 
                    ? marketStats.total_available_properties
                    : (dashboardData?.user_stats.viewed_properties || 0)
                  }
                </p>
              </div>
              <div className="bg-white rounded-lg p-4 shadow-sm">
                <p className="text-sm text-gray-600 mb-1">
                  {marketStats && marketStats.potential_savings > 0 ? 'Budget Headroom' : 'Viewed in Budget'}
                </p>
                <p className="text-lg font-bold text-orange-600">
                  {marketStats && marketStats.potential_savings > 0
                    ? `₹${(marketStats.potential_savings / 100000).toFixed(1)}L`
                    : (marketStats?.properties_viewed_in_budget || dashboardData?.user_stats.viewed_properties || 0)
                  }
                </p>
              </div>
            </div>
            <div className="mt-4 p-3 bg-white rounded-lg">
              <p className="text-xs text-gray-600 mb-2">Search Activity Progress</p>
              <div className="w-full bg-gray-200 rounded-full h-3">
                <div className="bg-gradient-to-r from-green-500 to-blue-500 h-3 rounded-full" style={{ width: `${budgetUtilization}%` }}></div>
              </div>
              <p className="text-xs text-gray-500 mt-1">{budgetUtilization}% activity completed</p>
            </div>
          </div>
        )}

        {/* Engagement Score */}
        {!loading && (
          <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg shadow-sm p-6 border border-blue-200">
            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
              <Activity className="w-5 h-5 text-blue-600" />
              Your Engagement Score
            </h3>
            <div className="flex items-center justify-center mb-4">
              <div className="relative w-32 h-32">
                <svg className="transform -rotate-90 w-32 h-32">
                  <circle cx="64" cy="64" r="56" stroke="#E5E7EB" strokeWidth="8" fill="none" />
                  <circle 
                    cx="64" 
                    cy="64" 
                    r="56" 
                    stroke="url(#gradient)" 
                    strokeWidth="8" 
                    fill="none"
                    strokeDasharray={`${2 * Math.PI * 56}`}
                    strokeDashoffset={`${2 * Math.PI * 56 * (1 - engagementScore.total / 100)}`}
                    strokeLinecap="round"
                  />
                  <defs>
                    <linearGradient id="gradient" x1="0%" y1="0%" x2="100%" y2="100%">
                      <stop offset="0%" stopColor="#3B82F6" />
                      <stop offset="100%" stopColor="#1D4ED8" />
                    </linearGradient>
                  </defs>
                </svg>
                <div className="absolute inset-0 flex items-center justify-center">
                  <div className="text-center">
                    <p className="text-3xl font-bold text-gray-900">{engagementScore.total}</p>
                    <p className="text-xs text-gray-600">Score</p>
                  </div>
                </div>
              </div>
            </div>
            <div className="space-y-2">
              <div className="flex items-center justify-between text-sm">
                <span className="text-gray-600">Profile Complete</span>
                <span className="font-semibold text-green-600">{engagementScore.breakdown.profile}/25</span>
              </div>
              <div className="flex items-center justify-between text-sm">
                <span className="text-gray-600">Search Activity</span>
                <span className="font-semibold text-blue-600">{engagementScore.breakdown.search}/25</span>
              </div>
              <div className="flex items-center justify-between text-sm">
                <span className="text-gray-600">Property Views</span>
                <span className="font-semibold text-purple-600">{engagementScore.breakdown.view}/25</span>
              </div>
              <div className="flex items-center justify-between text-sm">
                <span className="text-gray-600">Interactions</span>
                <span className="font-semibold text-orange-600">{engagementScore.breakdown.interaction}/25</span>
              </div>
            </div>
            <div className="mt-4 p-3 bg-white rounded-lg text-center">
              <p className="text-sm font-semibold text-blue-600">{engagementScore.icon} {engagementScore.level} Level</p>
              <p className="text-xs text-gray-600 mt-1">{engagementScore.message}</p>
            </div>
          </div>
        )}
      </div>

      {/* Matched Properties Section */}
      {!loading && (
        <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200">
          <div className="flex items-center justify-between mb-4">
            <div className="flex items-center gap-2">
              <span className="text-2xl">🎯</span>
              <h3 className="text-lg font-semibold text-gray-900">
                Properties Matching Your Preferences
              </h3>
            </div>
            {totalMatches > 0 && (
              <div className="text-sm text-gray-600">
                Showing {matchedProperties.length} of {totalMatches} matches
              </div>
            )}
          </div>
          
          {matchedLoading ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {[1, 2, 3].map(i => (
                <div key={i} className="border rounded-lg p-4 animate-pulse">
                  <div className="h-48 bg-gray-200 rounded mb-4"></div>
                  <div className="h-4 bg-gray-200 rounded w-3/4 mb-2"></div>
                  <div className="h-3 bg-gray-200 rounded w-1/2 mb-4"></div>
                  <div className="h-8 bg-gray-200 rounded"></div>
                </div>
              ))}
            </div>
          ) : matchedProperties.length > 0 ? (
            <>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {matchedProperties.map((property: any) => (
                  <div key={property.property_id} className="relative">
                    {/* Match Score Badge */}
                    <div className="absolute top-2 right-2 z-10">
                      <span className={`px-3 py-1 rounded-full text-xs font-bold shadow-lg ${
                        property.match_score >= 80 ? 'bg-green-500 text-white' :
                        property.match_score >= 60 ? 'bg-yellow-500 text-white' :
                        'bg-gray-500 text-white'
                      }`}>
                        {property.match_score}% Match
                      </span>
                    </div>
                    
                    <PropertyCard
                      id={property.property_id}
                      title={property.title}
                      location={`${property.locality}, ${property.city}`}
                      beds={property.bedrooms}
                      baths={property.bathrooms}
                      area={property.built_up_area ? `${property.built_up_area} sq ft` : 'N/A'}
                      price={property.sale_price 
                        ? `₹${(property.sale_price / 100000).toFixed(1)}L`
                        : property.monthly_rent 
                        ? `₹${(property.monthly_rent / 1000).toFixed(0)}K/mo`
                        : 'Price on request'
                      }
                      imageUrl={property.image_url}
                      amenities={property.amenities || []}
                    />
                  </div>
                ))}
              </div>
              
              {totalMatches > matchedProperties.length && (
                <div className="mt-4 text-center">
                  <button 
                    onClick={() => navigate('/properties')}
                    className="px-6 py-2 text-blue-600 hover:text-blue-700 font-medium hover:underline"
                  >
                    View All {totalMatches} Matches →
                  </button>
                </div>
              )}
            </>
          ) : (
            <div className="bg-gray-50 rounded-lg p-8 text-center border border-gray-200">
              <span className="text-4xl mb-2 block">🔍</span>
              <h3 className="text-lg font-semibold text-gray-900 mb-2">
                {hasPreferences ? 'No Matches Found' : 'Set Preferences to See Matches'}
              </h3>
              <p className="text-gray-600 mb-4">
                {hasPreferences 
                  ? 'Try adjusting your budget, location, or other criteria'
                  : 'Tell us what you\'re looking for and we\'ll show you the best properties'
                }
              </p>
              <button 
                onClick={() => hasPreferences ? navigate('/properties') : setShowPreferencesModal(true)}
                className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 font-medium"
              >
                {hasPreferences ? 'Browse All Properties' : 'Set Preferences'}
              </button>
            </div>
          )}
        </div>
      )}

      {/* Preferences Modal */}
      {showPreferencesModal && (
        <div className="fixed inset-0 backdrop-blur-sm flex items-start justify-center z-50 py-4 overflow-y-auto">
          <div className="bg-white rounded-xl p-6 w-full max-w-4xl shadow-2xl border mx-4">
            <div className="flex justify-between items-center mb-4">
              <h3 className="text-xl font-bold text-gray-900">Set Your Preferences</h3>
              <button 
                onClick={() => setShowPreferencesModal(false)}
                className="text-gray-400 hover:text-gray-600"
              >
                <X className="w-6 h-6" />
              </button>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {/* Budget */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Budget Range (₹)</label>
                <div className="grid grid-cols-2 gap-2">
                  <input type="number" placeholder="Min Budget" className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.budget.min} onChange={(e) => setPreferences(prev => ({ ...prev, budget: { ...prev.budget, min: Number(e.target.value) } }))} />
                  <input type="number" placeholder="Max Budget" className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.budget.max} onChange={(e) => setPreferences(prev => ({ ...prev, budget: { ...prev.budget, max: Number(e.target.value) } }))} />
                </div>
              </div>

              {/* Listing Type */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Listing Type</label>
                <select className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.listingType} onChange={(e) => setPreferences(prev => ({ ...prev, listingType: e.target.value }))}>
                  <option value="">Select Listing Type</option>
                  <option value="sell">Buy</option>
                  <option value="rent">Rent</option>
                  <option value="lease">Lease</option>
                </select>
              </div>

              {/* Location */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Preferred Locations</label>
                <input 
                  type="text" 
                  placeholder="Enter cities/areas (comma separated)" 
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                  value={preferences.location.join(', ')} 
                  onChange={(e) => setPreferences(prev => ({ ...prev, location: e.target.value.split(',').map(loc => loc.trim()).filter(loc => loc) }))}
                />
              </div>

              {/* Property Type */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Property Type</label>
                <div className="grid grid-cols-2 gap-2">
                  {['apartment', 'villa', 'house', 'commercial', 'pg-coliving'].map(type => (
                    <label key={type} className="flex items-center text-sm">
                      <input 
                        type="checkbox" 
                        className="mr-2" 
                        checked={preferences.propertyType.includes(type)}
                        onChange={(e) => { 
                          if (e.target.checked) { 
                            setPreferences(prev => ({ ...prev, propertyType: [...prev.propertyType, type] })); 
                          } else { 
                            setPreferences(prev => ({ ...prev, propertyType: prev.propertyType.filter(t => t !== type) })); 
                          } 
                        }} 
                      />
                      {type.charAt(0).toUpperCase() + type.slice(1).replace('-', '/')}
                    </label>
                  ))}
                </div>
              </div>

              {/* Built-up Area */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Built-up Area (sq ft)</label>
                <div className="grid grid-cols-2 gap-2">
                  <input type="number" placeholder="Min Area" className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.builtUpArea.min} onChange={(e) => setPreferences(prev => ({ ...prev, builtUpArea: { ...prev.builtUpArea, min: Number(e.target.value) } }))} />
                  <input type="number" placeholder="Max Area" className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.builtUpArea.max} onChange={(e) => setPreferences(prev => ({ ...prev, builtUpArea: { ...prev.builtUpArea, max: Number(e.target.value) } }))} />
                </div>
              </div>

              {/* Bedrooms */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Bedrooms</label>
                <select className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.bedrooms} onChange={(e) => setPreferences(prev => ({ ...prev, bedrooms: e.target.value }))}>
                  <option value="">Select Bedrooms</option>
                  <option value="1">1 BHK</option>
                  <option value="2">2 BHK</option>
                  <option value="3">3 BHK</option>
                  <option value="4">4 BHK</option>
                  <option value="5">5+ BHK</option>
                </select>
              </div>

              {/* Bathrooms */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Bathrooms</label>
                <select className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.bathrooms} onChange={(e) => setPreferences(prev => ({ ...prev, bathrooms: e.target.value }))}>
                  <option value="">Select Bathrooms</option>
                  <option value="1">1</option>
                  <option value="2">2</option>
                  <option value="3">3</option>
                  <option value="4">4+</option>
                </select>
              </div>

              {/* Furnishing */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Furnishing</label>
                <select className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.furnishing} onChange={(e) => setPreferences(prev => ({ ...prev, furnishing: e.target.value }))}>
                  <option value="">Select Furnishing</option>
                  <option value="unfurnished">Unfurnished</option>
                  <option value="semi-furnished">Semi-Furnished</option>
                  <option value="fully-furnished">Fully Furnished</option>
                </select>
              </div>

              {/* Facing */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Facing</label>
                <select className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.facing} onChange={(e) => setPreferences(prev => ({ ...prev, facing: e.target.value }))}>
                  <option value="">Select Facing</option>
                  <option value="north">North</option>
                  <option value="south">South</option>
                  <option value="east">East</option>
                  <option value="west">West</option>
                  <option value="north-east">North-East</option>
                  <option value="north-west">North-West</option>
                  <option value="south-east">South-East</option>
                  <option value="south-west">South-West</option>
                </select>
              </div>

              {/* Property Age */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Property Age</label>
                <select className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.propertyAge} onChange={(e) => setPreferences(prev => ({ ...prev, propertyAge: e.target.value }))}>
                  <option value="">Select Age</option>
                  <option value="under-construction">Under Construction</option>
                  <option value="0-1">0-1 Years</option>
                  <option value="1-5">1-5 Years</option>
                  <option value="5-10">5-10 Years</option>
                  <option value="10+">10+ Years</option>
                </select>
              </div>

              {/* Possession Status */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Possession Status</label>
                <select className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.possessionStatus} onChange={(e) => setPreferences(prev => ({ ...prev, possessionStatus: e.target.value }))}>
                  <option value="">Select Possession</option>
                  <option value="ready-to-move">Ready to Move</option>
                  <option value="under-construction">Under Construction</option>
                  <option value="new-launch">New Launch</option>
                </select>
              </div>

              {/* Gated Community */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Gated Community</label>
                <select className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.gatedCommunity} onChange={(e) => setPreferences(prev => ({ ...prev, gatedCommunity: e.target.value }))}>
                  <option value="">Any</option>
                  <option value="yes">Yes</option>
                  <option value="no">No</option>
                </select>
              </div>
            </div>

            {/* Amenities */}
            <div className="mt-4">
              <label className="block text-sm font-medium text-gray-700 mb-2">Preferred Amenities</label>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-2">
                {['Swimming Pool', 'Gym', 'Parking', 'Security', 'Garden', 'Elevator', 'Power Backup', 'Water Supply', 'Club House', 'Children Play Area', 'Jogging Track', 'CCTV'].map(amenity => (
                  <label key={amenity} className="flex items-center text-sm">
                    <input 
                      type="checkbox" 
                      className="mr-2" 
                      checked={preferences.amenities.includes(amenity)}
                      onChange={(e) => { 
                        if (e.target.checked) { 
                          setPreferences(prev => ({ ...prev, amenities: [...prev.amenities, amenity] })); 
                        } else { 
                          setPreferences(prev => ({ ...prev, amenities: prev.amenities.filter(a => a !== amenity) })); 
                        } 
                      }} 
                    />
                    {amenity}
                  </label>
                ))}
              </div>
            </div>

            {/* Features */}
            <div className="mt-4">
              <label className="block text-sm font-medium text-gray-700 mb-2">Additional Features</label>
              <div className="grid grid-cols-2 md:grid-cols-3 gap-2">
                {['Modular Kitchen', 'Servant Room', 'Study Room', 'Pooja Room', 'Terrace Garden', 'Intercom', 'Piped Gas', 'Maintenance Staff', 'Visitor Parking'].map(feature => (
                  <label key={feature} className="flex items-center text-sm">
                    <input 
                      type="checkbox" 
                      className="mr-2" 
                      checked={preferences.features.includes(feature)}
                      onChange={(e) => { 
                        if (e.target.checked) { 
                          setPreferences(prev => ({ ...prev, features: [...prev.features, feature] })); 
                        } else { 
                          setPreferences(prev => ({ ...prev, features: prev.features.filter(f => f !== feature) })); 
                        } 
                      }} 
                    />
                    {feature}
                  </label>
                ))}
              </div>
            </div>

            {/* Property Highlights */}
            <div className="mt-4">
              <label className="block text-sm font-medium text-gray-700 mb-2">Property Highlights</label>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-2">
                {['Prime Location', 'Vastu Compliant', 'Corner Property', 'Park Facing', 'Gated Community', 'Near Metro', 'Investment Property', 'Ready to Move'].map(highlight => (
                  <label key={highlight} className="flex items-center text-sm">
                    <input 
                      type="checkbox" 
                      className="mr-2" 
                      checked={preferences.highlights.includes(highlight)}
                      onChange={(e) => { 
                        if (e.target.checked) { 
                          setPreferences(prev => ({ ...prev, highlights: [...prev.highlights, highlight] })); 
                        } else { 
                          setPreferences(prev => ({ ...prev, highlights: prev.highlights.filter(h => h !== highlight) })); 
                        } 
                      }} 
                    />
                    {highlight}
                  </label>
                ))}
              </div>
            </div>

            {/* Timeline */}
            <div className="mt-4">
              <label className="block text-sm font-medium text-gray-700 mb-2">Timeline</label>
              <select className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value={preferences.timeline} onChange={(e) => setPreferences(prev => ({ ...prev, timeline: e.target.value }))}>
                <option value="">Select Timeline</option>
                <option value="immediate">Immediate</option>
                <option value="1-3months">1-3 Months</option>
                <option value="3-6months">3-6 Months</option>
                <option value="6+months">6+ Months</option>
              </select>
            </div>

            <div className="flex gap-3 mt-6">
              <button
                onClick={handleSavePreferences}
                disabled={preferencesLoading}
                className="px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors disabled:opacity-50"
              >
                {preferencesLoading ? 'Saving...' : (hasPreferences ? 'Update Preferences' : 'Save Preferences')}
              </button>
              <button
                onClick={() => setShowPreferencesModal(false)}
                className="px-4 py-2 bg-white text-black border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}

    </div>
  );
}