import { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { Card } from "../ui/card";
import { MapPin, ArrowLeft, QrCode } from "lucide-react";
import { Button } from "../ui/button";
import { QRScanner } from "../QRScanner";
import { toast } from "sonner";
import QRCodeLib from "qrcode";

interface BadgeData {
  id: string;
  propertyName: string;
  propertyAddress: string;
  badgeType: string;
  badgeLevel: string;
  timestamp: string;
  earnedPoints: number;
  verificationMethod: string[];
}

export const EarnBadge = () => {
  const navigate = useNavigate();
  const [properties, setProperties] = useState<any[]>([]);
  const [selectedProperty, setSelectedProperty] = useState<any>(null);
  const [loadingProperties, setLoadingProperties] = useState(false);
  const [showQRScanner, setShowQRScanner] = useState(false);
  const [propertyQRCode, setPropertyQRCode] = useState<string>('');
  const [verificationStatus, setVerificationStatus] = useState<'idle' | 'verifying' | 'success' | 'error'>('idle');

  useEffect(() => {
    fetchAvailableProperties();
  }, []);

  const fetchAvailableProperties = async () => {
    setLoadingProperties(true);
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userEmail = user.email;
      
      const response = await fetch('/api/properties/');
      if (response.ok) {
        const result = await response.json();
        const allProperties = result.properties || [];
        const filteredProperties = allProperties.filter((p: any) => p.email !== userEmail);
        setProperties(filteredProperties);
      }
    } catch (error) {
      toast.error('Failed to load properties');
    } finally {
      setLoadingProperties(false);
    }
  };

  const generatePropertyQRCode = async (property: any) => {
    const qrData = JSON.stringify({
      propertyId: property.id,
      title: property.title,
      address: property.address,
      city: property.city,
      latitude: property.latitude,
      longitude: property.longitude
    });
    const qrCodeUrl = await QRCodeLib.toDataURL(qrData, { width: 200 });
    setPropertyQRCode(qrCodeUrl);
  };

  const handlePropertySelect = async (property: any) => {
    setSelectedProperty(property);
    await generatePropertyQRCode(property);
    setShowQRScanner(true);
  };

  const confirmQRScan = () => {
    setVerificationStatus('verifying');
    toast.success('Verifying visit...');
    
    setTimeout(() => {
      const newBadge: BadgeData = {
        id: `badge-${Date.now()}`,
        propertyName: selectedProperty.title,
        propertyAddress: `${selectedProperty.address}, ${selectedProperty.city}`,
        badgeType: 'Verified Property Visit',
        badgeLevel: 'Level 1',
        timestamp: new Date().toISOString(),
        earnedPoints: 50,
        verificationMethod: ['QR Code'],
      };
      
      const existingBadges = JSON.parse(localStorage.getItem('visitBadges') || '[]');
      localStorage.setItem('visitBadges', JSON.stringify([...existingBadges, newBadge]));
      
      setVerificationStatus('success');
      toast.success('Badge earned successfully!');
      
      setTimeout(() => {
        navigate('/user/dashboard?section=MyBadges');
      }, 1500);
    }, 1000);
  };

  if (showQRScanner && selectedProperty) {
    return (
      <div className="space-y-6">
        <Button variant="ghost" onClick={() => setShowQRScanner(false)} className="mb-4">
          <ArrowLeft className="w-4 h-4 mr-2" />
          Back to Properties
        </Button>
        
        <Card className="p-6">
          <h2 className="text-xl font-bold mb-4">Scan QR Code at Property</h2>
          <div className="mb-4">
            <p className="text-gray-600 mb-2">Property: {selectedProperty.title}</p>
            <p className="text-gray-500 text-sm">{selectedProperty.address}, {selectedProperty.city}</p>
          </div>
          
          <div className="bg-gray-50 p-4 rounded-lg mb-4">
            <QRScanner
              onScanSuccess={(data) => {
                try {
                  const scannedData = JSON.parse(data);
                  if (scannedData.propertyId === selectedProperty.id) {
                    confirmQRScan();
                  } else {
                    toast.error('QR code does not match selected property');
                  }
                } catch {
                  toast.error('Invalid QR code');
                }
              }}
              onScanError={(error) => toast.error(error)}
            />
          </div>
          
          <div className="text-center">
            <p className="text-sm text-gray-500 mb-2">Or scan this QR code for testing:</p>
            {propertyQRCode && (
              <img src={propertyQRCode} alt="Property QR Code" className="mx-auto" />
            )}
          </div>
        </Card>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Earn Badge</h1>
          <p className="text-gray-600 mt-1">Select a property to visit and earn your badge</p>
        </div>
        <Button variant="outline" onClick={() => navigate('/user/dashboard?section=MyBadges')}>
          <ArrowLeft className="w-4 h-4 mr-2" />
          Back to My Badges
        </Button>
      </div>

      {loadingProperties ? (
        <div className="text-center py-12">
          <p className="text-gray-500">Loading properties...</p>
        </div>
      ) : properties.length === 0 ? (
        <Card className="p-12 text-center">
          <MapPin className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-semibold text-gray-900 mb-2">No Properties Available</h3>
          <p className="text-gray-600">There are no properties available to visit at the moment.</p>
        </Card>
      ) : (
        <div className="grid gap-4">
          {properties.map((property) => (
            <Card key={property.id} className="p-6 hover:shadow-lg transition-shadow cursor-pointer" onClick={() => handlePropertySelect(property)}>
              <div className="flex items-start justify-between">
                <div className="flex-1">
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">{property.title}</h3>
                  <div className="flex items-start text-gray-600 mb-2">
                    <MapPin className="w-4 h-4 mr-2 mt-1 flex-shrink-0" />
                    <span className="text-sm">{property.address}, {property.city}</span>
                  </div>
                  {property.price && (
                    <p className="text-lg font-semibold text-blue-600">₹{property.price.toLocaleString()}</p>
                  )}
                </div>
                <Button>
                  <QrCode className="w-4 h-4 mr-2" />
                  Check In
                </Button>
              </div>
            </Card>
          ))}
        </div>
      )}
    </div>
  );
};
