import React, { useState, useEffect } from 'react';
import { CheckCircle, Clock, AlertCircle, FileText, ChevronRight } from 'lucide-react';

interface EscrowMilestone {
  id: string;
  property_id: string;
  payment_id?: string;
  milestone_name: string;
  milestone_number: number;
  total_milestones: number;
  amount: number;
  status: 'pending' | 'paid' | 'in_escrow' | 'verification_pending' | 'verified' | 'released' | 'refunded' | 'disputed';
  verification_required: boolean;
  verified_at?: string;
  released_at?: string;
  payment_due_date?: string;
  created_at: string;
}

interface EscrowTrackerProps {
  propertyId: string;
}

export function EscrowTracker({ propertyId }: EscrowTrackerProps) {
  const [milestones, setMilestones] = useState<EscrowMilestone[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedMilestone, setSelectedMilestone] = useState<EscrowMilestone | null>(null);

  useEffect(() => {
    fetchMilestones();
  }, [propertyId]);

  const fetchMilestones = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await fetch(`/api/payments/escrow/${propertyId}`, {
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });

      if (response.ok) {
        const data = await response.json();
        setMilestones(data);
      }
    } catch (error) {
      console.error('Failed to fetch escrow milestones:', error);
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      pending: 'bg-gray-100 text-gray-700 border-gray-300',
      paid: 'bg-blue-100 text-blue-700 border-blue-300',
      in_escrow: 'bg-purple-100 text-purple-700 border-purple-300',
      verification_pending: 'bg-yellow-100 text-yellow-700 border-yellow-300',
      verified: 'bg-green-100 text-green-700 border-green-300',
      released: 'bg-green-100 text-green-700 border-green-300',
      refunded: 'bg-red-100 text-red-700 border-red-300',
      disputed: 'bg-red-100 text-red-700 border-red-300'
    };
    return colors[status] || 'bg-gray-100 text-gray-700 border-gray-300';
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'pending':
        return <Clock className="w-5 h-5" />;
      case 'paid':
      case 'in_escrow':
      case 'verification_pending':
        return <AlertCircle className="w-5 h-5" />;
      case 'verified':
      case 'released':
        return <CheckCircle className="w-5 h-5" />;
      default:
        return <Clock className="w-5 h-5" />;
    }
  };

  const getStatusLabel = (status: string) => {
    const labels: Record<string, string> = {
      pending: 'Payment Pending',
      paid: 'Paid',
      in_escrow: 'In Escrow',
      verification_pending: 'Awaiting Verification',
      verified: 'Verified',
      released: 'Released',
      refunded: 'Refunded',
      disputed: 'Disputed'
    };
    return labels[status] || status;
  };

  const calculateProgress = () => {
    const completed = milestones.filter(m =>
      m.status === 'released' || m.status === 'verified'
    ).length;
    return (completed / milestones.length) * 100;
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center p-12">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (milestones.length === 0) {
    return (
      <div className="text-center py-12 bg-gray-50 rounded-lg border-2 border-dashed border-gray-300">
        <FileText className="w-12 h-12 text-gray-400 mx-auto mb-3" />
        <h3 className="font-semibold text-gray-900 mb-2">No Escrow Milestones</h3>
        <p className="text-gray-600">This property doesn't have escrow milestones configured.</p>
      </div>
    );
  }

  const totalAmount = milestones.reduce((sum, m) => sum + m.amount, 0);
  const paidAmount = milestones
    .filter(m => m.status !== 'pending')
    .reduce((sum, m) => sum + m.amount, 0);

  return (
    <div className="space-y-6">
      {/* Header */}
      <div>
        <h2 className="text-2xl font-bold text-gray-900">Escrow Milestones</h2>
        <p className="text-gray-600 mt-1">Track your property purchase payment milestones</p>
      </div>

      {/* Progress Overview */}
      <div className="bg-gradient-to-br from-blue-50 to-purple-50 border border-blue-200 rounded-lg p-6">
        <div className="flex items-center justify-between mb-4">
          <div>
            <h3 className="text-lg font-semibold text-gray-900">Overall Progress</h3>
            <p className="text-sm text-gray-600">
              {milestones.filter(m => m.status === 'released' || m.status === 'verified').length} of {milestones.length} milestones completed
            </p>
          </div>
          <div className="text-right">
            <p className="text-2xl font-bold text-blue-600">
              ₹{paidAmount.toLocaleString('en-IN')}
            </p>
            <p className="text-sm text-gray-600">
              of ₹{totalAmount.toLocaleString('en-IN')}
            </p>
          </div>
        </div>

        {/* Progress Bar */}
        <div className="w-full bg-gray-200 rounded-full h-3">
          <div
            className="bg-gradient-to-r from-blue-600 to-purple-600 h-3 rounded-full transition-all duration-500"
            style={{ width: `${calculateProgress()}%` }}
          ></div>
        </div>
        <p className="text-sm text-gray-600 mt-2 text-right">
          {calculateProgress().toFixed(0)}% Complete
        </p>
      </div>

      {/* Milestones Timeline */}
      <div className="space-y-4">
        {milestones.map((milestone, index) => (
          <div
            key={milestone.id}
            className="bg-white border-2 border-gray-200 rounded-lg p-5 hover:shadow-md transition-all cursor-pointer"
            onClick={() => setSelectedMilestone(milestone)}
          >
            <div className="flex items-start gap-4">
              {/* Milestone Number */}
              <div className="flex-shrink-0">
                <div className={`w-12 h-12 rounded-full flex items-center justify-center font-bold text-lg border-2 ${milestone.status === 'released' || milestone.status === 'verified'
                    ? 'bg-green-100 text-green-700 border-green-500'
                    : milestone.status === 'pending'
                      ? 'bg-gray-100 text-gray-500 border-gray-300'
                      : 'bg-blue-100 text-blue-700 border-blue-500'
                  }`}>
                  {milestone.milestone_number}
                </div>
              </div>

              {/* Milestone Details */}
              <div className="flex-1">
                <div className="flex items-start justify-between mb-2">
                  <div>
                    <h3 className="font-semibold text-gray-900 text-lg">
                      {milestone.milestone_name}
                    </h3>
                    <p className="text-sm text-gray-600">
                      Milestone {milestone.milestone_number} of {milestone.total_milestones}
                    </p>
                  </div>
                  <div className="text-right">
                    <p className="text-xl font-bold text-gray-900">
                      ₹{milestone.amount.toLocaleString('en-IN')}
                    </p>
                  </div>
                </div>

                {/* Status Badge */}
                <div className="flex items-center gap-3 mb-3">
                  <span className={`px-3 py-1 rounded-full text-sm font-medium border flex items-center gap-2 ${getStatusColor(milestone.status)}`}>
                    {getStatusIcon(milestone.status)}
                    {getStatusLabel(milestone.status)}
                  </span>
                  {milestone.payment_due_date && milestone.status === 'pending' && (
                    <span className="text-sm text-gray-600">
                      Due: {new Date(milestone.payment_due_date).toLocaleDateString('en-IN')}
                    </span>
                  )}
                </div>

                {/* Verification Status */}
                {milestone.verification_required && milestone.status === 'verification_pending' && (
                  <div className="bg-yellow-50 border border-yellow-200 rounded p-3 text-sm">
                    <p className="text-yellow-800 font-medium">
                      ⏳ Awaiting verification from admin
                    </p>
                  </div>
                )}

                {milestone.verified_at && (
                  <div className="bg-green-50 border border-green-200 rounded p-3 text-sm">
                    <p className="text-green-800">
                      ✓ Verified on {new Date(milestone.verified_at).toLocaleDateString('en-IN')}
                    </p>
                  </div>
                )}

                {milestone.released_at && (
                  <div className="bg-green-50 border border-green-200 rounded p-3 text-sm">
                    <p className="text-green-800">
                      ✓ Released on {new Date(milestone.released_at).toLocaleDateString('en-IN')}
                    </p>
                  </div>
                )}
              </div>

              <ChevronRight className="w-5 h-5 text-gray-400" />
            </div>

            {/* Connector Line */}
            {index < milestones.length - 1 && (
              <div className="ml-6 mt-2 mb-2">
                <div className={`w-0.5 h-8 ${milestone.status === 'released' || milestone.status === 'verified'
                    ? 'bg-green-500'
                    : 'bg-gray-300'
                  }`}></div>
              </div>
            )}
          </div>
        ))}
      </div>

      {/* Milestone Detail Modal */}
      {selectedMilestone && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg max-w-2xl w-full p-6 max-h-[90vh] overflow-y-auto">
            <div className="flex items-start justify-between mb-6">
              <div>
                <h3 className="text-2xl font-bold text-gray-900">
                  {selectedMilestone.milestone_name}
                </h3>
                <p className="text-gray-600 mt-1">
                  Milestone {selectedMilestone.milestone_number} of {selectedMilestone.total_milestones}
                </p>
              </div>
              <button
                onClick={() => setSelectedMilestone(null)}
                className="text-gray-400 hover:text-gray-600"
              >
                ✕
              </button>
            </div>

            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-gray-600">Amount</p>
                  <p className="text-xl font-bold text-gray-900">
                    ₹{selectedMilestone.amount.toLocaleString('en-IN')}
                  </p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Status</p>
                  <span className={`inline-flex px-3 py-1 rounded-full text-sm font-medium border ${getStatusColor(selectedMilestone.status)}`}>
                    {getStatusLabel(selectedMilestone.status)}
                  </span>
                </div>
              </div>

              {selectedMilestone.payment_due_date && (
                <div>
                  <p className="text-sm text-gray-600">Payment Due Date</p>
                  <p className="font-medium text-gray-900">
                    {new Date(selectedMilestone.payment_due_date).toLocaleDateString('en-IN', {
                      year: 'numeric',
                      month: 'long',
                      day: 'numeric'
                    })}
                  </p>
                </div>
              )}

              {selectedMilestone.verification_required && (
                <div className="bg-blue-50 border border-blue-200 rounded p-4">
                  <p className="text-sm font-medium text-blue-900">Verification Required</p>
                  <p className="text-sm text-blue-700 mt-1">
                    This milestone requires admin verification before funds can be released.
                  </p>
                </div>
              )}

              {selectedMilestone.status === 'pending' && (
                <button
                  className="w-full px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors font-semibold"
                  onClick={() => {
                    // Navigate to payment
                    setSelectedMilestone(null);
                    window.location.hash = '#payments';
                  }}
                >
                  Pay Now
                </button>
              )}
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
