import React, { useEffect, useRef, useState } from 'react';
import L from 'leaflet';
import 'leaflet/dist/leaflet.css';

interface Facility {
  name?: string;
  category?: string;
  latitude?: number;
  longitude?: number;
  distance_km?: number;
}

interface FacilityGroup {
  category: string;
  count: number;
  items: Facility[];
}

interface MapTileConfig {
  primary: string;
  fallback_1: string;
  fallback_2: string;
  attribution: string;
}

interface FacilitiesMapProps {
  centerLat: number;
  centerLon: number;
  facilities: FacilityGroup[];
  radiusKm: number;
  tileConfig?: MapTileConfig;
}

const categoryColors: Record<string, string> = {
  schools: '#3B82F6',
  healthcare: '#EF4444',
  grocery: '#10B981',
  transit: '#F59E0B',
  parks: '#22C55E',
  banks: '#8B5CF6',
  restaurants: '#EC4899',
  police: '#1E40AF',
  fuel: '#F97316',
  pharmacies: '#DC2626',
  atms: '#7C3AED',
  malls: '#DB2777',
  fitness: '#059669',
  cinema: '#BE185D',
  library: '#0891B2',
  worship: '#CA8A04',
};

const categoryIcons: Record<string, string> = {
  schools: '🏫',
  healthcare: '🏥',
  grocery: '🛒',
  transit: '🚇',
  parks: '🌳',
  banks: '🏦',
  restaurants: '🍽️',
  police: '👮',
  fuel: '⛽',
  pharmacies: '💊',
  atms: '🏧',
  malls: '🛍️',
  fitness: '💪',
  cinema: '🎬',
  library: '📚',
  worship: '🕌',
};

export function FacilitiesMap({ centerLat, centerLon, facilities, radiusKm, tileConfig }: FacilitiesMapProps) {
  const mapRef = useRef<HTMLDivElement>(null);
  const mapInstanceRef = useRef<L.Map | null>(null);
  const tileLayerRef = useRef<L.TileLayer | null>(null);
  const markersRef = useRef<L.Marker[]>([]);
  const [selectedCategory, setSelectedCategory] = useState<string | null>(null);
  const [mapError, setMapError] = useState<string | null>(null);

  // Default tile configuration
  const defaultTileConfig: MapTileConfig = {
    primary: 'https://{s}.basemaps.cartocdn.com/rastertiles/voyager/{z}/{x}/{y}.png',
    fallback_1: 'https://{s}.tile.openstreetmap.fr/hot/{z}/{x}/{y}.png',
    fallback_2: 'https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png',
    attribution: '© OpenStreetMap contributors, © CARTO'
  };

  const tiles = tileConfig || defaultTileConfig;

  useEffect(() => {
    if (!mapRef.current || mapInstanceRef.current) return;

    // Validate coordinates
    if (isNaN(centerLat) || isNaN(centerLon)) {
      console.error('Invalid coordinates:', { centerLat, centerLon });
      setMapError('Invalid map coordinates');
      return;
    }

    try {
      // Initialize map
      const map = L.map(mapRef.current).setView([centerLat, centerLon], 13);
      mapInstanceRef.current = map;

      // Try primary tile server
      const primaryLayer = L.tileLayer(tiles.primary, {
        attribution: tiles.attribution,
        maxZoom: 19,
      });

      primaryLayer.on('tileerror', () => {
        console.warn('Primary tile server failed, switching to fallback 1');
        if (tileLayerRef.current) {
          map.removeLayer(tileLayerRef.current);
        }
        
        const fallback1Layer = L.tileLayer(tiles.fallback_1, {
          attribution: tiles.attribution,
          maxZoom: 19,
        });

        fallback1Layer.on('tileerror', () => {
          console.warn('Fallback 1 failed, switching to fallback 2');
          if (tileLayerRef.current) {
            map.removeLayer(tileLayerRef.current);
          }
          
          const fallback2Layer = L.tileLayer(tiles.fallback_2, {
            attribution: tiles.attribution,
            maxZoom: 19,
          });

          fallback2Layer.on('tileerror', () => {
            console.error('All tile servers failed');
            setMapError('Map tiles failed to load. Please refresh the page.');
          });

          fallback2Layer.addTo(map);
          tileLayerRef.current = fallback2Layer;
        });

        fallback1Layer.addTo(map);
        tileLayerRef.current = fallback1Layer;
      });

      primaryLayer.addTo(map);
      tileLayerRef.current = primaryLayer;

      // Add center marker (search location)
      const centerIcon = L.divIcon({
        html: '<div style="background: #0056D2; width: 20px; height: 20px; border-radius: 50%; border: 3px solid white; box-shadow: 0 2px 4px rgba(0,0,0,0.3);"></div>',
        className: '',
        iconSize: [20, 20],
        iconAnchor: [10, 10],
      });
      L.marker([centerLat, centerLon], { icon: centerIcon })
        .addTo(map)
        .bindPopup('<b>Search Location</b>');

      // Add radius circle
      L.circle([centerLat, centerLon], {
        radius: radiusKm * 1000,
        color: '#0056D2',
        fillColor: '#0056D2',
        fillOpacity: 0.1,
        weight: 2,
      }).addTo(map);

      setMapError(null);
    } catch (error) {
      console.error('Map initialization error:', error);
      setMapError('Failed to initialize map');
    }

    return () => {
      if (mapInstanceRef.current) {
        mapInstanceRef.current.remove();
        mapInstanceRef.current = null;
      }
      tileLayerRef.current = null;
    };
  }, [centerLat, centerLon, radiusKm, tiles]);

  useEffect(() => {
    if (!mapInstanceRef.current) return;

    const map = mapInstanceRef.current;

    // Clear existing markers
    markersRef.current.forEach(marker => marker.remove());
    markersRef.current = [];

    // Filter facilities based on selected category
    const filteredFacilities = selectedCategory
      ? facilities.filter(group => group.category === selectedCategory)
      : facilities;

    // Add facility markers
    let totalFacilities = 0;
    let facilitiesWithCoords = 0;
    
    filteredFacilities.forEach((group) => {
      const color = categoryColors[group.category] || '#6B7280';
      const emoji = categoryIcons[group.category] || '📍';

      group.items.forEach((facility) => {
        totalFacilities++;
        
        if (facility.latitude && facility.longitude && 
            !isNaN(facility.latitude) && !isNaN(facility.longitude)) {
          facilitiesWithCoords++;
          
          const icon = L.divIcon({
            html: `<div style="font-size: 24px; text-shadow: 0 2px 4px rgba(0,0,0,0.3);">${emoji}</div>`,
            className: '',
            iconSize: [24, 24],
            iconAnchor: [12, 12],
          });

          const marker = L.marker([facility.latitude, facility.longitude], { icon })
            .addTo(map)
            .bindPopup(`
              <div style="min-width: 150px;">
                <b>${facility.name}</b><br/>
                <span style="color: ${color}; font-weight: 600;">${group.category}</span><br/>
                <span style="color: #6B7280; font-size: 12px;">${facility.distance_km?.toFixed(2)} km away</span>
              </div>
            `);
          
          markersRef.current.push(marker);
        }
      });
    });
    
    console.log(`Map: ${facilitiesWithCoords} of ${totalFacilities} facilities have valid coordinates`);
  }, [facilities, selectedCategory]);

  return (
    <div className="space-y-4">
      {mapError && (
        <div className="bg-yellow-50 border border-yellow-200 text-yellow-800 px-4 py-3 rounded-lg">
          <p className="text-sm font-medium">{mapError}</p>
        </div>
      )}

      {/* Category Filter Buttons */}
      <div className="flex flex-wrap gap-2">
        <button
          onClick={() => setSelectedCategory(null)}
          className={`flex items-center gap-2 px-4 py-2 rounded-lg font-medium text-sm transition-colors ${
            selectedCategory === null
              ? 'bg-[#0056D2] text-white shadow-md hover:bg-[#004bb5]'
              : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
          }`}
        >
          <span>All Facilities</span>
        </button>
        {facilities.map((group) => (
          <button
            key={group.category}
            onClick={() => setSelectedCategory(group.category)}
            className={`flex items-center gap-2 px-4 py-2 rounded-lg font-medium text-sm transition-colors ${
              selectedCategory === group.category
                ? 'bg-[#0056D2] text-white shadow-md hover:bg-[#004bb5]'
                : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
            }`}
          >
            <span style={{ fontSize: '18px' }}>{categoryIcons[group.category] || '📍'}</span>
            <span className="capitalize">{group.category}</span>
            <span className="text-xs opacity-90">({group.count})</span>
          </button>
        ))}
      </div>

      {/* Map */}
      <div ref={mapRef} style={{ height: '500px', width: '100%', borderRadius: '12px', overflow: 'hidden' }} />
    </div>
  );
}
