import React, { useState, useEffect } from "react";
import {
  getSentInquiries,
  getReceivedInquiries,
  Inquiry,
} from "../../api/inquiries";
import {
  getUserBookings,
  getReceivedBookings,
  VisitBooking,
} from "../../api/visitBookings";
import {
  MessageSquare,
  Calendar,
  DollarSign,
  Phone,
  Mail,
  User,
  Building,
  Clock,
  CheckCircle,
  XCircle,
  X,
  MapPin,
} from "lucide-react";

export const InquiryManagement: React.FC = () => {
  const [activeTab, setActiveTab] = useState<"sent" | "received">("sent");
  const [inquiries, setInquiries] = useState<Inquiry[]>([]);
  const [visitBookings, setVisitBookings] = useState<VisitBooking[]>([]);
  const [selectedInquiry, setSelectedInquiry] = useState<Inquiry | null>(null);
  const [selectedBooking, setSelectedBooking] = useState<VisitBooking | null>(
    null
  );
  const [ownerDetails, setOwnerDetails] = useState<any>(null);

  const [loading, setLoading] = useState(true);
  const [totalCount, setTotalCount] = useState(0);

  const user = JSON.parse(localStorage.getItem("user") || "{}");

  useEffect(() => {
    if (selectedInquiry || selectedBooking) {
      document.body.style.overflow = "hidden";
    } else {
      document.body.style.overflow = "unset";
      setOwnerDetails(null); // Clear owner details when modal closes
    }
    return () => {
      document.body.style.overflow = "unset";
    };
  }, [selectedInquiry, selectedBooking]);

  useEffect(() => {
    fetchInquiries();
    fetchVisitBookings();
  }, [activeTab]);

  const handleCall = async (phone: string, inquiryId: string) => {
    if (phone && inquiryId) {
      // Open phone dialer
      window.open(`tel:${phone}`, "_self");
      
      // Update existing lead to prospect status (NOT create new lead)
      await updateLeadToProspect(inquiryId);
      
      // Show success message
      setTimeout(() => {
        alert("Thanks for contacting.");
      }, 1000);
    }
  };

  const handleWhatsApp = async (phone: string, inquiryId: string) => {
    if (phone && inquiryId) {
      // Clean phone number (remove spaces, dashes, etc.)
      const cleanPhone = phone.replace(/[^+\d]/g, "");
      // Open WhatsApp
      window.open(`https://wa.me/${cleanPhone}`, "_blank");
      
      // Update existing lead to prospect status (NOT create new lead)
      await updateLeadToProspect(inquiryId);
      
      // Show success message
      setTimeout(() => {
        alert("Thanks for contacting via Whatsapp.");
      }, 1000);
    }
  };

  const updateInquiryStatus = async (inquiryId: string, status: string) => {
    try {
      // Find the inquiry to check current status (idempotency check)
      const currentInquiry = inquiries.find(
        (inq) => inq.inquiry_id === inquiryId
      );
      if (currentInquiry?.status === status) {
        console.log(
          `Inquiry ${inquiryId} already has status ${status}, skipping update`
        );
        return;
      }

      const API_BASE =
        import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
      const token = localStorage.getItem("access_token");

      // Update inquiry status
      const response = await fetch(
        `${API_BASE}/inquiries/${inquiryId}/status`,
        {
          method: "PUT",
          headers: {
            "Content-Type": "application/json",
            ...(token && { Authorization: `Bearer ${token}` }),
          },
          body: JSON.stringify({
            status,
            prospect_at: new Date().toISOString(),
          }),
        }
      );

      if (response.ok) {
        // Update local state
        setInquiries((prevInquiries) =>
          prevInquiries.map((inquiry) =>
            inquiry.inquiry_id === inquiryId ? { ...inquiry, status } : inquiry
          )
        );
      }
    } catch (error) {
      console.error("Error updating inquiry status:", error);
    }
  };

  const updateLeadToProspect = async (inquiryId: string) => {
    try {
      const API_BASE =
        import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
      const token = localStorage.getItem("access_token");

      // Update existing lead status to prospect using inquiry_id
      const response = await fetch(
        `${API_BASE}/leads-management/leads/by-inquiry/${inquiryId}/prospect`,
        {
          method: "PUT",
          headers: {
            "Content-Type": "application/json",
            ...(token && { Authorization: `Bearer ${token}` }),
          },
        }
      );

      if (response.ok) {
        console.log("Lead updated to prospect successfully");
        // Update inquiry status to prospect
        await updateInquiryStatus(inquiryId, "prospect");
      } else {
        const errorData = await response.json();
        console.error("Failed to update lead to prospect:", errorData);
      }
    } catch (error) {
      console.error("Error updating lead to prospect:", error);
    }
  };

  const fetchOwnerDetails = async (ownerId: string) => {
    try {
      const token = localStorage.getItem("access_token");
      const API_BASE =
        import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
      const response = await fetch(`${API_BASE}/profile/${ownerId}`, {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "application/json",
        },
      });

      if (response.ok) {
        const ownerProfile = await response.json();
        return {
          name:
            `${ownerProfile.first_name || ""} ${
              ownerProfile.last_name || ""
            }`.trim() || "Property Owner",
          email: ownerProfile.email || "",
          phone: ownerProfile.phone_number || "",
        };
      }
    } catch (error) {
      console.error("Error fetching owner profile:", error);
    }
    return null;
  };

  const fetchInquiries = async () => {
    try {
      setLoading(true);
      let response;

      console.log(
        "Fetching inquiries for user:",
        user.user_id,
        "Tab:",
        activeTab
      );

      if (activeTab === "sent") {
        response = await getSentInquiries(user.user_id, 1, 20);
      } else {
        response = await getReceivedInquiries(user.user_id, 1, 20);
      }

      console.log("🔍 API Response:", response);
      console.log("🔍 Raw inquiries from API:", response.inquiries);

      // Filter inquiries to ensure they belong to current user
      const filteredInquiries = response.inquiries.filter((inquiry) => {
        if (activeTab === "sent") {
          return inquiry.buyer_id === user.user_id;
        } else {
          return inquiry.owner_id === user.user_id;
        }
      });

      // For sent inquiries, fetch owner details from profile API
      if (activeTab === "sent") {
        const inquiriesWithOwnerDetails = await Promise.all(
          filteredInquiries.map(async (inquiry) => {
            const ownerDetails = await fetchOwnerDetails(inquiry.owner_id);
            return {
              ...inquiry,
              owner_name: ownerDetails?.name || inquiry.owner_name,
              owner_email: ownerDetails?.email || inquiry.owner_email,
              owner_phone: ownerDetails?.phone || inquiry.owner_phone,
            };
          })
        );
        setInquiries(inquiriesWithOwnerDetails);
      } else {
        setInquiries(filteredInquiries);
      }

      setTotalCount(filteredInquiries.length);
    } catch (error) {
      console.error("Error fetching inquiries:", error);
      setInquiries([]);
      setTotalCount(0);
    } finally {
      setLoading(false);
    }
  };

  const fetchVisitBookings = async () => {
    try {
      console.log("🔍 User fetching visit bookings for user:", user.user_id);

      if (activeTab === "sent") {
        // Get bookings made BY this user
        const sentBookings = await getUserBookings(user.user_id);
        console.log("📊 Sent bookings:", sentBookings);
        setVisitBookings(sentBookings);
      } else {
        // Get bookings made FOR properties owned BY this user
        const receivedBookings = await getReceivedBookings(user.user_id);
        console.log("📊 Received bookings:", receivedBookings);
        setVisitBookings(receivedBookings);
      }
    } catch (error) {
      console.error("❌ User error fetching visit bookings:", error);
      // Fallback to localStorage for testing
      const localBookings = JSON.parse(
        localStorage.getItem("visit_bookings") || "[]"
      );
      if (activeTab === "sent") {
        setVisitBookings(
          localBookings.filter((b: any) => b.user_id === user.user_id)
        );
      } else {
        setVisitBookings(
          localBookings.filter((b: any) => b.property_owner_id === user.user_id)
        );
      }
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
      hour: "2-digit",
      minute: "2-digit",
    });
  };

  const getStatusColor = (status: string) => {
    switch (status.toLowerCase()) {
      case "pending":
        return "text-yellow-700 bg-yellow-100";
      case "responded":
        return "text-blue-700 bg-blue-100";
      case "closed":
        return "text-gray-700 bg-gray-100";
      default:
        return "text-gray-700 bg-gray-100";
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status.toLowerCase()) {
      case "responded":
        return <CheckCircle className="w-4 h-4" />;
      case "closed":
        return <XCircle className="w-4 h-4" />;
      default:
        return <Clock className="w-4 h-4" />;
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-semibold text-gray-900">Inquiry Management</h3>
        <div className="bg-white px-4 py-2 rounded-lg border border-gray-200 shadow-sm">
          <span className="text-sm font-medium text-gray-700">Total: </span>
          <span className="text-lg font-bold text-blue-600">{totalCount}</span>
        </div>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-xl border border-gray-200 p-1 shadow-sm">
        <div className="grid w-full grid-cols-2 gap-1">
          {[
            { key: "sent", label: "Sent Inquiries", icon: MessageSquare },
            { key: "received", label: "Received Inquiries", icon: Building },
          ].map((tab) => {
            const isActive = activeTab === tab.key;
            const IconComponent = tab.icon;

            return (
              <button
                key={tab.key}
                onClick={() => setActiveTab(tab.key as any)}
                className={`flex items-center justify-center gap-3 px-6 py-4 rounded-lg text-sm font-semibold transition-all duration-200 ${
                  isActive
                    ? "bg-blue-500 text-white shadow-lg"
                    : "text-gray-600 bg-transparent hover:bg-gray-50"
                }`}
              >
                <IconComponent className="w-5 h-5" />
                <span>{tab.label}</span>
              </button>
            );
          })}
        </div>
      </div>

      {/* Content */}
      {loading ? (
        <div className="flex justify-center py-12">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      ) : inquiries.length === 0 && visitBookings.length === 0 ? (
        <div className="text-center py-12">
          <MessageSquare className="mx-auto h-12 w-12 text-gray-400" />
          <h3 className="mt-2 text-sm font-medium text-gray-900">
            No inquiries found
          </h3>
          <p className="mt-1 text-sm text-gray-500">
            {activeTab === "sent"
              ? "You haven't sent any inquiries or visit bookings yet."
              : "You haven't received any inquiries or visit bookings yet."}
          </p>
        </div>
      ) : (
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50 border-b">
                <tr>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide w-1/4">
                    Property
                  </th>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide w-1/4">
                    {activeTab === "sent" ? "Owner/Contact" : "Inquirer"}
                  </th>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide w-1/4">
                    Type
                  </th>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide w-1/4">
                    Actions
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {/* Regular Inquiries */}
                {inquiries.map((inquiry) => (
                  <tr
                    key={`inquiry-${inquiry.inquiry_id}`}
                    className="hover:bg-gray-50 transition-colors"
                  >
                    <td className="px-6 py-4">
                      <div
                        className="text-sm font-semibold text-gray-900 truncate"
                        title={inquiry.property_title}
                      >
                        {inquiry.property_title}
                      </div>
                      <div className="text-xs text-gray-500">
                        ID: {inquiry.property_id}
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      {activeTab === "sent" ? (
                        <div>
                          <div
                            className="text-sm text-gray-900 truncate"
                            title={inquiry.owner_name || "Property Owner"}
                          >
                            {inquiry.owner_name || "Property Owner"}
                          </div>
                          <div
                            className="text-xs text-gray-500 truncate"
                            title={inquiry.owner_phone || "+91 98765 43210"}
                          >
                            {inquiry.owner_phone || "+91 98765 43210"}
                          </div>
                        </div>
                      ) : (
                        <div>
                          <div
                            className="text-sm text-gray-900 truncate"
                            title={inquiry.buyer_name}
                          >
                            {inquiry.buyer_name}
                          </div>
                          <div
                            className="text-xs text-gray-500 truncate"
                            title={inquiry.buyer_email}
                          >
                            {inquiry.buyer_email}
                          </div>
                        </div>
                      )}
                    </td>
                    <td className="px-6 py-4">
                      <span className="px-2 py-1 text-xs font-medium bg-blue-100 text-blue-800 rounded-full">
                        Property Inquiry
                      </span>
                    </td>
                    <td className="px-6 py-4">
                      <button
                        onClick={() => setSelectedInquiry(inquiry)}
                        className="inline-flex items-center px-3 py-1.5 text-xs font-medium text-white bg-green-500 rounded-md hover:bg-green-600 transition-all duration-200"
                      >
                        View
                      </button>
                    </td>
                  </tr>
                ))}

                {/* Visit Bookings */}
                {visitBookings.map((booking) => (
                  <tr
                    key={`booking-${booking.id}`}
                    className="hover:bg-gray-50 transition-colors"
                  >
                    <td className="px-6 py-4">
                      <div className="text-sm font-semibold text-gray-900">
                        Property ID: {booking.property_id}
                      </div>
                      <div className="text-xs text-gray-500">
                        {booking.purpose_of_visit} -{" "}
                        {new Date(booking.visit_date).toLocaleDateString()}
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="text-sm text-gray-900">
                        {booking.full_name}
                      </div>
                      <div className="text-xs text-gray-500">
                        {booking.mobile_number}
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <span className="px-2 py-1 text-xs font-medium bg-purple-100 text-purple-800 rounded-full">
                        Visit Booking
                      </span>
                    </td>
                    <td className="px-6 py-4">
                      <button
                        onClick={() => setSelectedBooking(booking)}
                        className="inline-flex items-center px-3 py-1.5 text-xs font-medium text-white bg-blue-500 rounded-md hover:bg-blue-600 transition-all duration-200"
                      >
                        View
                      </button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}

      {/* Visit Booking Details Modal */}
      {selectedBooking && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center p-4 z-50">
          <div className="bg-white rounded-2xl w-full max-w-3xl max-h-[90vh] overflow-y-auto">
            <div className="px-6 py-4 border-b border-gray-100">
              <div className="flex items-center justify-between">
                <h2 className="text-xl font-bold text-gray-900">
                  Visit Booking Details
                </h2>
                <button
                  onClick={() => setSelectedBooking(null)}
                  className="p-2 hover:text-gray-700"
                >
                  <X className="w-5 h-5 text-gray-500" />
                </button>
              </div>
            </div>
            <div className="p-6 space-y-6">
              <div className="grid grid-cols-2 gap-6">
                <div>
                  <h4 className="font-semibold text-gray-900 mb-3">
                    Visit Information
                  </h4>
                  <div className="space-y-2">
                    <p>
                      <span className="font-medium">Date:</span>{" "}
                      {new Date(
                        selectedBooking.visit_date
                      ).toLocaleDateString()}
                    </p>
                    <p>
                      <span className="font-medium">Time:</span>{" "}
                      {selectedBooking.visit_start_time}
                    </p>
                    <p>
                      <span className="font-medium">Duration:</span>{" "}
                      {selectedBooking.visit_duration} minutes
                    </p>
                    <p>
                      <span className="font-medium">Purpose:</span>{" "}
                      {selectedBooking.purpose_of_visit}
                    </p>
                  </div>
                </div>
                <div>
                  <h4 className="font-semibold text-gray-900 mb-3">
                    Contact Details
                  </h4>
                  <div className="space-y-2">
                    <p>
                      <span className="font-medium">Name:</span>{" "}
                      {selectedBooking.full_name}
                    </p>
                    <p>
                      <span className="font-medium">Mobile:</span>{" "}
                      {selectedBooking.mobile_number}
                    </p>
                    <p>
                      <span className="font-medium">Email:</span>{" "}
                      {selectedBooking.email || "Not provided"}
                    </p>
                    <p>
                      <span className="font-medium">Visitors:</span>{" "}
                      {selectedBooking.number_of_visitors}
                    </p>
                  </div>
                </div>
              </div>
              {selectedBooking.custom_requests && (
                <div>
                  <h4 className="font-semibold text-gray-900 mb-2">
                    Special Requests
                  </h4>
                  <p className="text-gray-700">
                    {selectedBooking.custom_requests}
                  </p>
                </div>
              )}
            </div>
          </div>
        </div>
      )}

      {/* Inquiry Details Modal */}
      {selectedInquiry && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center p-4 z-50">
          <div
            className="bg-white rounded-2xl w-full max-w-3xl flex flex-col border border-gray-200 overflow-hidden font-inter"
            style={{
              boxShadow:
                "0 25px 50px -12px rgba(0, 0, 0, 0.25), 0 0 0 1px rgba(0, 0, 0, 0.05), 0 10px 15px -3px rgba(0, 0, 0, 0.1)",
              maxHeight: "90vh",
            }}
          >
            {/* Header */}
            <div className="px-6 py-4 border-b border-gray-100">
              <div className="flex items-center justify-between">
                <div>
                  <h2 className="text-xl font-bold text-gray-900 tracking-tight">
                    Inquiry Details
                  </h2>
                  <p className="text-gray-600 text-sm mt-1 font-medium">
                    Inquiry ID: {selectedInquiry.inquiry_id}
                  </p>
                </div>
                <button
                  onClick={() => setSelectedInquiry(null)}
                  className="p-2 hover:text-gray-700 transition-colors"
                >
                  <X className="w-5 h-5 text-gray-500" />
                </button>
              </div>
            </div>

            {/* Content */}
            <div
              className="flex-1"
              style={{
                minHeight: 0,
                overflowY: "auto",
                scrollbarWidth: "thin",
                scrollbarColor: "#cbd5e1 #f1f5f9",
              }}
            >
              <div className="p-6 space-y-6">
                <div className="bg-blue-50 p-4 rounded-lg border border-blue-100">
                  <h3 className="text-lg font-semibold text-gray-900 tracking-tight">
                    {selectedInquiry.property_title}
                  </h3>
                  <p className="text-sm text-gray-600 mt-1 font-medium">
                    {selectedInquiry.property_address}
                  </p>
                  <p className="text-lg font-bold text-green-600 mt-2">
                    ₹{selectedInquiry.property_price?.toLocaleString()}
                  </p>
                </div>

                <div className="grid grid-cols-2 gap-6">
                  <div>
                    {activeTab === "sent" ? (
                      <div className="bg-green-50 p-4 rounded-lg border border-green-200">
                        <h4 className="text-base font-semibold text-gray-900 mb-2 tracking-tight">
                          Property Owner Details
                        </h4>
                        <div className="flex items-center text-green-600 text-sm font-medium mb-3">
                          <CheckCircle className="w-4 h-4 mr-2" />
                          Contact details unlocked after inquiry
                        </div>
                        <div className="space-y-3">
                          <div>
                            <label className="block text-sm font-medium text-gray-600 mb-1">
                              Owner Name
                            </label>
                            <p className="text-sm font-semibold text-gray-900">
                              {selectedInquiry.owner_name || "Property Owner"}
                            </p>
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-600 mb-1">
                              Phone Number
                            </label>
                            <p className="text-sm font-semibold text-gray-900">
                              {selectedInquiry.owner_phone || "+91 98765 43210"}
                            </p>
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-600 mb-1">
                              Email Address
                            </label>
                            <p className="text-sm font-semibold text-gray-900 break-words">
                              {selectedInquiry.owner_email ||
                                "farhantigadi123@gmail.com"}
                            </p>
                          </div>
                        </div>
                        <div className="mt-4 flex gap-2">
                          {(selectedInquiry.owner_phone ||
                            "+91 98765 43210") && (
                            <button
                              onClick={() =>
                                handleWhatsApp(
                                  selectedInquiry.owner_phone ||
                                    "+91 98765 43210",
                                  selectedInquiry.inquiry_id
                                )
                              }
                              className="flex items-center gap-2 px-3 py-2 text-sm font-medium text-white bg-green-500 border border-green-500 rounded-lg hover:bg-green-600 transition-colors"
                            >
                              <svg
                                className="w-4 h-4"
                                viewBox="0 0 24 24"
                                fill="currentColor"
                              >
                                <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893A11.821 11.821 0 0020.885 3.787" />
                              </svg>
                              WhatsApp
                            </button>
                          )}
                          {(selectedInquiry.owner_phone ||
                            "+91 98765 43210") && (
                            <button
                              onClick={() =>
                                handleCall(
                                  selectedInquiry.owner_phone ||
                                    "+91 98765 43210",
                                  selectedInquiry.inquiry_id
                                )
                              }
                              className="flex items-center gap-2 px-3 py-2 text-sm font-medium text-blue-700 bg-blue-50 border border-blue-200 rounded-lg hover:bg-blue-100 transition-colors"
                            >
                              <Phone className="w-4 h-4" />
                              Call
                            </button>
                          )}
                        </div>
                      </div>
                    ) : (
                      <div>
                        <h4 className="text-base font-semibold text-gray-900 mb-3 tracking-tight">
                          Buyer Information
                        </h4>
                        <div className="space-y-3">
                          <div>
                            <label className="block text-sm font-medium text-gray-500 mb-1">
                              Name
                            </label>
                            <p className="text-sm font-semibold text-gray-900">
                              {selectedInquiry.buyer_name}
                            </p>
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-500 mb-1">
                              Phone
                            </label>
                            <p className="text-sm font-semibold text-gray-900">
                              {selectedInquiry.buyer_phone}
                            </p>
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-500 mb-1">
                              Email
                            </label>
                            <p className="text-sm font-semibold text-gray-900 break-words">
                              {selectedInquiry.buyer_email}
                            </p>
                          </div>
                        </div>
                      </div>
                    )}
                  </div>

                  <div>
                    <h4 className="text-base font-semibold text-gray-900 mb-3 tracking-tight">
                      Inquiry Details
                    </h4>
                    <div className="space-y-3">
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">
                          Reason to Buy
                        </label>
                        <p className="text-sm text-gray-700 leading-relaxed">
                          {selectedInquiry.reason_to_buy}
                        </p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">
                          Buying Timeline
                        </label>
                        <p className="text-sm font-semibold text-gray-900">
                          {selectedInquiry.buying_timeline}
                        </p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">
                          Interested in Loan
                        </label>
                        <p className="text-sm font-semibold text-gray-900">
                          {selectedInquiry.interested_in_loan ? "Yes" : "No"}
                        </p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">
                          Wants to Visit
                        </label>
                        <p className="text-sm font-semibold text-gray-900">
                          {selectedInquiry.interested_in_visit ? "Yes" : "No"}
                        </p>
                      </div>
                    </div>
                  </div>
                </div>

                <div className="flex justify-between items-center pt-4 border-t border-gray-200">
                  <span className="text-sm font-medium text-gray-500">
                    Created on {formatDate(selectedInquiry.created_at)}
                  </span>
                  <span
                    className={`px-3 py-1.5 text-xs font-semibold rounded-lg ${
                      selectedInquiry.status.toLowerCase() === "responded"
                        ? "bg-green-100 text-green-800"
                        : selectedInquiry.status.toLowerCase() === "closed"
                        ? "bg-red-100 text-red-800"
                        : "bg-blue-100 text-blue-800"
                    }`}
                  >
                    {selectedInquiry.status.toUpperCase()}
                  </span>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};
