import React, { useState, useEffect } from 'react';
import {
  CreditCard, Smartphone, Wallet, Building2, Banknote,
  QrCode, CheckCircle, Clock, AlertCircle, Shield, Copy,
  Check, Loader2, ChevronRight, Info
} from 'lucide-react';

interface PaymentChannel {
  id: string;
  name: string;
  icon: any;
  description: string;
  enabled: boolean;
}

interface SavedMethod {
  id: string;
  channel: string;
  display_info: any;
  is_default: boolean;
}

export function MultiChannelPayment() {
  const [selectedChannel, setSelectedChannel] = useState<string>('UPI');
  const [savedMethods, setSavedMethods] = useState<SavedMethod[]>([]);
  const [loading, setLoading] = useState(false);
  const [paymentStatus, setPaymentStatus] = useState<string>('idle');
  const [upiId, setUpiId] = useState('');
  const [qrCode, setQrCode] = useState('');
  const [copiedUPI, setCopiedUPI] = useState(false);

  const channels: PaymentChannel[] = [
    {
      id: 'UPI',
      name: 'UPI',
      icon: Smartphone,
      description: 'Pay via Google Pay, PhonePe, Paytm',
      enabled: true
    },
    {
      id: 'CARD',
      name: 'Card',
      icon: CreditCard,
      description: 'Debit/Credit Card (Visa, Master, Amex)',
      enabled: true
    },
    {
      id: 'WALLET',
      name: 'Wallet',
      icon: Wallet,
      description: 'Paytm, PhonePe, Amazon Pay',
      enabled: true
    },
    {
      id: 'NETBANKING',
      name: 'Net Banking',
      icon: Building2,
      description: 'Pay via your bank account',
      enabled: true
    },
    {
      id: 'BANK_TRANSFER',
      name: 'Bank Transfer',
      icon: Banknote,
      description: 'NEFT/IMPS/RTGS',
      enabled: true
    }
  ];

  useEffect(() => {
    fetchSavedMethods();
  }, []);

  const fetchSavedMethods = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await fetch('/api/payments/multi-channel/methods/saved', {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      const data = await response.json();
      setSavedMethods(data.methods || []);
    } catch (error) {
      console.error('Failed to fetch saved methods:', error);
    }
  };

  const initiatePayment = async () => {
    setLoading(true);
    setPaymentStatus('processing');

    try {
      const token = localStorage.getItem('token');
      const response = await fetch('/api/payments/multi-channel/create', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          amount_cents: 100000, // Example: ₹1000
          currency: 'INR',
          channel: selectedChannel,
          gateway: 'RAZORPAY',
          metadata: {
            purpose: 'property_payment'
          }
        })
      });

      const data = await response.json();

      if (selectedChannel === 'UPI') {
        // Show QR code or UPI ID
        setQrCode(data.additional_data?.qr_code || '');
        setUpiId(data.additional_data?.upi_id || 'payment@nal.upi');
        setPaymentStatus('awaiting_confirmation');
      } else if (selectedChannel === 'CARD') {
        // Redirect to hosted checkout
        window.location.href = data.additional_data?.checkout_url;
      } else {
        setPaymentStatus('success');
      }
    } catch (error) {
      console.error('Payment initiation failed:', error);
      setPaymentStatus('failed');
    } finally {
      setLoading(false);
    }
  };

  const copyUPIId = () => {
    navigator.clipboard.writeText(upiId);
    setCopiedUPI(true);
    setTimeout(() => setCopiedUPI(false), 2000);
  };

  const renderChannelSelector = () => (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
      {channels.map((channel) => {
        const Icon = channel.icon;
        const isSelected = selectedChannel === channel.id;

        return (
          <button
            key={channel.id}
            onClick={() => setSelectedChannel(channel.id)}
            disabled={!channel.enabled}
            className={`p-4 border-2 rounded-lg text-left transition-all ${isSelected
                ? 'border-blue-600 bg-blue-50'
                : 'border-gray-200 hover:border-gray-300 bg-white'
              } ${!channel.enabled ? 'opacity-50 cursor-not-allowed' : ''}`}
          >
            <div className="flex items-start gap-3">
              <div className={`p-2 rounded-lg ${isSelected ? 'bg-blue-100' : 'bg-gray-100'}`}>
                <Icon className={`w-5 h-5 ${isSelected ? 'text-blue-600' : 'text-gray-600'}`} />
              </div>
              <div className="flex-1">
                <h3 className={`font-semibold ${isSelected ? 'text-blue-900' : 'text-gray-900'}`}>
                  {channel.name}
                </h3>
                <p className="text-xs text-gray-600 mt-1">{channel.description}</p>
              </div>
              {isSelected && (
                <CheckCircle className="w-5 h-5 text-blue-600" />
              )}
            </div>
          </button>
        );
      })}
    </div>
  );

  const renderUPIPayment = () => (
    <div className="space-y-4">
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
        <div className="flex items-start gap-3">
          <Info className="w-5 h-5 text-blue-600 mt-0.5" />
          <div>
            <h4 className="font-medium text-blue-900">Pay via UPI</h4>
            <p className="text-sm text-blue-700 mt-1">
              Scan QR code or copy UPI ID to pay using any UPI app
            </p>
          </div>
        </div>
      </div>

      {qrCode && (
        <div className="bg-white border border-gray-200 rounded-lg p-6 text-center">
          <div className="inline-block p-4 bg-white border-2 border-gray-300 rounded-lg">
            <QrCode className="w-32 h-32 text-gray-400" />
            {/* In production, render actual QR code */}
          </div>
          <p className="text-sm text-gray-600 mt-4">Scan with any UPI app</p>
        </div>
      )}

      <div className="bg-white border border-gray-200 rounded-lg p-4">
        <label className="block text-sm font-medium text-gray-700 mb-2">
          Or pay using UPI ID
        </label>
        <div className="flex items-center gap-2">
          <input
            type="text"
            value={upiId}
            readOnly
            className="flex-1 px-4 py-2 border border-gray-300 rounded-lg bg-gray-50"
          />
          <button
            onClick={copyUPIId}
            className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
          >
            {copiedUPI ? (
              <Check className="w-5 h-5 text-green-600" />
            ) : (
              <Copy className="w-5 h-5 text-gray-600" />
            )}
          </button>
        </div>
      </div>

      {paymentStatus === 'awaiting_confirmation' && (
        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
          <div className="flex items-center gap-3">
            <Loader2 className="w-5 h-5 text-yellow-600 animate-spin" />
            <div>
              <h4 className="font-medium text-yellow-900">Waiting for payment confirmation</h4>
              <p className="text-sm text-yellow-700 mt-1">
                Complete payment in your UPI app
              </p>
            </div>
          </div>
        </div>
      )}
    </div>
  );

  const renderSavedMethods = () => (
    <div className="space-y-3">
      <h3 className="font-semibold text-gray-900">Saved Payment Methods</h3>
      {savedMethods.length === 0 ? (
        <div className="text-center py-8 bg-gray-50 rounded-lg border border-gray-200">
          <Wallet className="w-12 h-12 text-gray-300 mx-auto mb-3" />
          <p className="text-gray-500">No saved payment methods</p>
          <p className="text-sm text-gray-400 mt-1">Save a method during checkout for faster payments</p>
        </div>
      ) : (
        savedMethods.map((method) => (
          <div
            key={method.id}
            className="flex items-center justify-between p-4 border border-gray-200 rounded-lg hover:bg-gray-50"
          >
            <div className="flex items-center gap-3">
              <CreditCard className="w-5 h-5 text-gray-600" />
              <div>
                <p className="font-medium text-gray-900">
                  {method.channel === 'CARD' && `•••• ${method.display_info.last4}`}
                  {method.channel === 'UPI' && method.display_info.upi_id}
                </p>
                <p className="text-xs text-gray-500">
                  {method.channel === 'CARD' && method.display_info.brand}
                </p>
              </div>
            </div>
            {method.is_default && (
              <span className="px-2 py-1 bg-blue-100 text-blue-700 rounded text-xs font-medium">
                Default
              </span>
            )}
          </div>
        ))
      )}
    </div>
  );

  return (
    <div className="space-y-6">
      {/* Header */}
      <div>
        <h2 className="text-2xl font-bold text-gray-900 flex items-center gap-2">
          <Shield className="w-7 h-7 text-blue-600" />
          Multi-Channel Payment
        </h2>
        <p className="text-gray-600 mt-1">Choose your preferred payment method</p>
      </div>

      {/* Payment Amount Summary */}
      <div className="bg-gradient-to-r from-blue-600 to-blue-700 rounded-lg p-6 text-white">
        <p className="text-sm opacity-90">Amount to Pay</p>
        <p className="text-4xl font-bold mt-2">₹1,000</p>
        <p className="text-sm opacity-75 mt-1">Property booking payment</p>
      </div>

      {/* Channel Selector */}
      <div>
        <h3 className="font-semibold text-gray-900 mb-4">Select Payment Method</h3>
        {renderChannelSelector()}
      </div>

      {/* Channel-specific UI */}
      {selectedChannel === 'UPI' && renderUPIPayment()}

      {/* Saved Methods */}
      {renderSavedMethods()}

      {/* Pay Button */}
      <button
        onClick={initiatePayment}
        disabled={loading || paymentStatus === 'processing'}
        className="w-full py-4 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors font-semibold text-lg flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
      >
        {loading ? (
          <>
            <Loader2 className="w-5 h-5 animate-spin" />
            Processing...
          </>
        ) : (
          <>
            Proceed to Pay
            <ChevronRight className="w-5 h-5" />
          </>
        )}
      </button>

      {/* Security Notice */}
      <div className="bg-green-50 border border-green-200 rounded-lg p-4">
        <div className="flex items-start gap-3">
          <Shield className="w-5 h-5 text-green-600 mt-0.5" />
          <div>
            <h4 className="font-medium text-green-900">Secure Payment</h4>
            <p className="text-sm text-green-700 mt-1">
              Your payment is secured with 256-bit encryption. We never store your card details.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}
