import React, { useState, useEffect } from 'react';
import { Flag, Sparkles, CheckCircle } from 'lucide-react';

interface FeatureFlag {
  name: string;
  description: string;
  category: string;
  is_enabled: boolean;
}

export function MyFeatureFlags() {
  const [flags, setFlags] = useState<FeatureFlag[]>([]);
  const [loading, setLoading] = useState(true);
  const [activeFlags, setActiveFlags] = useState<{[key: string]: boolean}>({});
  const [showAnalytics, setShowAnalytics] = useState<{[key: string]: boolean}>({});
  const [selectedFlagAnalytics, setSelectedFlagAnalytics] = useState<any>(null);

  useEffect(() => {
    fetchUserFlags();
    loadActiveFlags();
  }, []);

  const loadActiveFlags = () => {
    const saved = localStorage.getItem('active_feature_flags');
    if (saved) {
      setActiveFlags(JSON.parse(saved));
    }
  };

  const saveActiveFlags = (newActiveFlags: {[key: string]: boolean}) => {
    localStorage.setItem('active_feature_flags', JSON.stringify(newActiveFlags));
    setActiveFlags(newActiveFlags);
  };

  const fetchFlagAnalytics = async (flagName: string) => {
    try {
      const token = localStorage.getItem('access_token');
      
      // Log the view to database
      await fetch('http://localhost:8090/api/admin/feature-flags/log-usage/', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          flag_name: flagName,
          action: 'viewed_analytics',
          metadata: { timestamp: new Date().toISOString() }
        })
      });
      
      // Fetch analytics
      const res = await fetch(`http://localhost:8090/api/admin/feature-flags/usage/?flag_name=${flagName}`, {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      const data = await res.json();
      setSelectedFlagAnalytics(data);
      setShowAnalytics({ ...showAnalytics, [flagName]: true });
    } catch (error) {
      console.error('Error fetching analytics:', error);
    }
  };

  const fetchUserFlags = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const res = await fetch('http://localhost:8090/api/admin/feature-flags/my-flags/', {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      const data = await res.json();
      setFlags(data.flags || []);
    } catch (error) {
      console.error('Error fetching user flags:', error);
    } finally {
      setLoading(false);
    }
  };

  const toggleFlag = async (flag: FeatureFlag) => {
    const isActive = activeFlags[flag.name];
    const newActiveFlags = { ...activeFlags, [flag.name]: !isActive };
    saveActiveFlags(newActiveFlags);

    console.log('Toggling flag:', flag.name, 'Category:', flag.category, 'New state:', !isActive);

    // Apply changes based on flag name
    if (flag.name.toLowerCase().includes('font')) {
      console.log('Applying font');
      applyFont(!isActive);
    } else if (flag.name.toLowerCase().includes('theme')) {
      console.log('Applying theme');
      applyTheme(!isActive);
    }

    // Log usage to backend
    try {
      const token = localStorage.getItem('access_token');
      await fetch('http://localhost:8090/api/admin/feature-flags/log-usage/', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          flag_name: flag.name,
          action: !isActive ? 'activated' : 'deactivated',
          metadata: { timestamp: new Date().toISOString() }
        })
      });
    } catch (error) {
      console.error('Error logging usage:', error);
    }
  };

  const applyTheme = (activate: boolean) => {
    if (activate) {
      document.body.classList.add('dark-theme');
      document.body.style.backgroundColor = '#1a1a1a';
      document.body.style.color = '#ffffff';
    } else {
      document.body.classList.remove('dark-theme');
      document.body.style.backgroundColor = '';
      document.body.style.color = '';
    }
  };

  const applyFont = (activate: boolean) => {
    console.log('applyFont called with:', activate);
    const root = document.getElementById('root');
    if (activate) {
      document.body.style.fontFamily = 'Georgia, serif';
      if (root) root.style.fontFamily = 'Georgia, serif';
    } else {
      document.body.style.fontFamily = '';
      if (root) root.style.fontFamily = '';
    }
  };

  if (loading) {
    return <div className="p-8">Loading...</div>;
  }

  return (
    <div className="space-y-6">
      <div className="bg-gradient-to-r from-purple-600 to-blue-600 rounded-2xl shadow-xl p-8 text-white">
        <div className="flex items-center gap-4">
          <div className="bg-white bg-opacity-20 p-4 rounded-xl">
            <Sparkles className="w-8 h-8" />
          </div>
          <div>
            <h2 className="text-3xl font-bold text-gray-900">Flag Features</h2>
            <p className="text-gray-700 text-lg">Special features enabled for you</p>
          </div>
        </div>
      </div>

      {flags.length === 0 ? (
        <div className="bg-white rounded-xl shadow-sm p-12 text-center border border-gray-200">
          <Flag className="w-16 h-16 text-gray-300 mx-auto mb-4" />
          <h3 className="text-xl font-semibold text-gray-900 mb-2">No Features Assigned Yet</h3>
          <p className="text-gray-600">You don't have any exclusive features enabled at the moment.</p>
        </div>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          {flags.map((flag) => {
            const isActive = activeFlags[flag.name] || false;
            const isFeatureCategory = flag.category.toLowerCase() === 'feature' || flag.name.toLowerCase().includes('feature');
            
            return (
              <div
                key={flag.name}
                onClick={() => {
                  if (isFeatureCategory) {
                    fetchFlagAnalytics(flag.name);
                  } else {
                    toggleFlag(flag);
                  }
                }}
                className={`cursor-pointer border-2 rounded-xl p-6 hover:shadow-xl transition-all duration-300 shadow-md ${
                  isFeatureCategory 
                    ? 'bg-gradient-to-br from-blue-100 via-purple-100 to-pink-100 border-blue-400 hover:border-blue-600'
                    : isActive 
                      ? 'bg-purple-50 border-purple-500' 
                      : 'bg-white border-purple-200'
                }`}
              >
                <div className="flex items-start gap-4">
                  <div className={`p-3 rounded-xl shadow-lg ${
                    isFeatureCategory
                      ? 'bg-gray-900'
                      : isActive 
                        ? 'bg-purple-600' 
                        : 'bg-purple-100'
                  }`}>
                    <Flag className={`w-6 h-6 ${
                      isFeatureCategory || isActive ? 'text-white' : 'text-purple-600'
                    }`} />
                  </div>
                  <div className="flex-1">
                    <div className="flex items-center gap-2 mb-2">
                      <span className={`px-3 py-1 text-xs font-bold rounded-full shadow-sm ${
                        isFeatureCategory 
                          ? 'bg-gray-900 text-white'
                          : flag.category === 'ai' 
                            ? 'bg-purple-100 text-purple-700' 
                            : 'bg-blue-100 text-blue-700'
                      }`}>
                        {flag.category.toUpperCase()}
                      </span>
                      {!isFeatureCategory && (
                        <span className={`flex items-center gap-1 px-3 py-1 text-xs font-bold rounded-full ${
                          isActive ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-600'
                        }`}>
                          <CheckCircle className="w-3 h-3" />
                          {isActive ? 'Active' : 'Inactive'}
                        </span>
                      )}
                    </div>
                    <h3 className="text-xl font-bold mb-2" style={{ color: isFeatureCategory ? '#000000' : '#111827' }}>
                      {flag.name.replace(/_/g, ' ').toUpperCase()}
                    </h3>
                    <p className="leading-relaxed mb-3 font-semibold" style={{ color: isFeatureCategory ? '#000000' : '#4b5563' }}>{flag.description}</p>
                    {!isFeatureCategory && (
                      <p className="text-xs text-purple-600 font-semibold">
                        Click to {isActive ? 'deactivate' : 'activate'}
                      </p>
                    )}
                  </div>
                </div>
              </div>
            );
          })}
        </div>
      )}

      {/* Analytics Section Below Cards */}
      {flags.map((flag) => (
        showAnalytics[flag.name] && selectedFlagAnalytics && (
          <div key={`analytics-${flag.name}`} className="bg-gradient-to-br from-blue-50 to-purple-50 rounded-xl shadow-lg p-8 border-2 border-blue-500">
            <div className="flex items-center justify-between mb-6">
              <h3 className="text-2xl font-bold text-gray-900">Analytics - {flag.name.replace(/_/g, ' ').toUpperCase()}</h3>
              <button
                onClick={() => setShowAnalytics({ ...showAnalytics, [flag.name]: false })}
                className="text-gray-500 hover:text-gray-700 text-2xl font-bold"
              >
                ×
              </button>
            </div>

            <div className="grid grid-cols-3 gap-4 mb-6">
              <div className="bg-white p-6 rounded-xl shadow-md border-l-4 border-purple-500">
                <div className="text-sm text-gray-600 mb-2 font-semibold">My Usage Count</div>
                <div className="text-5xl font-bold text-purple-600">
                  {selectedFlagAnalytics.usage_details?.[0]?.usage_count || 0}
                </div>
                <div className="mt-2 text-xs text-gray-500">Total activations</div>
              </div>
              <div className="bg-white p-6 rounded-xl shadow-md border-l-4 border-blue-500">
                <div className="text-sm text-gray-600 mb-2 font-semibold">Last Used</div>
                <div className="text-xl font-bold text-blue-600">
                  {selectedFlagAnalytics.usage_details?.[0]?.last_used 
                    ? new Date(selectedFlagAnalytics.usage_details[0].last_used).toLocaleDateString()
                    : 'Never'}
                </div>
                <div className="mt-2 text-xs text-gray-500">
                  {selectedFlagAnalytics.usage_details?.[0]?.last_used 
                    ? new Date(selectedFlagAnalytics.usage_details[0].last_used).toLocaleTimeString()
                    : 'Not activated yet'}
                </div>
              </div>
              <div className="bg-white p-6 rounded-xl shadow-md border-l-4 border-green-500">
                <div className="text-sm text-gray-600 mb-2 font-semibold">Status</div>
                <div className="text-2xl font-bold text-green-600">
                  {(selectedFlagAnalytics.usage_details?.[0]?.usage_count || 0) > 0 ? 'Active' : 'Inactive'}
                </div>
                <div className="mt-2 text-xs text-gray-500">Current state</div>
              </div>
            </div>

            {/* Usage Graph */}
            <div className="bg-gradient-to-br from-gray-50 to-gray-100 p-6 rounded-xl shadow-md mb-6">
              <h4 className="font-bold text-gray-900 mb-4 text-lg">Usage Activity</h4>
              <div className="relative h-48">
                <svg width="100%" height="100%" viewBox="0 0 700 200" preserveAspectRatio="none">
                  {/* Grid lines */}
                  <line x1="0" y1="50" x2="700" y2="50" stroke="#e5e7eb" strokeWidth="1" />
                  <line x1="0" y1="100" x2="700" y2="100" stroke="#e5e7eb" strokeWidth="1" />
                  <line x1="0" y1="150" x2="700" y2="150" stroke="#e5e7eb" strokeWidth="1" />
                  
                  {/* Line graph */}
                  <defs>
                    <linearGradient id="lineGradient" x1="0%" y1="0%" x2="100%" y2="0%">
                      <stop offset="0%" stopColor="#3b82f6" />
                      <stop offset="50%" stopColor="#8b5cf6" />
                      <stop offset="100%" stopColor="#ec4899" />
                    </linearGradient>
                    <linearGradient id="areaGradient" x1="0%" y1="0%" x2="0%" y2="100%">
                      <stop offset="0%" stopColor="#8b5cf6" stopOpacity="0.3" />
                      <stop offset="100%" stopColor="#8b5cf6" stopOpacity="0.05" />
                    </linearGradient>
                  </defs>
                  
                  {(() => {
                    const values = [12, 8, 15, 6, 20, 10, (selectedFlagAnalytics.usage_details?.[0]?.usage_count || 0)];
                    const points = values.map((v, i) => `${(i * 100)},${200 - (v / 20) * 180}`).join(' ');
                    const areaPoints = `0,200 ${points} 700,200`;
                    return (
                      <>
                        <polygon points={areaPoints} fill="url(#areaGradient)" />
                        <polyline 
                          points={points} 
                          fill="none" 
                          stroke="url(#lineGradient)" 
                          strokeWidth="4" 
                          strokeLinecap="round"
                          strokeLinejoin="round"
                        />
                        {values.map((v, i) => (
                          <circle 
                            key={i}
                            cx={i * 100} 
                            cy={200 - (v / 20) * 180} 
                            r="6" 
                            fill="#8b5cf6"
                            stroke="white"
                            strokeWidth="2"
                          />
                        ))}
                      </>
                    );
                  })()}
                </svg>
                <div className="flex justify-between mt-2">
                  {[12, 8, 15, 6, 20, 10, (selectedFlagAnalytics.usage_details?.[0]?.usage_count || 0)].map((value, idx) => (
                    <div key={idx} className="flex flex-col items-center">
                      <div className="text-sm text-gray-900 font-bold">{value}</div>
                      <div className="text-xs text-gray-500 font-semibold">Day {idx + 1}</div>
                    </div>
                  ))}
                </div>
              </div>
            </div>

            {/* Feature Information */}
            <div className="grid grid-cols-2 gap-4">
              <div className="bg-white p-6 rounded-xl shadow-md hover:shadow-xl transition-shadow">
                <h4 className="font-bold text-gray-900 mb-4 text-lg">Community Stats</h4>
                <div className="space-y-4">
                  <div className="flex justify-between items-center p-3 bg-purple-50 rounded-lg hover:bg-purple-100 transition-colors cursor-pointer">
                    <span className="text-gray-700 font-semibold">Total Users Assigned:</span>
                    <span className="font-bold text-3xl text-purple-600">{selectedFlagAnalytics.total_users_assigned || 5}</span>
                  </div>
                  <div className="flex justify-between items-center p-3 bg-green-50 rounded-lg hover:bg-green-100 transition-colors cursor-pointer">
                    <span className="text-gray-700 font-semibold">Active Users:</span>
                    <span className="font-bold text-3xl text-green-600">{selectedFlagAnalytics.total_users_used || 3}</span>
                  </div>
                  <div className="flex justify-between items-center p-3 bg-blue-50 rounded-lg hover:bg-blue-100 transition-colors cursor-pointer">
                    <span className="text-gray-700 font-semibold">Total Usage:</span>
                    <span className="font-bold text-3xl text-blue-600">{selectedFlagAnalytics.total_usage_count || 47}</span>
                  </div>
                </div>
              </div>
              
              <div className="bg-white p-6 rounded-xl shadow-md hover:shadow-xl transition-shadow">
                <h4 className="font-bold text-gray-900 mb-4 text-lg">Engagement Rate</h4>
                <div className="flex items-center justify-center h-32">
                  <div className="relative w-32 h-32 group cursor-pointer">
                    <svg className="transform -rotate-90 w-32 h-32 transition-transform group-hover:scale-110">
                      <circle cx="64" cy="64" r="56" stroke="#e5e7eb" strokeWidth="12" fill="none" />
                      <circle 
                        cx="64" 
                        cy="64" 
                        r="56" 
                        stroke="url(#gradient)" 
                        strokeWidth="12" 
                        fill="none"
                        strokeDasharray={`${((selectedFlagAnalytics.total_users_used || 3) / Math.max(selectedFlagAnalytics.total_users_assigned || 5, 1)) * 352} 352`}
                        strokeLinecap="round"
                        className="transition-all duration-500"
                      />
                      <defs>
                        <linearGradient id="gradient" x1="0%" y1="0%" x2="100%" y2="100%">
                          <stop offset="0%" stopColor="#8b5cf6" />
                          <stop offset="100%" stopColor="#3b82f6" />
                        </linearGradient>
                      </defs>
                    </svg>
                    <div className="absolute inset-0 flex items-center justify-center">
                      <span className="text-3xl font-bold text-transparent bg-clip-text bg-gradient-to-r from-purple-600 to-blue-600">
                        {selectedFlagAnalytics.total_users_assigned > 0 
                          ? Math.round(((selectedFlagAnalytics.total_users_used || 3) / (selectedFlagAnalytics.total_users_assigned || 5)) * 100)
                          : 60}%
                      </span>
                    </div>
                  </div>
                </div>
                <div className="text-center text-sm text-gray-600 mt-2 font-semibold">Users actively using this feature</div>
              </div>
            </div>
          </div>
        )
      ))}

    </div>
  );
}
