import { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { Plus, MapPin, Trash2, Calendar } from 'lucide-react';
import { toast } from 'sonner';

export const MyItinerary = () => {
  const navigate = useNavigate();
  const [userProperties, setUserProperties] = useState<any[]>([]);

  useEffect(() => {
    const properties = JSON.parse(localStorage.getItem('user_itineraries') || '[]');
    setUserProperties(properties);
  }, []);

  const handleView = (id: string) => navigate(`/itinerary-detail/${id}`);
  const handleRemove = (id: number) => {
    const updated = userProperties.filter(p => p.id !== id);
    setUserProperties(updated);
    localStorage.setItem('user_itineraries', JSON.stringify(updated));
    toast.success('Removed from Itinerary', {
      description: 'Property removed from your itinerary'
    });
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between mb-6">
        <h3 className="text-lg font-semibold text-gray-900">My Itinerary</h3>
        <button onClick={() => navigate('/properties')} className="bg-black text-white px-4 py-2 rounded-lg hover:bg-gray-800 transition-colors flex items-center gap-2">
          <Plus className="w-5 h-5" />
          Browse Properties
        </button>
      </div>

      {userProperties.length === 0 ? (
        <div className="bg-white rounded-lg border border-gray-200 p-12 text-center">
          <h3 className="text-xl font-semibold text-gray-900 mb-2">No properties in itinerary</h3>
          <p className="text-gray-600 mb-6">Add properties to your itinerary to plan visits</p>
          <button onClick={() => navigate('/properties')} className="bg-black text-white px-6 py-3 rounded-lg hover:bg-gray-800 transition-colors inline-flex items-center gap-2">
            <Plus className="w-5 h-5" />
            Browse Properties
          </button>
        </div>
      ) : (
        <div className="bg-white rounded-lg border border-gray-200">
          <div className="divide-y divide-gray-200">
            {userProperties.map((property) => (
              <div key={property.id} className="p-4 hover:bg-gray-50 transition-colors">
                <div className="flex items-start justify-between gap-4">
                  <div onClick={() => handleView(property.id)} className="cursor-pointer flex-1">
                    <h3 className="text-lg font-semibold text-gray-900 mb-1">{property.title}</h3>
                    <div className="flex items-center gap-2 text-sm text-gray-600 mb-2">
                      <MapPin className="w-4 h-4" />
                      <span>{property.address}, {property.city}</span>
                    </div>
                    <div className="flex items-center gap-4 text-sm text-gray-600">
                      {property.bedrooms && <span>{property.bedrooms} BHK</span>}
                      {property.built_up_area && <span>{property.built_up_area} sq.ft</span>}
                      <span className="text-lg font-bold text-blue-600">₹{property.sale_price?.toLocaleString()}</span>
                    </div>
                    <div className="flex items-center gap-2 text-blue-600 font-medium mt-2">
                      <Calendar className="w-4 h-4" />
                      <span className="text-sm">Plan Visit</span>
                    </div>
                  </div>
                  <button 
                    onClick={(e) => { e.stopPropagation(); handleRemove(property.id); }} 
                    className="bg-red-600 text-white px-4 py-2 rounded-lg hover:bg-red-700 transition-colors flex items-center gap-2 flex-shrink-0"
                  >
                    <Trash2 className="w-4 h-4" />
                    Remove
                  </button>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}
    </div>
  );
};
