import React, { useState, useEffect } from 'react';
import { Eye, MapPin, Calendar, Edit, Trash2, RefreshCw, Heart, Share2 } from 'lucide-react';
import { toast } from 'sonner';
import { motion } from 'framer-motion';

interface Property {
  id: number;
  title: string;
  description: string;
  listing_type: string;
  property_type: string;
  status: string;
  address: string;
  city: string;
  locality: string;
  bedrooms: number;
  bathrooms: number;
  sale_price: number;
  monthly_rent: number;
  primary_image: string;
  views: number;
  inquiries: number;
  created_at: string;
  updated_at: string;
}

interface PropertiesResponse {
  properties: Property[];
  total_count: number;
  page: number;
  limit: number;
}

export function MyProperties() {
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);
  const [totalCount, setTotalCount] = useState(0);
  const [page, setPage] = useState(1);
  const [limit] = useState(10);

  const fetchMyProperties = async () => {
    try {
      setLoading(true);
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id;
      
      if (!userId) {
        toast.error('User not found. Please login again.');
        return;
      }

      // Fetch properties from agent dashboard API
      const response = await fetch(`http://localhost:8090/api/agent-dashboard/properties?user_id=${userId}`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('access_token')}`,
          'Content-Type': 'application/json'
        }
      });

      if (!response.ok) {
        throw new Error('Failed to fetch properties');
      }

      const userProperties = await response.json();
      
      // Apply pagination
      const startIndex = (page - 1) * limit;
      const endIndex = startIndex + limit;
      const paginatedProperties = userProperties.slice(startIndex, endIndex);
      
      setProperties(paginatedProperties);
      setTotalCount(userProperties.length);
    } catch (error) {
      console.error('Error fetching properties:', error);
      toast.error('Failed to load your properties');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchMyProperties();
  }, [page]);

  const formatPrice = (price: number, type: string) => {
    if (!price) return 'Price not set';
    if (type === 'rent') {
      return `₹${price.toLocaleString()}/month`;
    }
    return `₹${(price / 100000).toFixed(1)}L`;
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'pending': return 'bg-yellow-100 text-yellow-800';
      case 'sold': return 'bg-blue-100 text-blue-800';
      case 'inactive': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">My Properties</h2>
          <p className="text-gray-600">Manage your listed properties</p>
        </div>
        <button
          onClick={fetchMyProperties}
          className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
        >
          <RefreshCw className="w-4 h-4" />
          Refresh
        </button>
      </div>

      {properties.length === 0 ? (
        <div className="text-center py-12 bg-white rounded-lg border border-gray-200">
          <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
            <Eye className="w-8 h-8 text-gray-400" />
          </div>
          <h3 className="text-lg font-medium text-gray-900 mb-2">No Properties Listed</h3>
          <p className="text-gray-600 mb-4">You haven't listed any properties yet.</p>
          <button
            onClick={() => window.location.href = '/list-property'}
            className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            List Your First Property
          </button>
        </div>
      ) : (
        <>
          <div className="mb-4">
            <h3 className="text-lg font-medium text-gray-900">
              Your Properties ({totalCount})
            </h3>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {properties.map((property) => (
              <motion.div
                key={property.id}
                whileHover={{ scale: 1.02 }}
                className="bg-white rounded-xl shadow-md overflow-hidden relative border border-gray-200 hover:shadow-lg transition-all duration-300 cursor-pointer w-full h-full flex flex-col"
                onClick={() => window.open(`/property/${property.id}`, '_blank')}
              >
                {/* Image Section */}
                <div className="relative h-48 w-full overflow-hidden">
                  {property.primary_image ? (
                    <img
                      src={property.primary_image}
                      alt={property.title}
                      className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
                      onError={(e) => {
                        (e.target as HTMLImageElement).src = '/placeholder-property.jpg';
                      }}
                    />
                  ) : (
                    <div className="w-full h-full bg-gradient-to-br from-blue-50 to-indigo-100 flex items-center justify-center">
                      <div className="text-center">
                        <svg
                          className="w-12 h-12 text-blue-300 mx-auto mb-2"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={1.5}
                            d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14"
                          />
                        </svg>
                        <p className="text-blue-400 text-sm font-medium">
                          No image available
                        </p>
                      </div>
                    </div>
                  )}

                  {/* Badges */}
                  <div className="absolute top-3 left-3 flex flex-wrap gap-2">
                    <span className={`text-white text-xs font-bold px-2 py-1 rounded-full shadow ${getStatusColor(property.status).includes('green') ? 'bg-green-600' : getStatusColor(property.status).includes('yellow') ? 'bg-yellow-600' : getStatusColor(property.status).includes('blue') ? 'bg-blue-600' : 'bg-gray-600'}`}>
                      {property.status.toUpperCase()}
                    </span>
                    <span className="bg-[#0056D2] text-white text-xs font-bold px-2 py-1 rounded-full shadow">
                      {property.listing_type.toUpperCase()}
                    </span>
                  </div>

                  {/* Action Buttons */}
                  <div className="absolute top-3 right-3 flex flex-col gap-2">
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        toast.info('Edit functionality coming soon!');
                      }}
                      className="bg-white/90 p-2 rounded-full shadow-md hover:scale-110 transition-all"
                      title="Edit Property"
                    >
                      <Edit className="h-4 w-4 text-gray-600" />
                    </button>
                    
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        if (navigator.share) {
                          navigator.share({
                            title: property.title,
                            text: `Check out my property: ${property.title}`,
                            url: window.location.origin + `/property/${property.id}`
                          });
                        } else {
                          navigator.clipboard.writeText(window.location.origin + `/property/${property.id}`);
                          toast.success('Property link copied to clipboard!');
                        }
                      }}
                      className="bg-white/90 p-2 rounded-full shadow-md hover:scale-110 transition-all"
                      title="Share Property"
                    >
                      <Share2 className="h-4 w-4 text-gray-600" />
                    </button>
                  </div>
                </div>

                {/* Details Section */}
                <div className="p-4 flex-1 flex flex-col">
                  {/* Title & Price */}
                  <div className="mb-3">
                    <h3 className="text-lg font-semibold text-gray-900 mb-2 line-clamp-2">
                      {property.title}
                    </h3>
                    <p className="text-blue-600 text-xl font-bold mb-1">
                      {formatPrice(
                        property.listing_type === 'rent' ? property.monthly_rent : property.sale_price,
                        property.listing_type
                      )}
                    </p>
                    <p className="text-gray-500 text-sm flex items-center">
                      <MapPin className="w-4 h-4 mr-1 text-gray-400" />
                      {property.locality}, {property.city}
                    </p>
                  </div>

                  {/* Stats */}
                  <div className="flex justify-between text-sm text-gray-600 mb-3">
                    <div className="flex items-center gap-1">
                      <Eye className="w-4 h-4 text-[#0056D2]" />
                      <span className="font-medium">{property.views} views</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <svg className="w-4 h-4 text-[#0056D2]" fill="currentColor" viewBox="0 0 20 20">
                        <path d="M2.003 5.884L10 9.882l7.997-3.998A2 2 0 0016 4H4a2 2 0 00-1.997 1.884z" />
                        <path d="M18 8.118l-8 4-8-4V14a2 2 0 002 2h12a2 2 0 002-2V8.118z" />
                      </svg>
                      <span className="font-medium">{property.inquiries} inquiries</span>
                    </div>
                  </div>

                  {/* Specs */}
                  <div className="flex justify-between text-sm text-gray-600 border-t pt-3 mt-auto">
                    <div className="flex items-center gap-1">
                      <svg className="w-4 h-4 text-[#0056D2]" fill="currentColor" viewBox="0 0 20 20">
                        <path d="M10.707 2.293a1 1 0 00-1.414 0l-7 7a1 1 0 001.414 1.414L4 10.414V17a1 1 0 001 1h2a1 1 0 001-1v-2h2v2a1 1 0 001 1h2a1 1 0 001-1v-6.586l.293.293a1 1 0 001.414-1.414l-7-7z" />
                      </svg>
                      <span className="font-medium">{property.bedrooms} Beds</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <svg className="w-4 h-4 text-[#0056D2]" fill="currentColor" viewBox="0 0 20 20">
                        <path d="M3 4a1 1 0 011-1h12a1 1 0 011 1v2a1 1 0 01-1 1H4a1 1 0 01-1-1V4zM3 10a1 1 0 011-1h6a1 1 0 011 1v6a1 1 0 01-1 1H4a1 1 0 01-1-1v-6z" />
                      </svg>
                      <span className="font-medium">{property.bathrooms} Baths</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <svg className="w-4 h-4 text-[#0056D2]" fill="currentColor" viewBox="0 0 20 20">
                        <path fillRule="evenodd" d="M3 3a1 1 0 000 2v8a2 2 0 002 2h2.586l-1.293 1.293a1 1 0 101.414 1.414L10 15.414l2.293 2.293a1 1 0 001.414-1.414L12.414 15H15a2 2 0 002-2V5a1 1 0 100-2H3z" clipRule="evenodd" />
                      </svg>
                      <span className="font-medium">{property.built_up_area || 'N/A'} sqft</span>
                    </div>
                  </div>

                  {/* Listed Date */}
                  <div className="text-xs text-gray-500 mt-2 text-center">
                    Listed {new Date(property.created_at).toLocaleDateString()}
                  </div>
                </div>
              </motion.div>
            ))}
          </div>

          {/* Pagination */}
          {totalCount > limit && (
            <div className="flex items-center justify-between bg-white px-6 py-4 border border-gray-200 rounded-lg mt-6">
              <div className="text-sm text-gray-700">
                Showing {((page - 1) * limit) + 1} to {Math.min(page * limit, totalCount)} of {totalCount} properties
              </div>
              <div className="flex gap-2">
                <button
                  onClick={() => setPage(page - 1)}
                  disabled={page === 1}
                  className="px-4 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                >
                  Previous
                </button>
                <button
                  onClick={() => setPage(page + 1)}
                  disabled={page * limit >= totalCount}
                  className="px-4 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                >
                  Next
                </button>
              </div>
            </div>
          )}
        </>
      )}
    </div>
  );
}