import React from 'react';
import { useOwnerPlans, useMySubscription } from '../../hooks/useListingPackages';

const API_BASE_URL = import.meta.env.VITE_API_URL || 'http://localhost:8000';

interface OwnerListingPackagesProps {
  userId?: number;
}

const OwnerListingPackages: React.FC<OwnerListingPackagesProps> = ({ userId = 1 }) => {
  const { plans, loading } = useOwnerPlans();
  const { subscription } = useMySubscription(userId);

  const handleSubscribe = async (planId: number) => {
    try {
      const response = await fetch(`${API_BASE_URL}/api/owner/subscribe-plan/`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ plan_id: planId, user_id: userId })
      });
      
      const data = await response.json();
      if (data.success) {
        alert('Plan subscription initiated! Redirecting to payment...');
        // Redirect to payment gateway or show payment modal
      }
    } catch (error) {
      console.error('Error subscribing to plan:', error);
      alert('Failed to subscribe. Please try again.');
    }
  };

  if (loading) return <div className="p-8">Loading plans...</div>;

  return (
    <div className="p-8 bg-gray-50 min-h-screen">
      <div className="max-w-7xl mx-auto">
        <h1 className="text-4xl font-bold text-center mb-4">Choose Your Listing Package</h1>
        <p className="text-center text-gray-600 mb-8">Select the perfect plan to showcase your properties</p>
        
        {subscription && (
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 mb-8">
            <h3 className="text-xl font-semibold mb-2">Current Plan: {subscription.plan.name}</h3>
            <p className="text-gray-700">
              Valid until: {new Date(subscription.end_date).toLocaleDateString('en-IN', { 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric' 
              })}
            </p>
          </div>
        )}

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
          {plans.map((plan) => (
            <div 
              key={plan.id} 
              className={`bg-white rounded-xl shadow-lg overflow-hidden transform transition hover:scale-105 ${
                plan.is_recommended ? 'ring-4 ring-yellow-400' : ''
              }`}
            >
              {plan.is_recommended && (
                <div className="bg-yellow-400 text-center py-2 font-semibold text-sm">
                  ⭐ RECOMMENDED
                </div>
              )}
              
              <div className="p-6">
                <h3 className="text-2xl font-bold mb-2">{plan.name}</h3>
                <p className="text-gray-600 mb-4">{plan.package.description}</p>
                
                <div className="mb-6">
                  <div className="flex items-baseline">
                    <span className="text-4xl font-bold">{plan.currency} {plan.price.toLocaleString('en-IN')}</span>
                    <span className="text-gray-500 ml-2">/ {plan.billing_cycle}</span>
                  </div>
                  {plan.discount_percent > 0 && (
                    <div className="text-green-600 font-semibold mt-1">
                      Save {plan.discount_percent}%
                    </div>
                  )}
                </div>

                <div className="mb-6">
                  <h4 className="font-semibold mb-3 text-gray-700">Features:</h4>
                  <ul className="space-y-2">
                    {plan.package.features_json.map((feature, index) => (
                      <li key={index} className="flex items-start">
                        <svg className="w-5 h-5 text-green-500 mr-2 flex-shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                          <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                        </svg>
                        <span className="text-sm text-gray-700">{feature}</span>
                      </li>
                    ))}
                  </ul>
                </div>

                <div className="mb-6 bg-gray-50 rounded-lg p-4">
                  <h4 className="font-semibold mb-2 text-gray-700">Limits:</h4>
                  <ul className="space-y-1 text-sm">
                    {Object.entries(plan.limits_json).map(([key, value]) => (
                      <li key={key} className="flex justify-between">
                        <span className="text-gray-600 capitalize">{key.replace(/_/g, ' ')}:</span>
                        <span className="font-semibold">{String(value)}</span>
                      </li>
                    ))}
                  </ul>
                </div>

                <button 
                  onClick={() => handleSubscribe(plan.id)}
                  disabled={subscription?.plan.id === plan.id}
                  className={`w-full py-3 rounded-lg font-semibold transition ${
                    subscription?.plan.id === plan.id
                      ? 'bg-gray-300 text-gray-600 cursor-not-allowed'
                      : plan.is_recommended
                      ? 'bg-yellow-500 hover:bg-yellow-600 text-white'
                      : 'bg-blue-600 hover:bg-blue-700 text-white'
                  }`}
                >
                  {subscription?.plan.id === plan.id ? 'Current Plan' : 'Subscribe Now'}
                </button>
              </div>
            </div>
          ))}
        </div>

        <div className="mt-12 bg-white rounded-lg shadow p-6">
          <h3 className="text-xl font-semibold mb-4">Need Help Choosing?</h3>
          <p className="text-gray-600 mb-4">
            Not sure which package is right for you? Contact our support team for personalized recommendations.
          </p>
          <button className="bg-gray-800 text-white px-6 py-2 rounded-lg hover:bg-gray-900">
            Contact Support
          </button>
        </div>
      </div>
    </div>
  );
};

export default OwnerListingPackages;
