import React, { useState, useEffect } from 'react';
import {
  CreditCard,
  Download,
  Search,
  Calendar,
  CheckCircle,
  Clock,
  XCircle,
  FileText,
  AlertCircle,
  TrendingUp,
  Wallet,
  Receipt,
  Home,
  RefreshCw,
  DollarSign,
  QrCode,
  Shield,
  Milestone
} from 'lucide-react';
import { PropertyPurchasePayments } from './PropertyPurchasePayments';
import { UPIPayment } from './UPIPayment';
import { SavedPaymentMethods } from './SavedPaymentMethods';
import { EscrowTracker } from './EscrowTracker';
import { MultiChannelPayment } from './MultiChannelPayment';

interface Payment {
  id: string;
  order_id: string;
  transaction_id?: string;
  amount: number;
  payment_type: string;
  payment_method?: string;
  payment_gateway?: string;
  status: string;
  created_at: string;
  paid_at?: string;
  due_date?: string;
  property_id?: string;
  refund_amount?: number;
  refund_reason?: string;
  refunded_at?: string;
  milestone_number?: number;
  total_milestones?: number;
}

interface EMISchedule {
  emi_number: number;
  total_emis: number;
  amount: number;
  due_date: string;
  status: 'paid' | 'pending' | 'overdue';
  paid_date?: string;
  payment_id?: string;
}

export function PaymentsTransactions() {
  const [activeTab, setActiveTab] = useState<'property_purchase' | 'multi_channel' | 'upi_payment' | 'saved_methods' | 'escrow' | 'pending' | 'completed' | 'refunds' | 'emi'>('property_purchase');
  const [searchTerm, setSearchTerm] = useState('');
  const [loading, setLoading] = useState(false);
  const [pendingPayments, setPendingPayments] = useState<Payment[]>([]);
  const [completedPayments, setCompletedPayments] = useState<Payment[]>([]);
  const [refundedPayments, setRefundedPayments] = useState<Payment[]>([]);
  const [emiSchedules, setEmiSchedules] = useState<EMISchedule[]>([]);

  useEffect(() => {
    if (activeTab !== 'property_purchase') {
      fetchPayments();
    }
  }, [activeTab]);

  const fetchPayments = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');

      if (activeTab === 'pending') {
        const response = await fetch('/api/payments/history?status=pending&page_size=50', {
          headers: { 'Authorization': `Bearer ${token}` }
        });
        const data = await response.json();
        setPendingPayments(data.payments || []);
      } else if (activeTab === 'completed') {
        const response = await fetch('/api/payments/history?status=success&page_size=50', {
          headers: { 'Authorization': `Bearer ${token}` }
        });
        const data = await response.json();
        setCompletedPayments(data.payments || []);
      } else if (activeTab === 'refunds') {
        const response = await fetch('/api/payments/history?status=refunded&page_size=50', {
          headers: { 'Authorization': `Bearer ${token}` }
        });
        const data = await response.json();
        setRefundedPayments(data.payments || []);
      } else if (activeTab === 'emi') {
        // Fetch EMI schedules - for now using milestone payments
        const response = await fetch('/api/payments/history?payment_type=milestone&page_size=50', {
          headers: { 'Authorization': `Bearer ${token}` }
        });
        const data = await response.json();

        // Convert milestone payments to EMI format
        const emiData: EMISchedule[] = (data.payments || []).map((p: Payment) => ({
          emi_number: p.milestone_number || 1,
          total_emis: p.total_milestones || 1,
          amount: p.amount,
          due_date: p.due_date || p.created_at,
          status: p.status === 'success' ? 'paid' : p.status === 'pending' ? 'pending' : 'overdue',
          paid_date: p.paid_at,
          payment_id: p.id
        }));
        setEmiSchedules(emiData);
      }
    } catch (error) {
      console.error('Failed to fetch payments:', error);
    } finally {
      setLoading(false);
    }
  };

  const handlePayNow = async (payment: Payment) => {
    // Redirect to property purchase tab for payment
    setActiveTab('property_purchase');
  };

  const downloadReceipt = async (paymentId: string) => {
    try {
      const token = localStorage.getItem('token');
      const response = await fetch(`/api/payments/${paymentId}/receipt`, {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      const data = await response.json();
      alert(`Receipt: ${data.receipt_number}\nDownload link: ${data.file_path}`);
    } catch (error) {
      console.error('Failed to download receipt:', error);
      alert('Failed to download receipt');
    }
  };

  const getStatusBadge = (status: string) => {
    const badges: Record<string, { bg: string; text: string; icon: any }> = {
      pending: { bg: 'bg-yellow-100', text: 'text-yellow-700', icon: Clock },
      success: { bg: 'bg-green-100', text: 'text-green-700', icon: CheckCircle },
      failed: { bg: 'bg-red-100', text: 'text-red-700', icon: XCircle },
      refunded: { bg: 'bg-blue-100', text: 'text-blue-700', icon: RefreshCw },
      paid: { bg: 'bg-green-100', text: 'text-green-700', icon: CheckCircle },
      overdue: { bg: 'bg-red-100', text: 'text-red-700', icon: AlertCircle }
    };
    const badge = badges[status] || badges.pending;
    const Icon = badge.icon;

    return (
      <span className={`inline-flex items-center gap-1 px-3 py-1 rounded-full text-xs font-medium ${badge.bg} ${badge.text}`}>
        <Icon className="w-3 h-3" />
        {status.charAt(0).toUpperCase() + status.slice(1)}
      </span>
    );
  };

  const getPaymentTypeLabel = (type: string) => {
    const labels: Record<string, string> = {
      booking: 'Booking Amount',
      token: 'Token Payment',
      milestone: 'Milestone Payment',
      full_payment: 'Full Payment',
      stamp_duty: 'Stamp Duty',
      registration: 'Registration Fee',
      service_charge: 'Service Charge'
    };
    return labels[type] || type;
  };

  const renderPaymentCard = (payment: Payment, showActions: boolean = true) => (
    <div key={payment.id} className="bg-white border border-gray-200 rounded-lg p-5 hover:shadow-md transition-shadow">
      <div className="flex items-start justify-between">
        <div className="flex-1">
          <div className="flex items-center gap-3 mb-2">
            <span className="px-3 py-1 bg-blue-100 text-blue-700 rounded-full text-xs font-medium">
              {getPaymentTypeLabel(payment.payment_type)}
            </span>
            {getStatusBadge(payment.status)}
            {payment.milestone_number && (
              <span className="text-xs text-gray-500">
                {payment.milestone_number}/{payment.total_milestones}
              </span>
            )}
          </div>

          <h3 className="font-semibold text-gray-900 mb-1">Order #{payment.order_id}</h3>
          {payment.transaction_id && (
            <p className="text-sm text-gray-600 mb-2">Transaction: {payment.transaction_id}</p>
          )}

          <div className="flex items-center gap-4 text-xs text-gray-500 mt-2">
            <span className="flex items-center gap-1">
              <Calendar className="w-3 h-3" />
              {new Date(payment.created_at).toLocaleDateString('en-IN')}
            </span>
            {payment.due_date && (
              <span className="flex items-center gap-1">
                <AlertCircle className="w-3 h-3" />
                Due: {new Date(payment.due_date).toLocaleDateString('en-IN')}
              </span>
            )}
            {payment.paid_at && (
              <span className="flex items-center gap-1 text-green-600">
                <CheckCircle className="w-3 h-3" />
                Paid: {new Date(payment.paid_at).toLocaleDateString('en-IN')}
              </span>
            )}
            {payment.payment_gateway && (
              <span className="flex items-center gap-1">
                <CreditCard className="w-3 h-3" />
                {payment.payment_gateway}
              </span>
            )}
          </div>
        </div>

        <div className="text-right ml-6">
          <p className="text-2xl font-bold text-gray-900 mb-2">
            ₹{payment.amount.toLocaleString('en-IN')}
          </p>

          {showActions && (
            <div className="space-y-2">
              {payment.status === 'pending' && (
                <button
                  onClick={() => handlePayNow(payment)}
                  className="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-sm font-medium"
                >
                  Pay Now
                </button>
              )}
              {payment.status === 'success' && (
                <button
                  onClick={() => downloadReceipt(payment.id)}
                  className="w-full px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors text-sm font-medium flex items-center justify-center gap-2"
                >
                  <Download className="w-4 h-4" />
                  Receipt
                </button>
              )}
            </div>
          )}
        </div>
      </div>
    </div>
  );

  const renderRefundCard = (payment: Payment) => (
    <div key={payment.id} className="bg-white border border-gray-200 rounded-lg p-5 hover:shadow-md transition-shadow">
      <div className="flex items-start justify-between">
        <div className="flex-1">
          <div className="flex items-center gap-3 mb-2">
            {getStatusBadge('refunded')}
            <span className="px-3 py-1 bg-purple-100 text-purple-700 rounded-full text-xs font-medium">
              {getPaymentTypeLabel(payment.payment_type)}
            </span>
          </div>

          <h3 className="font-semibold text-gray-900 mb-1">Order #{payment.order_id}</h3>
          {payment.transaction_id && (
            <p className="text-sm text-gray-600 mb-2">Transaction: {payment.transaction_id}</p>
          )}

          {payment.refund_reason && (
            <div className="mt-2 p-2 bg-blue-50 border border-blue-200 rounded text-xs text-blue-800">
              <strong>Reason:</strong> {payment.refund_reason}
            </div>
          )}

          <div className="flex items-center gap-4 text-xs text-gray-500 mt-3">
            <span className="flex items-center gap-1">
              <Calendar className="w-3 h-3" />
              Original: {new Date(payment.created_at).toLocaleDateString('en-IN')}
            </span>
            {payment.refunded_at && (
              <span className="flex items-center gap-1 text-blue-600">
                <RefreshCw className="w-3 h-3" />
                Refunded: {new Date(payment.refunded_at).toLocaleDateString('en-IN')}
              </span>
            )}
          </div>
        </div>

        <div className="text-right ml-6">
          <p className="text-sm text-gray-500 mb-1">Original Amount</p>
          <p className="text-xl font-bold text-gray-400 line-through mb-2">
            ₹{payment.amount.toLocaleString('en-IN')}
          </p>
          <p className="text-sm text-blue-600 font-medium mb-1">Refunded Amount</p>
          <p className="text-2xl font-bold text-blue-600">
            ₹{(payment.refund_amount || payment.amount).toLocaleString('en-IN')}
          </p>
        </div>
      </div>
    </div>
  );

  const renderEMICard = (emi: EMISchedule) => (
    <div key={`${emi.payment_id}-${emi.emi_number}`} className="bg-white border border-gray-200 rounded-lg p-5 hover:shadow-md transition-shadow">
      <div className="flex items-start justify-between">
        <div className="flex-1">
          <div className="flex items-center gap-3 mb-2">
            <span className="px-3 py-1 bg-indigo-100 text-indigo-700 rounded-full text-xs font-medium">
              EMI {emi.emi_number}/{emi.total_emis}
            </span>
            {getStatusBadge(emi.status)}
          </div>

          <h3 className="font-semibold text-gray-900 mb-1">
            Installment {emi.emi_number} of {emi.total_emis}
          </h3>

          <div className="mt-3">
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div
                className="bg-indigo-600 h-2 rounded-full transition-all"
                style={{ width: `${(emi.emi_number / emi.total_emis) * 100}%` }}
              ></div>
            </div>
            <p className="text-xs text-gray-500 mt-1">
              {emi.emi_number} of {emi.total_emis} payments completed
            </p>
          </div>

          <div className="flex items-center gap-4 text-xs text-gray-500 mt-3">
            <span className="flex items-center gap-1">
              <Calendar className="w-3 h-3" />
              Due: {new Date(emi.due_date).toLocaleDateString('en-IN')}
            </span>
            {emi.paid_date && (
              <span className="flex items-center gap-1 text-green-600">
                <CheckCircle className="w-3 h-3" />
                Paid: {new Date(emi.paid_date).toLocaleDateString('en-IN')}
              </span>
            )}
          </div>
        </div>

        <div className="text-right ml-6">
          <p className="text-2xl font-bold text-gray-900 mb-2">
            ₹{emi.amount.toLocaleString('en-IN')}
          </p>

          {emi.status === 'pending' && emi.payment_id && (
            <button
              onClick={() => handlePayNow({ id: emi.payment_id } as Payment)}
              className="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors text-sm font-medium"
            >
              Pay EMI
            </button>
          )}
          {emi.status === 'paid' && emi.payment_id && (
            <button
              onClick={() => downloadReceipt(emi.payment_id!)}
              className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors text-sm font-medium flex items-center gap-2"
            >
              <Download className="w-4 h-4" />
              Receipt
            </button>
          )}
        </div>
      </div>
    </div>
  );

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-semibold text-gray-900">Payments & Transactions</h3>
      </div>

      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-gradient-to-br from-yellow-50 to-yellow-100 rounded-lg shadow-sm border border-yellow-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-yellow-700 font-medium">Pending</p>
              <p className="text-2xl font-bold text-yellow-900 mt-1">{pendingPayments.length}</p>
            </div>
            <Clock className="w-8 h-8 text-yellow-600" />
          </div>
        </div>

        <div className="bg-gradient-to-br from-green-50 to-green-100 rounded-lg shadow-sm border border-green-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-green-700 font-medium">Completed</p>
              <p className="text-2xl font-bold text-green-900 mt-1">{completedPayments.length}</p>
            </div>
            <CheckCircle className="w-8 h-8 text-green-600" />
          </div>
        </div>

        <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg shadow-sm border border-blue-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-blue-700 font-medium">Refunds</p>
              <p className="text-2xl font-bold text-blue-900 mt-1">{refundedPayments.length}</p>
            </div>
            <RefreshCw className="w-8 h-8 text-blue-600" />
          </div>
        </div>

        <div className="bg-gradient-to-br from-indigo-50 to-indigo-100 rounded-lg shadow-sm border border-indigo-200 p-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-indigo-700 font-medium">EMI Active</p>
              <p className="text-2xl font-bold text-indigo-900 mt-1">{emiSchedules.filter(e => e.status === 'pending').length}</p>
            </div>
            <Wallet className="w-8 h-8 text-indigo-600" />
          </div>
        </div>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6" aria-label="Tabs">
            <button
              onClick={() => setActiveTab('property_purchase')}
              className={`py-4 px-1 border-b-2 font-medium text-sm flex items-center gap-2 ${activeTab === 'property_purchase'
                ? 'border-blue-500 text-blue-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
            >
              <Home className="w-4 h-4" />
              Property Purchase
            </button>
            <button
              onClick={() => setActiveTab('multi_channel')}
              className={`py-4 px-1 border-b-2 font-medium text-sm flex items-center gap-2 ${activeTab === 'multi_channel'
                ? 'border-blue-500 text-blue-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
            >
              <CreditCard className="w-4 h-4" />
              Multi-Channel Payment
            </button>
            <button
              onClick={() => setActiveTab('pending')}
              className={`py-4 px-1 border-b-2 font-medium text-sm flex items-center gap-2 ${activeTab === 'pending'
                ? 'border-blue-500 text-blue-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
            >
              <Clock className="w-4 h-4" />
              Pending Payments
            </button>
            <button
              onClick={() => setActiveTab('completed')}
              className={`py-4 px-1 border-b-2 font-medium text-sm flex items-center gap-2 ${activeTab === 'completed'
                ? 'border-blue-500 text-blue-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
            >
              <CheckCircle className="w-4 h-4" />
              Completed Payments
            </button>
            <button
              onClick={() => setActiveTab('refunds')}
              className={`py-4 px-1 border-b-2 font-medium text-sm flex items-center gap-2 ${activeTab === 'refunds'
                ? 'border-blue-500 text-blue-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
            >
              <RefreshCw className="w-4 h-4" />
              Refunds
            </button>
            <button
              onClick={() => setActiveTab('emi')}
              className={`py-4 px-1 border-b-2 font-medium text-sm flex items-center gap-2 ${activeTab === 'emi'
                ? 'border-blue-500 text-blue-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
            >
              <Wallet className="w-4 h-4" />
              EMI Tracking
            </button>
          </nav>
        </div>

        {/* Tab Content */}
        <div className="p-6">
          {/* Property Purchase Payments */}
          {activeTab === 'property_purchase' && (
            <PropertyPurchasePayments />
          )}

          {/* Multi-Channel Payment */}
          {activeTab === 'multi_channel' && (
            <MultiChannelPayment />
          )}

          {/* Pending Payments */}
          {activeTab === 'pending' && (
            <div className="space-y-4">
              {loading ? (
                <div className="text-center py-12">
                  <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
                  <p className="text-gray-500 mt-4">Loading pending payments...</p>
                </div>
              ) : pendingPayments.length === 0 ? (
                <div className="text-center py-12">
                  <Clock className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                  <p className="text-gray-500">No pending payments</p>
                </div>
              ) : (
                pendingPayments.map(payment => renderPaymentCard(payment))
              )}
            </div>
          )}

          {/* Completed Payments */}
          {activeTab === 'completed' && (
            <div className="space-y-4">
              {loading ? (
                <div className="text-center py-12">
                  <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
                  <p className="text-gray-500 mt-4">Loading completed payments...</p>
                </div>
              ) : completedPayments.length === 0 ? (
                <div className="text-center py-12">
                  <CheckCircle className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                  <p className="text-gray-500">No completed payments</p>
                </div>
              ) : (
                completedPayments.map(payment => renderPaymentCard(payment))
              )}
            </div>
          )}

          {/* Refunds */}
          {activeTab === 'refunds' && (
            <div className="space-y-4">
              {loading ? (
                <div className="text-center py-12">
                  <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
                  <p className="text-gray-500 mt-4">Loading refunds...</p>
                </div>
              ) : refundedPayments.length === 0 ? (
                <div className="text-center py-12">
                  <RefreshCw className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                  <p className="text-gray-500">No refunds processed</p>
                </div>
              ) : (
                refundedPayments.map(payment => renderRefundCard(payment))
              )}
            </div>
          )}

          {/* EMI Tracking */}
          {activeTab === 'emi' && (
            <div className="space-y-4">
              {loading ? (
                <div className="text-center py-12">
                  <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
                  <p className="text-gray-500 mt-4">Loading EMI schedules...</p>
                </div>
              ) : emiSchedules.length === 0 ? (
                <div className="text-center py-12">
                  <Wallet className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                  <p className="text-gray-500">No EMI schedules found</p>
                  <p className="text-sm text-gray-400 mt-1">EMI payments will appear here once you opt for installment plans</p>
                </div>
              ) : (
                <>
                  <div className="bg-indigo-50 border border-indigo-200 rounded-lg p-4 mb-4">
                    <div className="flex items-start gap-3">
                      <DollarSign className="w-5 h-5 text-indigo-600 mt-0.5" />
                      <div>
                        <h3 className="font-semibold text-indigo-900">EMI Payment Schedule</h3>
                        <p className="text-sm text-indigo-700 mt-1">
                          Track your installment payments and upcoming dues
                        </p>
                      </div>
                    </div>
                  </div>
                  {emiSchedules.map(emi => renderEMICard(emi))}
                </>
              )}
            </div>
          )}
        </div>
      </div>
    </div>
  );
}
