import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Button } from '../ui/button';
import { Badge } from '../ui/badge';
import { Plus, Trash2, Target, Shield, TrendingUp } from 'lucide-react';
import './number-input.css';

interface PropertyInput {
  id: string;
  location: string;
  price: number;
  bedrooms: number;
  bathrooms: number;
  area: number;
  propertyType: string;
  age: number;
}

interface PropertyAnalysis {
  property: PropertyInput;
  mlPrediction: {
    predictedPrice: number;
    confidence: number;
    marketPosition: 'underpriced' | 'fair' | 'overpriced';
    psf: number;
  };
  safetyScore: number;
  recommendation: 'SAFE_TO_BUY' | 'NOT_SAFE' | 'WAIT' | 'OVERPRICED';
  roi: number;
  riskLevel: 'low' | 'medium' | 'high';
  marketTrend: string;
  reasons: string[];
  risks: string[];
  predictiveAnalysis?: any;
  whatIfAnalysis?: any;
  personalizedScore?: number;
}

const ProfessionalPropertyComparison: React.FC = () => {
  const [properties, setProperties] = useState<PropertyInput[]>([
    { id: '1', location: '', price: 0, bedrooms: 0, bathrooms: 0, area: 0, propertyType: 'APARTMENT', age: 0 }
  ]);
  const [analysis, setAnalysis] = useState<PropertyAnalysis[]>([]);
  const [loading, setLoading] = useState(false);
  const [scenarioAnalysis, setScenarioAnalysis] = useState<any>(null);
  const [userPreferences, setUserPreferences] = useState({
    safetyWeight: 40,
    roiWeight: 30,
    marketWeight: 20,
    riskWeight: 10,
    investmentPurpose: 'self-use',
    riskAppetite: 'moderate',
    timeHorizon: 3
  });

  const addProperty = () => {
    if (properties.length < 5) {
      setProperties([...properties, {
        id: Date.now().toString(),
        location: '',
        price: 0,
        bedrooms: 0,
        bathrooms: 0,
        area: 0,
        propertyType: 'APARTMENT',
        age: 0
      }]);
    }
  };

  const removeProperty = (id: string) => {
    setProperties(properties.filter(p => p.id !== id));
  };

  const updateProperty = (id: string, field: keyof PropertyInput, value: any) => {
    setProperties(properties.map(p => 
      p.id === id ? { ...p, [field]: value } : p
    ));
  };

  const getLocationData = (location: string) => {
    const locationDB: { [key: string]: any } = {
      'kormangala': { multiplier: 1.2, growth: 15, risk: 0.2, connectivity: 9, amenities: 8 },
      'indiranagar': { multiplier: 1.5, growth: 12, risk: 0.1, connectivity: 10, amenities: 9 },
      'yelahanka': { multiplier: 0.8, growth: 25, risk: 0.3, connectivity: 6, amenities: 6 },
      'tc palya': { multiplier: 0.9, growth: 18, risk: 0.25, connectivity: 7, amenities: 7 },
      'krpuram': { multiplier: 0.85, growth: 20, risk: 0.3, connectivity: 6, amenities: 6 },
      'whitefield': { multiplier: 1.1, growth: 22, risk: 0.2, connectivity: 8, amenities: 8 },
      'electronic city': { multiplier: 1.0, growth: 20, risk: 0.25, connectivity: 7, amenities: 7 },
      'hsr layout': { multiplier: 1.3, growth: 14, risk: 0.15, connectivity: 9, amenities: 9 }
    };
    
    const key = location.toLowerCase().trim();
    return locationDB[key] || { multiplier: 1.0, growth: 10, risk: 0.4, connectivity: 5, amenities: 5 };
  };

  const calculateDynamicPrice = (property: PropertyInput, basePrice: number) => {
    const locationData = getLocationData(property.location);
    const areaMultiplier = property.area > 1500 ? 1.1 : property.area < 800 ? 0.9 : 1.0;
    const ageDepreciation = Math.max(0.7, 1 - (property.age * 0.02));
    const typeMultiplier = property.propertyType === 'VILLA' ? 1.3 : property.propertyType === 'HOUSE' ? 1.1 : 1.0;
    
    return Math.floor(basePrice * locationData.multiplier * areaMultiplier * ageDepreciation * typeMultiplier);
  };

  const calculateSafetyScore = (property: PropertyInput, confidence: number) => {
    const locationData = getLocationData(property.location);
    let safetyScore = 50;
    
    safetyScore += (locationData.connectivity - 5) * 4;
    safetyScore += (locationData.amenities - 5) * 3;
    safetyScore -= locationData.risk * 100;
    
    if (property.age < 5) safetyScore += 10;
    else if (property.age > 20) safetyScore -= 15;
    
    if (property.area > 1200) safetyScore += 5;
    if (property.propertyType === 'APARTMENT') safetyScore += 5;
    
    safetyScore += (confidence - 0.5) * 40;
    
    return Math.max(20, Math.min(100, Math.round(safetyScore)));
  };

  const calculateROI = (property: PropertyInput, predictedPrice: number) => {
    const locationData = getLocationData(property.location);
    const priceDiff = predictedPrice - property.price;
    const baseROI = (priceDiff / property.price) * 100;
    // Cap ROI at reasonable levels to avoid extreme values
    const cappedROI = Math.min(Math.max(baseROI, -50), 200);
    const growthROI = locationData.growth;
    
    return Math.round(cappedROI * 0.7 + growthROI);
  };

  const getMarketPosition = (property: PropertyInput, predictedPrice: number): 'underpriced' | 'fair' | 'overpriced' => {
    const ratio = property.price / predictedPrice;
    if (ratio < 0.85) return 'underpriced';
    if (ratio > 1.15) return 'overpriced';
    return 'fair';
  };

  const getRecommendation = (safetyScore: number, roi: number, marketPosition: string, riskLevel: 'low' | 'medium' | 'high'): 'SAFE_TO_BUY' | 'NOT_SAFE' | 'WAIT' | 'OVERPRICED' => {
    if (marketPosition === 'overpriced') return 'OVERPRICED';
    if (safetyScore < 40 || riskLevel === 'high') return 'NOT_SAFE';
    if (safetyScore >= 75 && roi >= 15 && marketPosition === 'underpriced') return 'SAFE_TO_BUY';
    if (safetyScore >= 60 && roi >= 10) return 'SAFE_TO_BUY';
    return 'WAIT';
  };

  const getLocationInsights = (location: string, marketPosition: string, roi: number) => {
    const insights: { [key: string]: any } = {
      'kormangala': {
        reasons: ['Emerging IT corridor', 'Metro connectivity planned', 'Good rental demand'],
        risks: ['Traffic congestion during peak hours', 'Limited parking space']
      },
      'indiranagar': {
        reasons: ['Premium established location', 'Excellent connectivity', 'High resale value'],
        risks: ['Premium pricing', 'Limited growth potential']
      },
      'yelahanka': {
        reasons: ['Airport proximity advantage', 'Rapid infrastructure development', 'Affordable pricing'],
        risks: ['Distance from city center', 'Developing infrastructure']
      },
      'tc palya': {
        reasons: ['Good residential area', 'Decent school connectivity', 'Moderate pricing'],
        risks: ['Traffic congestion', 'Limited commercial development']
      },
      'krpuram': {
        reasons: ['Emerging locality', 'Good value for money', 'Upcoming developments'],
        risks: ['Limited current amenities', 'Developing area']
      }
    };
    
    const key = location.toLowerCase().trim();
    const locationInsight = insights[key] || {
      reasons: ['Standard residential area', 'Basic amenities available'],
      risks: ['Limited data available', 'Market volatility']
    };
    
    if (marketPosition === 'underpriced') {
      locationInsight.reasons.push('Currently undervalued in market');
    }
    if (roi > 20) {
      locationInsight.reasons.push('High growth potential identified');
    }
    
    return locationInsight;
  };

  const generatePredictiveAnalysis = (property: PropertyInput, currentAnalysis: any) => {
    const locationData = getLocationData(property.location);
    
    const predictROI = (years: number) => {
      const baseGrowth = locationData.growth;
      const marketCycle = 0.85;
      const infrastructureFactor = years > 2 ? 1.2 : 1.0;
      const compoundGrowth = Math.pow(1 + (baseGrowth * marketCycle * infrastructureFactor / 100), years) - 1;
      return Math.round(compoundGrowth * 100);
    };
    
    const predictPrice = (years: number) => {
      const currentPrice = currentAnalysis.mlPrediction.predictedPrice;
      const growthRate = predictROI(years) / 100;
      return Math.floor(currentPrice * (1 + growthRate));
    };
    
    const getMarketSentiment = () => {
      const sentimentFactors: { [key: string]: any } = {
        'kormangala': { trend: 'bullish', confidence: 0.8, drivers: ['IT expansion', 'Metro connectivity'] },
        'indiranagar': { trend: 'stable', confidence: 0.9, drivers: ['Established area', 'Premium location'] },
        'yelahanka': { trend: 'bullish', confidence: 0.7, drivers: ['Airport proximity', 'New developments'] },
        'tc palya': { trend: 'neutral', confidence: 0.6, drivers: ['Residential growth', 'Infrastructure'] },
        'krpuram': { trend: 'bullish', confidence: 0.65, drivers: ['Emerging area', 'Value pricing'] }
      };
      
      const key = property.location.toLowerCase().trim();
      return sentimentFactors[key] || { trend: 'neutral', confidence: 0.5, drivers: ['Standard growth'] };
    };
    
    const generateScenarios = () => {
      const baseROI = currentAnalysis.roi;
      
      return {
        bestCase: {
          roi: Math.round(baseROI * 1.4),
          price: predictPrice(userPreferences.timeHorizon) * 1.2,
          probability: 25,
          conditions: ['Economic boom', 'Infrastructure on time', 'High demand']
        },
        mostLikely: {
          roi: baseROI,
          price: predictPrice(userPreferences.timeHorizon),
          probability: 50,
          conditions: ['Normal market conditions', 'Steady growth', 'Moderate demand']
        },
        worstCase: {
          roi: Math.round(baseROI * 0.6),
          price: predictPrice(userPreferences.timeHorizon) * 0.8,
          probability: 25,
          conditions: ['Economic slowdown', 'Project delays', 'Oversupply']
        }
      };
    };
    
    return {
      timeframePredictions: {
        oneYear: { roi: predictROI(1), price: predictPrice(1) },
        threeYear: { roi: predictROI(3), price: predictPrice(3) },
        fiveYear: { roi: predictROI(5), price: predictPrice(5) }
      },
      marketSentiment: getMarketSentiment(),
      scenarios: generateScenarios(),
      riskFactors: {
        marketRisk: locationData.risk,
        liquidityRisk: property.propertyType === 'VILLA' ? 0.3 : 0.1,
        regulatoryRisk: 0.15,
        overallRisk: Math.round((locationData.risk + 0.15) * 100)
      }
    };
  };
  
  const generateWhatIfAnalysis = (property: PropertyInput, currentAnalysis: any) => {
    const basePrice = property.price;
    const predictedPrice = currentAnalysis.mlPrediction.predictedPrice;
    
    const priceNegotiationImpact = (discountPercent: number) => {
      const newPrice = basePrice * (1 - discountPercent / 100);
      const newROI = Math.round(((predictedPrice - newPrice) / newPrice) * 100);
      // Cap ROI at reasonable levels
      const cappedROI = Math.min(newROI, 200);
      const newRecommendation = cappedROI > 50 ? 'STRONG_BUY' : cappedROI > 25 ? 'BUY' : 'WAIT';
      
      return {
        newPrice: newPrice,
        newROI: cappedROI,
        roiImprovement: cappedROI - Math.min(currentAnalysis.roi, 200),
        recommendation: newRecommendation,
        savings: basePrice - newPrice
      };
    };
    
    const downPaymentAnalysis = (downPaymentPercent: number) => {
      const downPayment = basePrice * (downPaymentPercent / 100);
      const loanAmount = basePrice - downPayment;
      const emi = Math.round((loanAmount * 0.009) / (1 - Math.pow(1.009, -240)));
      const totalInterest = (emi * 240) - loanAmount;
      const totalCost = downPayment + totalInterest;
      
      return {
        downPayment: downPayment,
        loanAmount: loanAmount,
        emi: emi,
        totalInterest: totalInterest,
        totalCost: totalCost,
        effectiveROI: Math.round(((predictedPrice - totalCost) / totalCost) * 100)
      };
    };
    
    const purposeOptimization = () => {
      // Use more realistic rental yield calculation
      const rentalYield = Math.round((basePrice * 0.02) / 12); // 2% annual yield
      const monthlyEMI = Math.round((basePrice * 0.8 * 0.009) / (1 - Math.pow(1.009, -240)));
      
      return {
        selfUse: {
          benefit: 'No rent payment',
          monthlySaving: Math.max(rentalYield, 5000), // Minimum ₹5000
          taxBenefit: Math.round(basePrice * 0.015),
          recommendation: currentAnalysis.safetyScore > 60 ? 'Good for self-use' : 'Consider alternatives'
        },
        rental: {
          monthlyIncome: rentalYield,
          netCashFlow: rentalYield - monthlyEMI,
          breakEvenTime: Math.round(basePrice / (rentalYield * 12)),
          recommendation: rentalYield > monthlyEMI ? 'Positive cash flow' : 'Negative cash flow'
        },
        flipping: {
          minAppreciation: 15,
          timeToBreakeven: Math.max(1, Math.round(15 / Math.max(currentAnalysis.roi / userPreferences.timeHorizon, 5))),
          exitCosts: Math.round(basePrice * 0.08),
          recommendation: currentAnalysis.roi > 25 ? 'Good for flipping' : 'Hold longer'
        }
      };
    };
    
    return {
      priceNegotiation: {
        fivePercent: priceNegotiationImpact(5),
        tenPercent: priceNegotiationImpact(10),
        fifteenPercent: priceNegotiationImpact(15)
      },
      downPaymentScenarios: {
        twenty: downPaymentAnalysis(20),
        thirty: downPaymentAnalysis(30),
        forty: downPaymentAnalysis(40)
      },
      purposeAnalysis: purposeOptimization()
    };
  };
  
  const calculatePersonalizedScore = (analysis: any) => {
    const { safetyWeight, roiWeight, marketWeight, riskWeight } = userPreferences;
    
    const safetyScore = analysis.safetyScore;
    const roiScore = Math.min(analysis.roi, 50);
    const marketScore = analysis.mlPrediction.marketPosition === 'underpriced' ? 100 : 
                       analysis.mlPrediction.marketPosition === 'fair' ? 70 : 30;
    const riskScore = analysis.riskLevel === 'low' ? 100 : 
                     analysis.riskLevel === 'medium' ? 60 : 20;
    
    return (safetyScore * safetyWeight/100) + 
           (roiScore * roiWeight/100) + 
           (marketScore * marketWeight/100) + 
           (riskScore * riskWeight/100);
  };

  const enhancePropertyAnalysis = (property: PropertyInput, apiResult: any) => {
    const basePredictedPrice = apiResult.predicted_price || property.price * 1.1;
    const dynamicPrice = calculateDynamicPrice(property, basePredictedPrice);
    const confidence = apiResult.confidence_score || 0.75;
    const safetyScore = calculateSafetyScore(property, confidence);
    const roi = calculateROI(property, dynamicPrice);
    const marketPosition = getMarketPosition(property, dynamicPrice);
    const riskLevel: 'low' | 'medium' | 'high' = safetyScore >= 75 ? 'low' : safetyScore >= 50 ? 'medium' : 'high';
    const recommendation = getRecommendation(safetyScore, roi, marketPosition, riskLevel);
    const insights = getLocationInsights(property.location, marketPosition, roi);
    
    return {
      property: property,
      mlPrediction: {
        predictedPrice: dynamicPrice,
        confidence: Math.round(confidence * 100),
        marketPosition: marketPosition,
        psf: Math.floor(dynamicPrice / property.area)
      },
      safetyScore: safetyScore,
      recommendation: recommendation,
      roi: roi,
      riskLevel: riskLevel,
      marketTrend: 'stable',
      reasons: insights.reasons,
      risks: insights.risks
    };
  };

  const generateFallbackAnalysis = (property: PropertyInput) => {
    const locationData = getLocationData(property.location);
    const estimatedPrice = property.price * locationData.multiplier * 1.2;
    const safetyScore = calculateSafetyScore(property, 0.65);
    const roi = calculateROI(property, estimatedPrice);
    const marketPosition = getMarketPosition(property, estimatedPrice);
    const riskLevel: 'low' | 'medium' | 'high' = safetyScore >= 75 ? 'low' : safetyScore >= 50 ? 'medium' : 'high';
    const recommendation = getRecommendation(safetyScore, roi, marketPosition, riskLevel);
    const insights = getLocationInsights(property.location, marketPosition, roi);
    
    return {
      property: property,
      mlPrediction: {
        predictedPrice: Math.floor(estimatedPrice),
        confidence: 65,
        marketPosition: marketPosition,
        psf: Math.floor(estimatedPrice / property.area)
      },
      safetyScore: safetyScore,
      recommendation: recommendation,
      roi: roi,
      riskLevel: riskLevel,
      marketTrend: 'stable',
      reasons: [...insights.reasons, 'Estimated analysis - API unavailable'],
      risks: [...insights.risks, 'Limited real-time data']
    };
  };

  const analyzeProperties = async () => {
    const validProperties = properties.filter(p => p.location && p.price > 0);
    if (!validProperties.length) {
      alert('Please enter at least one property with location and price');
      return;
    }
    
    setLoading(true);
    try {
      const analysisPromises = validProperties.map(async (property) => {
        try {
          const controller = new AbortController();
          setTimeout(() => controller.abort(), 10000);
          
          const response = await fetch('http://localhost:8090/api/price-prediction/predict', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
              property_type: property.propertyType,
              listing_type: 'SELL',
              bedrooms: property.bedrooms,
              bathrooms: property.bathrooms,
              built_up_area: property.area,
              locality: property.location,
              age_years: property.age,
              prediction_year: 2025
            }),
            signal: controller.signal
          });
          
          if (!response.ok) throw new Error(`API Error: ${response.status}`);
          const apiResult = await response.json();
          
          console.log('API Response for', property.location, ':', apiResult);
          
          const enhancedAnalysis = enhancePropertyAnalysis(property, apiResult);
          return enhancedAnalysis;
        } catch (error) {
          console.log('API Error for', property.location, ':', error);
          
          const fallbackAnalysis = generateFallbackAnalysis(property);
          return fallbackAnalysis;
        }
      });

      const results = await Promise.all(analysisPromises);
      
      const enhancedResults = results.map(result => ({
        ...result,
        predictiveAnalysis: generatePredictiveAnalysis(result.property, result),
        whatIfAnalysis: generateWhatIfAnalysis(result.property, result),
        personalizedScore: calculatePersonalizedScore(result)
      }));
      
      setAnalysis(enhancedResults);
      
      if (enhancedResults.length > 1) {
        setScenarioAnalysis({
          bestProperty: enhancedResults.reduce((best, current) => 
            calculatePersonalizedScore(current) > calculatePersonalizedScore(best) ? current : best
          ),
          portfolioRisk: Math.round(enhancedResults.reduce((sum, r) => sum + r.predictiveAnalysis.riskFactors.overallRisk, 0) / enhancedResults.length),
          diversificationScore: enhancedResults.length > 2 ? 85 : enhancedResults.length > 1 ? 65 : 30
        });
      }
    } catch (error) {
      alert(`Analysis failed: ${(error as Error).message}`);
    } finally {
      setLoading(false);
    }
  };

  const getRecommendationColor = (rec: string) => ({
    'SAFE_TO_BUY': 'bg-green-100 text-green-800',
    'NOT_SAFE': 'bg-red-100 text-red-800',
    'OVERPRICED': 'bg-orange-100 text-orange-800',
    'WAIT': 'bg-yellow-100 text-yellow-800'
  }[rec] || 'bg-gray-100 text-gray-800');

  const formatCurrency = (amount: number) => {
    if (amount >= 10000000) return `₹${(amount / 10000000).toFixed(1)}Cr`;
    if (amount >= 100000) return `₹${(amount / 100000).toFixed(1)}L`;
    return `₹${amount.toLocaleString()}`;
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-50">
      <div className="bg-white shadow-sm border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="py-8">
            <div className="text-center">
              <h1 className="text-4xl font-bold bg-gradient-to-r from-blue-600 to-indigo-600 bg-clip-text text-transparent">
                AI-Powered Property Comparison
              </h1>
              <p className="mt-4 text-lg text-gray-600 max-w-3xl mx-auto">
                Compare up to 5 properties with real-time ML analysis, market trends, and intelligent recommendations
              </p>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <div className="bg-white shadow-xl rounded-3xl border border-gray-100 overflow-hidden mb-12">
          <div className="bg-gradient-to-r from-blue-600 to-indigo-600 px-8 py-6">
            <div className="flex items-center justify-between">
              <div>
                <h2 className="text-2xl font-bold text-black flex items-center">
                  <div className="w-8 h-8 bg-white/20 rounded-lg flex items-center justify-center mr-4">
                    
                  </div>
                  Property Details ({properties.length}/5)
                </h2>
                <p className="text-black mt-2">Enter property information for AI-powered analysis</p>
              </div>
              <button
                onClick={addProperty}
                disabled={properties.length >= 5}
                className="bg-white hover:bg-gray-100 disabled:bg-gray-400 text-blue-600 font-semibold px-6 py-3 rounded-xl transition-all duration-200 flex items-center space-x-2 shadow-lg"
              >
                <Plus className="w-5 h-5" />
                <span>Add Property</span>
              </button>
            </div>
          </div>

          <div className="p-8">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {properties.map((property, index) => (
                <div key={property.id} className="bg-gradient-to-br from-gray-50 to-blue-50 border-2 border-gray-200 rounded-2xl p-6 hover:shadow-lg transition-all duration-300">
                  <div className="flex items-center justify-between mb-4">
                    <h3 className="text-lg font-bold text-gray-900 flex items-center">
                      <div className="w-6 h-6 bg-blue-500 rounded-lg flex items-center justify-center text-white text-sm font-bold mr-3">
                        {index + 1}
                      </div>
                      Property {index + 1}
                    </h3>
                    {properties.length > 1 && (
                      <button 
                        onClick={() => removeProperty(property.id)}
                        className="bg-red-100 hover:bg-red-200 text-red-600 hover:text-red-700 p-2 rounded-lg transition-all shadow-sm"
                      >
                        <Trash2 className="w-4 h-4" />
                      </button>
                    )}
                  </div>

                  <div className="space-y-4">
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <label className="block text-sm font-semibold text-gray-700 mb-2">Location *</label>
                        <input
                          type="text"
                          placeholder="e.g., Indiranagar, Bangalore"
                          value={property.location}
                          onChange={(e) => updateProperty(property.id, 'location', e.target.value)}
                          className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all bg-white"
                          required
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-semibold text-gray-700 mb-2">Price (₹) *</label>
                        <input
                          type="number"
                          min="1"
                          placeholder="12000000"
                          value={property.price || ''}
                          onChange={(e) => updateProperty(property.id, 'price', e.target.value === '' ? 0 : parseInt(e.target.value) || 0)}
                          onWheel={(e) => (e.target as HTMLInputElement).blur()}
                          className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all bg-white"
                          required
                        />
                      </div>
                    </div>

                    <div className="grid grid-cols-3 gap-4">
                      <div>
                        <label className="block text-sm font-semibold text-gray-700 mb-2">BHK</label>
                        <input
                          type="number"
                          min="1"
                          max="10"
                          value={property.bedrooms || ''}
                          onChange={(e) => updateProperty(property.id, 'bedrooms', e.target.value === '' ? 0 : parseInt(e.target.value) || 0)}
                          onWheel={(e) => (e.target as HTMLInputElement).blur()}
                          className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all bg-white"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-semibold text-gray-700 mb-2">Bathrooms</label>
                        <input
                          type="number"
                          min="1"
                          max="10"
                          value={property.bathrooms || ''}
                          onChange={(e) => updateProperty(property.id, 'bathrooms', e.target.value === '' ? 0 : parseInt(e.target.value) || 0)}
                          onWheel={(e) => (e.target as HTMLInputElement).blur()}
                          className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all bg-white"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-semibold text-gray-700 mb-2">Area (sqft)</label>
                        <input
                          type="number"
                          min="100"
                          max="50000"
                          value={property.area || ''}
                          onChange={(e) => updateProperty(property.id, 'area', e.target.value === '' ? 0 : parseInt(e.target.value) || 0)}
                          onWheel={(e) => (e.target as HTMLInputElement).blur()}
                          className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all bg-white"
                        />
                      </div>
                    </div>

                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <label className="block text-sm font-semibold text-gray-700 mb-2">Property Type</label>
                        <select
                          value={property.propertyType}
                          onChange={(e) => updateProperty(property.id, 'propertyType', e.target.value)}
                          className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all bg-white"
                        >
                          <option value="APARTMENT"> Apartment</option>
                          <option value="VILLA">Villa</option>
                          <option value="HOUSE"> House</option>
                          <option value="PLOT"> Plot</option>
                        </select>
                      </div>
                      <div>
                        <label className="block text-sm font-semibold text-gray-700 mb-2">Age (Years)</label>
                        <input
                          type="number"
                          min="0"
                          max="50"
                          value={property.age || ''}
                          onChange={(e) => updateProperty(property.id, 'age', e.target.value === '' ? 0 : parseInt(e.target.value) || 0)}
                          onWheel={(e) => (e.target as HTMLInputElement).blur()}
                          className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all bg-white"
                        />
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>

            <div className="text-center mt-8">
              <button 
                onClick={analyzeProperties}
                disabled={loading || properties.every(p => !p.location || p.price === 0)}
                className="bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 disabled:bg-gray-400 font-bold px-12 py-4 rounded-2xl transition-all duration-300 flex items-center space-x-3 mx-auto shadow-xl hover:shadow-2xl transform hover:scale-105"
              >
                {loading ? (
                  <>
                    <svg className="animate-spin h-6 w-6" viewBox="0 0 24 24">
                      <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none" />
                      <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
                    </svg>
                    <span className="text-lg text-black">Analyzing with AI...</span>
                  </>
                ) : (
                  <>
                    <Target className="w-6 h-6" />
                    <span className="text-lg text-black">Analyze with AI & Market Trends</span>
                  </>
                )}
              </button>
            </div>
          </div>
        </div>

        {analysis.length > 0 && (
          <div className="bg-white shadow-xl rounded-3xl border border-gray-100 overflow-hidden">
            <div className="bg-gradient-to-r from-green-600 to-emerald-600 px-8 py-6">
              <h2 className="text-2xl font-bold text-black flex items-center">
                <div className="w-8 h-8 bg-white/20 rounded-lg flex items-center justify-center mr-4">
                  
                </div>
                AI Analysis Results
                <span className="ml-4 bg-white text-green-600 px-3 py-1 rounded-full text-sm font-semibold">Real-Time ML</span>
              </h2>
              <p className="text-white mt-2">Machine learning powered property analysis and recommendations</p>
            </div>

            <div className="p-8">
              <div className="overflow-x-auto mb-8">
                <table className="w-full border-collapse">
                  <thead>
                    <tr className="bg-gradient-to-r from-gray-50 to-blue-50">
                      <th className="border-2 border-gray-200 p-4 text-left font-bold text-gray-900">Property</th>
                      <th className="border-2 border-gray-200 p-4 text-center font-bold text-gray-900">Your Price</th>
                      <th className="border-2 border-gray-200 p-4 text-center font-bold text-gray-900">ML Prediction</th>
                      <th className="border-2 border-gray-200 p-4 text-center font-bold text-gray-900">Market Position</th>
                      <th className="border-2 border-gray-200 p-4 text-center font-bold text-gray-900">Safety Score</th>
                      <th className="border-2 border-gray-200 p-4 text-center font-bold text-gray-900">ROI</th>
                      <th className="border-2 border-gray-200 p-4 text-center font-bold text-gray-900">AI Recommendation</th>
                    </tr>
                  </thead>
                  <tbody>
                    {analysis.map((result, index) => (
                      <tr key={index} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                        <td className="border-2 border-gray-200 p-4">
                          <div>
                            <div className="font-bold text-gray-900">{result.property.location}</div>
                            <div className="text-sm text-gray-600">
                              {result.property.bedrooms}BHK • {result.property.area} sqft • {result.property.propertyType}
                            </div>
                          </div>
                        </td>
                        <td className="border-2 border-gray-200 p-4 text-center font-bold text-lg">
                          {formatCurrency(result.property.price)}
                        </td>
                        <td className="border-2 border-gray-200 p-4 text-center">
                          <div className="font-bold text-blue-600 text-lg">
                            {formatCurrency(result.mlPrediction.predictedPrice)}
                          </div>
                          <div className="text-xs text-gray-500">
                            {result.mlPrediction.confidence}% ML confidence
                          </div>
                        </td>
                        <td className="border-2 border-gray-200 p-4 text-center">
                          <Badge className={
                            result.mlPrediction.marketPosition === 'underpriced' ? 'bg-green-100 text-green-800 font-bold' :
                            result.mlPrediction.marketPosition === 'fair' ? 'bg-blue-100 text-blue-800 font-bold' :
                            'bg-red-100 text-red-800 font-bold'
                          }>
                            {result.mlPrediction.marketPosition.toUpperCase()}
                          </Badge>
                        </td>
                        <td className="border-2 border-gray-200 p-4 text-center">
                          <Badge className={
                            result.safetyScore >= 85 ? 'bg-green-100 text-green-800 font-bold' :
                            result.safetyScore >= 70 ? 'bg-yellow-100 text-yellow-800 font-bold' :
                            'bg-red-100 text-red-800 font-bold'
                          }>
                            {result.safetyScore}/100
                          </Badge>
                        </td>
                        <td className="border-2 border-gray-200 p-4 text-center font-bold text-green-600 text-lg">
                          +{result.roi}%
                        </td>
                        <td className="border-2 border-gray-200 p-4 text-center">
                          <Badge className={`${getRecommendationColor(result.recommendation)} font-bold`}>
                            {result.recommendation.replace('_', ' ')}
                          </Badge>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>

              <div className="bg-gradient-to-br from-purple-50 to-indigo-50 border-2 border-purple-200 rounded-2xl p-6 mb-8">
                <h3 className="text-xl font-bold mb-4 flex items-center text-purple-800">
                  <span className="mr-3">⚙️</span>
                  Investment Preferences
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                  <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">Safety Priority (%)</label>
                    <input
                      type="range"
                      min="10"
                      max="70"
                      value={userPreferences.safetyWeight}
                      onChange={(e) => setUserPreferences({...userPreferences, safetyWeight: parseInt(e.target.value)})}
                      className="w-full"
                    />
                    <span className="text-sm text-gray-600">{userPreferences.safetyWeight}%</span>
                  </div>
                  <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">ROI Priority (%)</label>
                    <input
                      type="range"
                      min="10"
                      max="70"
                      value={userPreferences.roiWeight}
                      onChange={(e) => setUserPreferences({...userPreferences, roiWeight: parseInt(e.target.value)})}
                      className="w-full"
                    />
                    <span className="text-sm text-gray-600">{userPreferences.roiWeight}%</span>
                  </div>
                  <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">Investment Purpose</label>
                    <select
                      value={userPreferences.investmentPurpose}
                      onChange={(e) => setUserPreferences({...userPreferences, investmentPurpose: e.target.value})}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg"
                    >
                      <option value="self-use">Self Use</option>
                      <option value="rental">Rental Income</option>
                      <option value="flipping">Flipping</option>
                    </select>
                  </div>
                  <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">Time Horizon</label>
                    <select
                      value={userPreferences.timeHorizon}
                      onChange={(e) => setUserPreferences({...userPreferences, timeHorizon: parseInt(e.target.value)})}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg"
                    >
                      <option value={1}>1 Year</option>
                      <option value={3}>3 Years</option>
                      <option value={5}>5 Years</option>
                    </select>
                  </div>
                </div>
              </div>

              {analysis.length > 0 && analysis[0].predictiveAnalysis && (
                <div className="bg-gradient-to-br from-blue-50 to-cyan-50 border-2 border-blue-200 rounded-2xl p-6 mb-8">
                  <h3 className="text-xl font-bold mb-4 flex items-center text-blue-800">
                    <span className="mr-3">🔮</span>
                    Predictive Analytics & Scenarios
                  </h3>
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    {analysis.map((result, index) => (
                      <div key={index} className="bg-white rounded-xl p-4 border border-blue-200">
                        <h4 className="font-bold text-gray-900 mb-3">{result.property.location}</h4>
                        
                        <div className="mb-4">
                          <h5 className="font-semibold text-gray-700 mb-2">📈 Future Projections</h5>
                          <div className="grid grid-cols-3 gap-2 text-sm">
                            <div className="text-center p-2 bg-green-50 rounded">
                              <div className="font-bold text-green-700">1 Year</div>
                              <div>+{result.predictiveAnalysis.timeframePredictions.oneYear.roi}%</div>
                              <div className="text-xs">{formatCurrency(result.predictiveAnalysis.timeframePredictions.oneYear.price)}</div>
                            </div>
                            <div className="text-center p-2 bg-blue-50 rounded">
                              <div className="font-bold text-blue-700">3 Years</div>
                              <div>+{result.predictiveAnalysis.timeframePredictions.threeYear.roi}%</div>
                              <div className="text-xs">{formatCurrency(result.predictiveAnalysis.timeframePredictions.threeYear.price)}</div>
                            </div>
                            <div className="text-center p-2 bg-purple-50 rounded">
                              <div className="font-bold text-purple-700">5 Years</div>
                              <div>+{result.predictiveAnalysis.timeframePredictions.fiveYear.roi}%</div>
                              <div className="text-xs">{formatCurrency(result.predictiveAnalysis.timeframePredictions.fiveYear.price)}</div>
                            </div>
                          </div>
                        </div>
                        
                        <div className="mb-4">
                          <h5 className="font-semibold text-gray-700 mb-2">📊 Market Sentiment</h5>
                          <div className="flex items-center justify-between">
                            <span className={`px-3 py-1 rounded-full text-sm font-bold ${
                              result.predictiveAnalysis.marketSentiment.trend === 'bullish' ? 'bg-green-100 text-green-800' :
                              result.predictiveAnalysis.marketSentiment.trend === 'stable' ? 'bg-blue-100 text-blue-800' :
                              'bg-gray-100 text-gray-800'
                            }`}>
                              {result.predictiveAnalysis.marketSentiment.trend.toUpperCase()}
                            </span>
                            <span className="text-sm text-gray-600">
                              {Math.round(result.predictiveAnalysis.marketSentiment.confidence * 100)}% confidence
                            </span>
                          </div>
                        </div>
                        
                        <div>
                          <h5 className="font-semibold text-gray-700 mb-2">🎯 Scenario Analysis</h5>
                          <div className="space-y-2 text-sm">
                            <div className="flex justify-between items-center p-2 bg-green-50 rounded">
                              <span className="font-medium text-green-700">Best Case ({result.predictiveAnalysis.scenarios.bestCase.probability}%)</span>
                              <span className="font-bold text-green-800">+{result.predictiveAnalysis.scenarios.bestCase.roi}%</span>
                            </div>
                            <div className="flex justify-between items-center p-2 bg-blue-50 rounded">
                              <span className="font-medium text-blue-700">Most Likely ({result.predictiveAnalysis.scenarios.mostLikely.probability}%)</span>
                              <span className="font-bold text-blue-800">+{result.predictiveAnalysis.scenarios.mostLikely.roi}%</span>
                            </div>
                            <div className="flex justify-between items-center p-2 bg-red-50 rounded">
                              <span className="font-medium text-red-700">Worst Case ({result.predictiveAnalysis.scenarios.worstCase.probability}%)</span>
                              <span className="font-bold text-red-800">+{result.predictiveAnalysis.scenarios.worstCase.roi}%</span>
                            </div>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
              
              {analysis.length > 0 && analysis[0].whatIfAnalysis && (
                <div className="bg-gradient-to-br from-orange-50 to-yellow-50 border-2 border-orange-200 rounded-2xl p-6 mb-8">
                  <h3 className="text-xl font-bold mb-4 flex items-center text-orange-800">
                    <span className="mr-3">🤔</span>
                    What-If Analysis & Decision Support
                  </h3>
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    {analysis.slice(0, 2).map((result, index) => (
                      <div key={index} className="bg-white rounded-xl p-4 border border-orange-200">
                        <h4 className="font-bold text-gray-900 mb-3">{result.property.location}</h4>
                        
                        <div className="mb-4">
                          <h5 className="font-semibold text-gray-700 mb-2">💰 Price Negotiation Impact</h5>
                          <div className="space-y-2 text-sm">
                            <div className="flex justify-between items-center p-2 bg-green-50 rounded">
                              <span>5% discount</span>
                              <span className="font-bold text-green-600">+{result.whatIfAnalysis.priceNegotiation.fivePercent.newROI}% ROI</span>
                            </div>
                            <div className="flex justify-between items-center p-2 bg-blue-50 rounded">
                              <span>10% discount</span>
                              <span className="font-bold text-blue-600">+{result.whatIfAnalysis.priceNegotiation.tenPercent.newROI}% ROI</span>
                            </div>
                            <div className="flex justify-between items-center p-2 bg-purple-50 rounded">
                              <span>15% discount</span>
                              <span className="font-bold text-purple-600">+{result.whatIfAnalysis.priceNegotiation.fifteenPercent.newROI}% ROI</span>
                            </div>
                          </div>
                        </div>
                        
                        <div>
                          <h5 className="font-semibold text-gray-700 mb-2">🏠 Purpose-Based Analysis</h5>
                          <div className="space-y-2 text-sm">
                            <div className="p-2 bg-blue-50 rounded">
                              <div className="font-medium text-blue-700">Self Use</div>
                              <div className="text-gray-600">Monthly saving: {formatCurrency(result.whatIfAnalysis.purposeAnalysis.selfUse.monthlySaving)}</div>
                            </div>
                            <div className="p-2 bg-green-50 rounded">
                              <div className="font-medium text-green-700">Rental Income</div>
                              <div className="text-gray-600">Net cash flow: {formatCurrency(result.whatIfAnalysis.purposeAnalysis.rental.netCashFlow)}/month</div>
                            </div>
                            <div className="p-2 bg-orange-50 rounded">
                              <div className="font-medium text-orange-700">Flipping</div>
                              <div className="text-gray-600">Break-even: {result.whatIfAnalysis.purposeAnalysis.flipping.timeToBreakeven} years</div>
                            </div>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              <div className="bg-gradient-to-br from-green-50 to-emerald-50 border-2 border-green-200 rounded-2xl p-8">
                <h3 className="text-2xl font-bold mb-4 flex items-center text-green-800">
                  <Target className="w-8 h-8 mr-3" />
                  AI Investment Recommendation
                </h3>
                {(() => {
                  const bestProperty = analysis.reduce((best, current) => {
                    const bestScore = calculatePersonalizedScore(best);
                    const currentScore = calculatePersonalizedScore(current);
                    return currentScore > bestScore ? current : best;
                  });
                  
                  const compositeScore = calculatePersonalizedScore(bestProperty);
                  
                  return (
                    <div className="space-y-4">
                      <div className="flex items-center justify-between">
                        <div>
                          <div className="text-xl font-bold text-green-800">
                            🏆 Best Choice: {bestProperty.property.location}
                          </div>
                          <div className="text-gray-600 mt-1">
                            Safety: {bestProperty.safetyScore}/100 • ROI: +{bestProperty.roi}% • 
                            Position: {bestProperty.mlPrediction.marketPosition} • Score: {Math.round(compositeScore)}/100
                          </div>
                        </div>
                        <Badge className="bg-green-100 text-green-800 text-xl px-6 py-3 font-bold">
                          {bestProperty.recommendation.replace('_', ' ')}
                        </Badge>
                      </div>
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
                        <div className="bg-white rounded-xl p-6 border border-green-200">
                          <h4 className="font-bold text-green-700 mb-3 flex items-center">
                            <Shield className="w-5 h-5 mr-2" />
                            Why This is Safe to Buy
                          </h4>
                          <ul className="space-y-2">
                            {bestProperty.reasons.map((reason, idx) => (
                              <li key={idx} className="text-gray-700 flex items-start">
                                <span className="text-green-500 mr-2">✓</span>
                                {reason}
                              </li>
                            ))}
                          </ul>
                        </div>
                        <div className="bg-white rounded-xl p-6 border border-orange-200">
                          <h4 className="font-bold text-orange-700 mb-3 flex items-center">
                            <TrendingUp className="w-5 h-5 mr-2" />
                            Consider These Factors
                          </h4>
                          <ul className="space-y-2">
                            {bestProperty.risks.map((risk, idx) => (
                              <li key={idx} className="text-gray-700 flex items-start">
                                <span className="text-orange-500 mr-2">⚠</span>
                                {risk}
                              </li>
                            ))}
                          </ul>
                        </div>
                      </div>
                    </div>
                  );
                })()}
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default ProfessionalPropertyComparison;