import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Button } from '../ui/button';
import { Badge } from '../ui/badge';

interface PropertyInput {
  id: string;
  location: string;
  price: number;
  bedrooms: number;
  bathrooms: number;
  area: number;
  propertyType: string;
  age: number;
}

interface PropertyAnalysis {
  property: PropertyInput;
  mlPrediction: {
    predictedPrice: number;
    confidence: number;
    marketPosition: 'underpriced' | 'fair' | 'overpriced';
    psf: number;
  };
  safetyScore: number;
  recommendation: 'SAFE_TO_BUY' | 'NOT_SAFE' | 'WAIT' | 'OVERPRICED';
  roi: number;
  riskLevel: 'low' | 'medium' | 'high';
  marketTrend: string;
  reasons: string[];
  risks: string[];
}

const PropertyComparison: React.FC = () => {
  const [properties, setProperties] = useState<PropertyInput[]>([
    { id: '1', location: '', price: 0, bedrooms: 2, bathrooms: 2, area: 1000, propertyType: 'APARTMENT', age: 0 }
  ]);
  const [analysis, setAnalysis] = useState<PropertyAnalysis[]>([]);
  const [loading, setLoading] = useState(false);

  const getSafetyColor = (score: number) => 
    score >= 85 ? 'text-green-600 bg-green-50' : 
    score >= 70 ? 'text-yellow-600 bg-yellow-50' : 'text-red-600 bg-red-50';

  const getMarketPositionColor = (pos: string) => ({
    'underpriced': 'bg-green-100 text-green-800',
    'fair': 'bg-blue-100 text-blue-800',
    'overpriced': 'bg-red-100 text-red-800'
  }[pos] || 'bg-gray-100 text-gray-800');

  const getInvestmentGradeColor = (grade: string) => 
    grade.startsWith('A') ? 'bg-green-100 text-green-800' :
    grade.startsWith('B') ? 'bg-yellow-100 text-yellow-800' : 'bg-red-100 text-red-800';

  const formatCurrency = (amount: number) => {
    if (amount >= 10000000) return `₹${(amount / 10000000).toFixed(1)}Cr`;
    if (amount >= 100000) return `₹${(amount / 100000).toFixed(1)}L`;
    return `₹${amount.toLocaleString()}`;
  };

  const [comparisonData, setComparisonData] = useState<any>(null);

  const analyzeProperty = async () => {
    setLoading(true);
    
    try {
      const controller = new AbortController();
      setTimeout(() => controller.abort(), 10000);
      
      const response = await fetch('http://localhost:8090/api/buyer/quick-analyze', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          locality: 'Indiranagar',
          current_price: 12400000,
          bedrooms: 3,
          bathrooms: 2,
          built_up_area: 1500,
          property_type: 'APARTMENT'
        }),
        signal: controller.signal
      });
      
      if (response.ok) {
        const result = await response.json();
        setComparisonData({
          targetProperty: {
            title: 'Sample Property Analysis',
            location: result.property.location,
            price: result.property.price,
            safetyScore: result.safetyScore,
            marketPosition: result.mlPrediction.marketPosition,
            investmentGrade: 'A',
            riskLevel: result.riskLevel,
            roi: result.roi,
            pricePerSqft: Math.floor(result.property.price / result.property.area),
            builderRating: 4.2
          },
          similarProperties: [{
            id: 1,
            title: 'Similar Property 1',
            price: result.property.price * 0.95,
            safetyScore: result.safetyScore - 5,
            marketPosition: 'fair',
            roi: result.roi - 2,
            riskLevel: 'low'
          }],
          recommendation: {
            action: result.recommendation.toLowerCase(),
            confidence: 85,
            reasons: result.reasons,
            risks: result.risks
          }
        });
      }
    } catch (error) {
      alert(error.name === 'AbortError' ? 'Timeout. Check backend on port 8090.' : 'Analysis failed.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="space-y-6 p-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Smart Property Comparison</h1>
          <p className="text-gray-600">AI-powered analysis to help you buy with confidence</p>
        </div>
        <Button onClick={() => window.location.href = '/search'} className="bg-blue-600">
          Find Properties
        </Button>
      </div>

      {/* Quick Trust Indicators */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center space-x-2">
              <div className="w-3 h-3 bg-green-500 rounded-full"></div>
              <span className="text-sm font-medium">Safe Properties</span>
            </div>
            <p className="text-2xl font-bold text-green-600">847</p>
            <p className="text-xs text-gray-500">Safety Score 85+</p>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center space-x-2">
              <div className="w-3 h-3 bg-blue-500 rounded-full"></div>
              <span className="text-sm font-medium">Fair Priced</span>
            </div>
            <p className="text-2xl font-bold text-blue-600">623</p>
            <p className="text-xs text-gray-500">Market Value ±5%</p>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center space-x-2">
              <div className="w-3 h-3 bg-purple-500 rounded-full"></div>
              <span className="text-sm font-medium">High ROI</span>
            </div>
            <p className="text-2xl font-bold text-purple-600">234</p>
            <p className="text-xs text-gray-500">15%+ Expected</p>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center space-x-2">
              <div className="w-3 h-3 bg-orange-500 rounded-full"></div>
              <span className="text-sm font-medium">Verified Sellers</span>
            </div>
            <p className="text-2xl font-bold text-orange-600">1,245</p>
            <p className="text-xs text-gray-500">KYC Completed</p>
          </CardContent>
        </Card>
      </div>

      {/* Property Comparison Matrix */}
      {comparisonData && (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <span>🏠</span>
              <span>Property Analysis Report</span>
              <Badge className={`ml-auto ${getSafetyColor(comparisonData.targetProperty.safetyScore)}`}>
                Safety Score: {comparisonData.targetProperty.safetyScore}/100
              </Badge>
            </CardTitle>
          </CardHeader>
          <CardContent>
            {/* Target Property Summary */}
            <div className="bg-blue-50 p-4 rounded-lg mb-6">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <h3 className="font-semibold text-lg">{comparisonData.targetProperty.title}</h3>
                  <p className="text-gray-600">{comparisonData.targetProperty.location}</p>
                  <p className="text-2xl font-bold text-blue-600">
                    {formatCurrency(comparisonData.targetProperty.price)}
                  </p>
                </div>
                <div className="space-y-2">
                  <div className="flex justify-between">
                    <span>Market Position:</span>
                    <Badge className={getMarketPositionColor(comparisonData.targetProperty.marketPosition)}>
                      {comparisonData.targetProperty.marketPosition.toUpperCase()}
                    </Badge>
                  </div>
                  <div className="flex justify-between">
                    <span>Investment Grade:</span>
                    <Badge className={getInvestmentGradeColor(comparisonData.targetProperty.investmentGrade)}>
                      {comparisonData.targetProperty.investmentGrade}
                    </Badge>
                  </div>
                  <div className="flex justify-between">
                    <span>Risk Level:</span>
                    <Badge className={comparisonData.targetProperty.riskLevel === 'low' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'}>
                      {comparisonData.targetProperty.riskLevel.toUpperCase()}
                    </Badge>
                  </div>
                </div>
                <div className="space-y-2">
                  <div className="flex justify-between">
                    <span>Expected ROI:</span>
                    <span className="font-semibold text-green-600">+{comparisonData.targetProperty.roi}%</span>
                  </div>
                  <div className="flex justify-between">
                    <span>Price/SqFt:</span>
                    <span className="font-semibold">₹{comparisonData.targetProperty.pricePerSqft.toLocaleString()}</span>
                  </div>
                  <div className="flex justify-between">
                    <span>Builder Rating:</span>
                    <span className="font-semibold">{comparisonData.targetProperty.builderRating}/5 ⭐</span>
                  </div>
                </div>
              </div>
            </div>

            {/* Comparison Table */}
            <div className="overflow-x-auto">
              <table className="w-full border-collapse border border-gray-200">
                <thead>
                  <tr className="bg-gray-50">
                    <th className="border border-gray-200 p-3 text-left">Property</th>
                    <th className="border border-gray-200 p-3 text-center">Price</th>
                    <th className="border border-gray-200 p-3 text-center">Safety Score</th>
                    <th className="border border-gray-200 p-3 text-center">Market Position</th>
                    <th className="border border-gray-200 p-3 text-center">ROI</th>
                    <th className="border border-gray-200 p-3 text-center">Risk</th>
                  </tr>
                </thead>
                <tbody>
                  <tr className="bg-blue-25">
                    <td className="border border-gray-200 p-3 font-semibold">
                      {comparisonData.targetProperty.title} (Target)
                    </td>
                    <td className="border border-gray-200 p-3 text-center font-semibold">
                      {formatCurrency(comparisonData.targetProperty.price)}
                    </td>
                    <td className="border border-gray-200 p-3 text-center">
                      <Badge className={getSafetyColor(comparisonData.targetProperty.safetyScore)}>
                        {comparisonData.targetProperty.safetyScore}
                      </Badge>
                    </td>
                    <td className="border border-gray-200 p-3 text-center">
                      <Badge className={getMarketPositionColor(comparisonData.targetProperty.marketPosition)}>
                        {comparisonData.targetProperty.marketPosition}
                      </Badge>
                    </td>
                    <td className="border border-gray-200 p-3 text-center text-green-600 font-semibold">
                      +{comparisonData.targetProperty.roi}%
                    </td>
                    <td className="border border-gray-200 p-3 text-center">
                      <Badge className={comparisonData.targetProperty.riskLevel === 'low' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'}>
                        {comparisonData.targetProperty.riskLevel}
                      </Badge>
                    </td>
                  </tr>
                  {comparisonData.similarProperties.map((property, index) => (
                    <tr key={property.id}>
                      <td className="border border-gray-200 p-3">{property.title}</td>
                      <td className="border border-gray-200 p-3 text-center">{formatCurrency(property.price)}</td>
                      <td className="border border-gray-200 p-3 text-center">
                        <Badge className={getSafetyColor(property.safetyScore)}>
                          {property.safetyScore}
                        </Badge>
                      </td>
                      <td className="border border-gray-200 p-3 text-center">
                        <Badge className={getMarketPositionColor(property.marketPosition)}>
                          {property.marketPosition}
                        </Badge>
                      </td>
                      <td className="border border-gray-200 p-3 text-center text-green-600">
                        +{property.roi}%
                      </td>
                      <td className="border border-gray-200 p-3 text-center">
                        <Badge className={property.riskLevel === 'low' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'}>
                          {property.riskLevel}
                        </Badge>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>

            {/* Investment Recommendation */}
            <div className="mt-6 p-4 bg-gradient-to-r from-green-50 to-blue-50 rounded-lg border border-green-200">
              <div className="flex items-center space-x-2 mb-3">
                <span className="text-2xl">🎯</span>
                <h3 className="text-lg font-semibold">Investment Recommendation</h3>
                <Badge className={`ml-auto ${comparisonData.recommendation.action === 'strong_buy' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'}`}>
                  {comparisonData.recommendation.action.replace('_', ' ').toUpperCase()}
                </Badge>
              </div>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <h4 className="font-semibold text-green-700 mb-2">✅ Positive Factors</h4>
                  <ul className="space-y-1">
                    {comparisonData.recommendation.reasons.map((reason, index) => (
                      <li key={index} className="text-sm text-gray-700">• {reason}</li>
                    ))}
                  </ul>
                </div>
                <div>
                  <h4 className="font-semibold text-orange-700 mb-2">⚠️ Risk Considerations</h4>
                  <ul className="space-y-1">
                    {comparisonData.recommendation.risks.map((risk, index) => (
                      <li key={index} className="text-sm text-gray-700">• {risk}</li>
                    ))}
                  </ul>
                </div>
              </div>
              
              <div className="mt-4 p-3 bg-white rounded border">
                <div className="flex items-center justify-between">
                  <span className="font-semibold">Confidence Level:</span>
                  <span className="text-lg font-bold text-blue-600">{comparisonData.recommendation.confidence}%</span>
                </div>
                <div className="w-full bg-gray-200 rounded-full h-2 mt-2">
                  <div 
                    className="bg-blue-600 h-2 rounded-full transition-all duration-500"
                    style={{ width: `${comparisonData.recommendation.confidence}%` }}
                  ></div>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Action Buttons */}
      <div className="flex space-x-4">
        <Button 
          onClick={analyzeProperty} 
          disabled={loading}
          className="bg-blue-600 hover:bg-blue-700"
        >
          {loading ? 'Analyzing...' : 'Analyze Property'}
        </Button>
        <Button variant="outline">
          Save Analysis
        </Button>
        <Button variant="outline">
          Schedule Visit
        </Button>
        <Button variant="outline">
          Contact Agent
        </Button>
      </div>
    </div>
  );
};

export default PropertyComparison;