import React, { useState, useEffect } from "react";
import { Download, FileText, Calendar, Loader2, Trash2 } from "lucide-react";
import { toast } from 'sonner';
import { propertyGenerateAPI, PropertyDocument } from '../../api/propertyGenerate';

interface TimePeriodModalProps {
  isOpen: boolean;
  onClose: () => void;
  onSelect: (period: string) => void;
}

function TimePeriodModal({ isOpen, onClose, onSelect }: TimePeriodModalProps) {
  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 flex items-center justify-center z-50">
      <div className="bg-white rounded-lg p-6 max-w-md w-full mx-4 shadow-2xl">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Select Time Period</h3>
        <p className="text-gray-600 mb-6">Choose the time period for your property activity report:</p>

        <div className="space-y-3">
          <button
            onClick={() => onSelect('30_days')}
            className="w-full text-left p-4 border border-gray-200 rounded-lg hover:bg-blue-50 hover:border-blue-300 transition-colors"
          >
            <div className="font-medium text-gray-900">Last 30 Days</div>
            <div className="text-sm text-gray-600">Recent activity and current data</div>
          </button>

          <button
            onClick={() => onSelect('60_days')}
            className="w-full text-left p-4 border border-gray-200 rounded-lg hover:bg-blue-50 hover:border-blue-300 transition-colors"
          >
            <div className="font-medium text-gray-900">Last 60 Days</div>
            <div className="text-sm text-gray-600">Extended activity overview</div>
          </button>

          <button
            onClick={() => onSelect('1_year')}
            className="w-full text-left p-4 border border-gray-200 rounded-lg hover:bg-blue-50 hover:border-blue-300 transition-colors"
          >
            <div className="font-medium text-gray-900">Last Year</div>
            <div className="text-sm text-gray-600">Comprehensive annual activity report</div>
          </button>
        </div>

        <div className="flex justify-end mt-6">
          <button
            onClick={onClose}
            className="px-4 py-2 text-gray-600 hover:text-gray-800 transition-colors"
          >
            Cancel
          </button>
        </div>
      </div>
    </div>
  );
}

export function PropertyGenerator() {
  const [documents, setDocuments] = useState<PropertyDocument[]>([]);
  const [loading, setLoading] = useState(false);
  const [generating, setGenerating] = useState(false);
  const [deletingId, setDeletingId] = useState<string | null>(null);
  const [showTimeModal, setShowTimeModal] = useState(false);

  const fetchDocuments = async () => {
    try {
      setLoading(true);
      console.log('Fetching documents...');
      const response = await propertyGenerateAPI.getUserDocuments();
      console.log('Fetched documents:', response);
      setDocuments(response);
    } catch (error) {
      console.error('Error fetching documents:', error);
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      toast.error(`Failed to load documents: ${errorMessage}`);
    } finally {
      setLoading(false);
    }
  };

  const generateDocument = () => {
    setShowTimeModal(true);
  };

  const handleTimePeriodSelect = async (timePeriod: string) => {
    setShowTimeModal(false);
    try {
      setGenerating(true);
      await propertyGenerateAPI.generateDocument(timePeriod);
      toast.success('Property document generated successfully!');
      fetchDocuments(); // Refresh the list
    } catch (error) {
      console.error('Error generating document:', error);
      toast.error('Failed to generate document');
    } finally {
      setGenerating(false);
    }
  };

  const deleteDocument = async (documentId: string) => {
    console.log('Delete button clicked for document:', documentId);

    const confirmed = window.confirm('Are you sure you want to delete this property report? This action cannot be undone.');
    if (!confirmed) {
      console.log('User cancelled delete operation');
      return;
    }

    console.log('User confirmed delete, proceeding...');

    // Set deleting state to show loading
    setDeletingId(documentId);

    try {
      console.log('Calling API to delete document:', documentId);
      const result = await propertyGenerateAPI.deleteDocument(documentId);
      console.log('API delete result:', result);
      console.log('Document deleted successfully from backend');
      toast.success('Document deleted successfully!');

      // Immediately update the local state to remove the document
      console.log('Updating local state to remove document:', documentId);
      setDocuments(prevDocuments => prevDocuments.filter(doc => doc.id !== documentId));
      console.log('Local state updated');

      // Also refresh from server to ensure consistency
      console.log('Refreshing document list from server...');
      await fetchDocuments();
      console.log('Document list refreshed from server');
    } catch (error) {
      console.error('Error deleting document:', error);
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      console.error('Error details:', errorMessage);
      toast.error(`Failed to delete document: ${errorMessage}`);
    } finally {
      // Clear deleting state
      setDeletingId(null);
    }
  };

  const downloadDocument = async (documentId: string) => {
    try {
      const response = await propertyGenerateAPI.downloadDocument(documentId);

      // Decode base64 PDF data
      const pdfData = atob(response.file_data);
      const uint8Array = new Uint8Array(pdfData.length);
      for (let i = 0; i < pdfData.length; i++) {
        uint8Array[i] = pdfData.charCodeAt(i);
      }

      // Create PDF blob
      const blob = new Blob([uint8Array], {
        type: response.file_type
      });

      // Create download link
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = response.file_name;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);

      toast.success('PDF download started!');
    } catch (error) {
      console.error('Error downloading document:', error);
      toast.error('Failed to download document');
    }
  };

  useEffect(() => {
    fetchDocuments();
  }, []);

  return (
    <div className="space-y-6">
      <TimePeriodModal
        isOpen={showTimeModal}
        onClose={() => setShowTimeModal(false)}
        onSelect={handleTimePeriodSelect}
      />

      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">Property Generator</h2>
          <p className="text-gray-600">Generate comprehensive reports of your property activities for selected time periods</p>
        </div>
        <button
          onClick={generateDocument}
          disabled={generating}
          className="flex items-center gap-2 px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0041A3] transition-colors font-medium disabled:opacity-50 disabled:cursor-not-allowed"
        >
          {generating ? (
            <Loader2 className="w-5 h-5 animate-spin" />
          ) : (
            <FileText className="w-5 h-5" />
          )}
          {generating ? 'Generating...' : 'Generate Report'}
        </button>
      </div>

      {/* Description */}
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-blue-900 mb-2">What gets included in your report?</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm text-blue-800">
          <ul className="space-y-1">
            <li>• My Properties</li>
            <li>• Wishlist/Favorites</li>
            <li>• Saved Properties</li>
            <li>• Search History</li>
          </ul>
          <ul className="space-y-1">
            <li>• Scheduled Visits</li>
            <li>• Property Visit Requests</li>
            <li>• Property Comparisons</li>
            <li>• Activity Summary</li>
          </ul>
        </div>
      </div>

      {/* Documents List */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">Generated Reports</h3>
          <p className="text-sm text-gray-600 mt-1">Your previously generated property reports</p>
        </div>

        <div className="p-6">
          {loading ? (
            <div className="flex items-center justify-center py-8">
              <Loader2 className="w-8 h-8 animate-spin text-gray-400" />
            </div>
          ) : documents.length > 0 ? (
            <div className="space-y-4">
              {documents.map((doc) => (
                <div key={doc.id} className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                  <div className="flex items-center gap-4">
                    <div className="w-10 h-10 bg-blue-50 rounded-lg flex items-center justify-center">
                      <FileText className="w-5 h-5 text-blue-600" />
                    </div>
                    <div>
                      <h4 className="font-medium text-gray-900">
                        Property Report - {new Date(doc.generated_at).toLocaleDateString()}
                      </h4>
                      <p className="text-sm text-gray-600">
                        Generated on {new Date(doc.generated_at).toLocaleString()}
                      </p>
                    </div>
                  </div>
                  <div className="flex items-center gap-2">
                    <button
                      onClick={() => downloadDocument(doc.id)}
                      className="flex items-center gap-2 px-4 py-2 bg-blue-100 text-blue-700 rounded-lg hover:bg-blue-200 transition-colors"
                    >
                      <Download className="w-4 h-4" />
                      Download
                    </button>
                    <button
                      onClick={async (e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        console.log('Delete button clicked for document:', doc.id);
                        try {
                          await deleteDocument(doc.id);
                        } catch (error) {
                          console.error('Delete button error:', error);
                        }
                      }}
                      className={`flex items-center gap-2 px-4 py-2 rounded-lg transition-colors ${
                        deletingId === doc.id
                          ? 'bg-gray-100 text-gray-500 cursor-not-allowed'
                          : 'bg-red-100 text-red-700 hover:bg-red-200'
                      }`}
                      type="button"
                      disabled={deletingId === doc.id}
                    >
                      {deletingId === doc.id ? (
                        <Loader2 className="w-4 h-4 animate-spin" />
                      ) : (
                        <Trash2 className="w-4 h-4" />
                      )}
                      {deletingId === doc.id ? 'Deleting...' : 'Delete'}
                    </button>
                  </div>
                </div>
              ))}
            </div>
          ) : (
            <div className="text-center py-8">
              <FileText className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h4 className="text-lg font-medium text-gray-900 mb-2">No reports generated yet</h4>
              <p className="text-gray-600 mb-4">Click "Generate Report" to create your first property activity report for any time period.</p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
}