import React, { useState, useEffect } from 'react';
import {
  CreditCard, Download, Calendar, CheckCircle, Clock,
  XCircle, FileText, AlertCircle, Upload, Eye, DollarSign,
  TrendingUp, Receipt, Shield, ChevronRight, Info
} from 'lucide-react';

interface PaymentBreakdown {
  base_amount: number;
  gst_amount: number;
  tds_amount: number;
  stamp_duty: number;
  registration_fee: number;
  other_charges: number;
  total_amount: number;
}

interface Payment {
  id: string;
  order_id: string;
  transaction_id?: string;
  property_id?: string;
  amount: number;
  breakdown: PaymentBreakdown;
  payment_type: string;
  payment_method?: string;
  payment_gateway?: string;
  status: string;
  failure_reason?: string;
  milestone_number?: number;
  total_milestones?: number;
  due_date?: string;
  paid_at?: string;
  created_at: string;
}

export function PropertyPurchasePayments() {
  const [activeTab, setActiveTab] = useState<'pending' | 'completed' | 'milestones'>('pending');
  const [payments, setPayments] = useState<Payment[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedPayment, setSelectedPayment] = useState<Payment | null>(null);
  const [showPaymentModal, setShowPaymentModal] = useState(false);
  const [selectedGateway, setSelectedGateway] = useState<'razorpay' | 'cashfree' | 'paytm'>('razorpay');

  useEffect(() => {
    fetchPayments();
  }, [activeTab]);

  const fetchPayments = async () => {
    try {
      setLoading(true);
      const status = activeTab === 'pending' ? 'pending' : activeTab === 'completed' ? 'success' : undefined;
      const response = await fetch(`/api/payments/history?status=${status || ''}`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });
      const data = await response.json();
      setPayments(data.payments || []);
    } catch (error) {
      console.error('Failed to fetch payments:', error);
    } finally {
      setLoading(false);
    }
  };

  const initiatePayment = async (payment: Payment) => {
    try {
      // Create payment order
      const response = await fetch('/api/payments/order', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        },
        body: JSON.stringify({
          property_id: payment.property_id,
          buyer_id: localStorage.getItem('userId'),
          payment_type: payment.payment_type,
          base_amount: payment.breakdown.base_amount,
          payment_gateway: selectedGateway
        })
      });

      const orderData = await response.json();

      // Initialize payment gateway
      if (selectedGateway === 'razorpay') {
        initializeRazorpay(orderData);
      } else if (selectedGateway === 'cashfree') {
        initializeCashfree(orderData);
      } else {
        initializePaytm(orderData);
      }
    } catch (error) {
      console.error('Payment initiation failed:', error);
      alert('Failed to initiate payment. Please try again.');
    }
  };

  const initializeRazorpay = (orderData: any) => {
    const options = {
      key: 'rzp_test_demo_key',
      amount: orderData.amount * 100,
      currency: 'INR',
      name: 'NAL Property Payments',
      description: `Payment for Order ${orderData.order_id}`,
      order_id: orderData.gateway_order_id,
      handler: async (response: any) => {
        await verifyPayment({
          order_id: orderData.order_id,
          payment_gateway: 'razorpay',
          gateway_payment_id: response.razorpay_payment_id,
          gateway_order_id: response.razorpay_order_id,
          gateway_signature: response.razorpay_signature
        });
      },
      prefill: {
        name: localStorage.getItem('userName') || '',
        email: localStorage.getItem('userEmail') || ''
      },
      theme: {
        color: '#3B82F6'
      }
    };

    // @ts-ignore
    const rzp = new window.Razorpay(options);
    rzp.open();
  };

  const initializeCashfree = (orderData: any) => {
    // Cashfree integration
    if (orderData.payment_link) {
      window.open(orderData.payment_link, '_blank');
    }
  };

  const initializePaytm = (orderData: any) => {
    // Paytm integration
    alert('Paytm payment integration - redirect to payment page');
  };

  const verifyPayment = async (verificationData: any) => {
    try {
      const response = await fetch('/api/payments/verify', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        },
        body: JSON.stringify(verificationData)
      });

      if (response.ok) {
        alert('Payment successful!');
        fetchPayments();
        setShowPaymentModal(false);
      } else {
        alert('Payment verification failed');
      }
    } catch (error) {
      console.error('Payment verification failed:', error);
      alert('Payment verification failed');
    }
  };

  const downloadReceipt = async (paymentId: string) => {
    try {
      const response = await fetch(`/api/payments/${paymentId}/receipt`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });
      const data = await response.json();
      alert(`Receipt: ${data.receipt_number}\nFile: ${data.file_path}`);
    } catch (error) {
      console.error('Failed to download receipt:', error);
    }
  };

  const getStatusBadge = (status: string) => {
    const badges: Record<string, { bg: string; text: string; icon: any }> = {
      pending: { bg: 'bg-yellow-100', text: 'text-yellow-700', icon: Clock },
      success: { bg: 'bg-green-100', text: 'text-green-700', icon: CheckCircle },
      failed: { bg: 'bg-red-100', text: 'text-red-700', icon: XCircle },
      processing: { bg: 'bg-blue-100', text: 'text-blue-700', icon: TrendingUp }
    };
    const badge = badges[status] || badges.pending;
    const Icon = badge.icon;

    return (
      <span className={`inline-flex items-center gap-1 px-3 py-1 rounded-full text-xs font-medium ${badge.bg} ${badge.text}`}>
        <Icon className="w-3 h-3" />
        {status.charAt(0).toUpperCase() + status.slice(1)}
      </span>
    );
  };

  const getPaymentTypeBadge = (type: string) => {
    const colors: Record<string, string> = {
      booking: 'bg-purple-100 text-purple-700',
      token: 'bg-blue-100 text-blue-700',
      milestone: 'bg-orange-100 text-orange-700',
      full_payment: 'bg-green-100 text-green-700',
      stamp_duty: 'bg-indigo-100 text-indigo-700'
    };

    return (
      <span className={`px-3 py-1 rounded-full text-xs font-medium ${colors[type] || 'bg-gray-100 text-gray-700'}`}>
        {type.replace('_', ' ').toUpperCase()}
      </span>
    );
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div>
        <h1 className="text-2xl font-bold text-gray-900">Property Purchase Payments</h1>
        <p className="text-gray-600 mt-1">Manage your property purchase payments and milestones</p>
      </div>

      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <div className="bg-gradient-to-br from-yellow-50 to-yellow-100 rounded-lg shadow-sm border border-yellow-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-yellow-700 font-medium">Pending Payments</p>
              <p className="text-3xl font-bold text-yellow-900 mt-2">
                ₹{payments.filter(p => p.status === 'pending').reduce((sum, p) => sum + p.amount, 0).toLocaleString()}
              </p>
              <p className="text-xs text-yellow-600 mt-1">
                {payments.filter(p => p.status === 'pending').length} payments due
              </p>
            </div>
            <div className="bg-yellow-200 p-3 rounded-lg">
              <Clock className="w-8 h-8 text-yellow-700" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-green-50 to-green-100 rounded-lg shadow-sm border border-green-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-green-700 font-medium">Completed Payments</p>
              <p className="text-3xl font-bold text-green-900 mt-2">
                ₹{payments.filter(p => p.status === 'success').reduce((sum, p) => sum + p.amount, 0).toLocaleString()}
              </p>
              <p className="text-xs text-green-600 mt-1">
                {payments.filter(p => p.status === 'success').length} payments completed
              </p>
            </div>
            <div className="bg-green-200 p-3 rounded-lg">
              <CheckCircle className="w-8 h-8 text-green-700" />
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg shadow-sm border border-blue-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-blue-700 font-medium">Total Paid</p>
              <p className="text-3xl font-bold text-blue-900 mt-2">
                ₹{payments.filter(p => p.status === 'success').reduce((sum, p) => sum + p.amount, 0).toLocaleString()}
              </p>
              <p className="text-xs text-blue-600 mt-1">All time</p>
            </div>
            <div className="bg-blue-200 p-3 rounded-lg">
              <DollarSign className="w-8 h-8 text-blue-700" />
            </div>
          </div>
        </div>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6" aria-label="Tabs">
            <button
              onClick={() => setActiveTab('pending')}
              className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors ${activeTab === 'pending'
                  ? 'border-blue-500 text-blue-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
            >
              Pending Payments
            </button>
            <button
              onClick={() => setActiveTab('completed')}
              className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors ${activeTab === 'completed'
                  ? 'border-blue-500 text-blue-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
            >
              Completed Payments
            </button>
            <button
              onClick={() => setActiveTab('milestones')}
              className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors ${activeTab === 'milestones'
                  ? 'border-blue-500 text-blue-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
            >
              Milestone Tracking
            </button>
          </nav>
        </div>

        {/* Tab Content */}
        <div className="p-6">
          {loading ? (
            <div className="text-center py-12">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
              <p className="text-gray-500 mt-4">Loading payments...</p>
            </div>
          ) : (
            <div className="space-y-4">
              {payments.length === 0 ? (
                <div className="text-center py-12">
                  <FileText className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                  <p className="text-gray-500">No payments found</p>
                </div>
              ) : (
                payments.map((payment) => (
                  <div
                    key={payment.id}
                    className="border border-gray-200 rounded-lg p-5 hover:border-blue-300 hover:shadow-md transition-all"
                  >
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        <div className="flex items-center gap-3 mb-3">
                          {getPaymentTypeBadge(payment.payment_type)}
                          {getStatusBadge(payment.status)}
                          {payment.milestone_number && (
                            <span className="text-xs text-gray-500">
                              Milestone {payment.milestone_number}/{payment.total_milestones}
                            </span>
                          )}
                        </div>

                        <h3 className="font-semibold text-gray-900 text-lg mb-2">
                          Order #{payment.order_id}
                        </h3>

                        {payment.transaction_id && (
                          <p className="text-sm text-gray-600 mb-2">
                            Transaction: {payment.transaction_id}
                          </p>
                        )}

                        {/* Payment Breakdown */}
                        <div className="mt-3 p-3 bg-gray-50 rounded-lg">
                          <p className="text-xs font-medium text-gray-700 mb-2">Payment Breakdown</p>
                          <div className="grid grid-cols-2 gap-2 text-xs">
                            <div className="flex justify-between">
                              <span className="text-gray-600">Base Amount:</span>
                              <span className="font-medium">₹{payment.breakdown.base_amount.toLocaleString()}</span>
                            </div>
                            <div className="flex justify-between">
                              <span className="text-gray-600">GST (18%):</span>
                              <span className="font-medium">₹{payment.breakdown.gst_amount.toLocaleString()}</span>
                            </div>
                            {payment.breakdown.tds_amount > 0 && (
                              <div className="flex justify-between">
                                <span className="text-gray-600">TDS (1%):</span>
                                <span className="font-medium text-red-600">-₹{payment.breakdown.tds_amount.toLocaleString()}</span>
                              </div>
                            )}
                            {payment.breakdown.stamp_duty > 0 && (
                              <div className="flex justify-between">
                                <span className="text-gray-600">Stamp Duty:</span>
                                <span className="font-medium">₹{payment.breakdown.stamp_duty.toLocaleString()}</span>
                              </div>
                            )}
                          </div>
                        </div>

                        <div className="mt-3 flex items-center gap-4 text-xs text-gray-500">
                          <span className="flex items-center gap-1">
                            <Calendar className="w-3 h-3" />
                            Created: {new Date(payment.created_at).toLocaleDateString('en-IN')}
                          </span>
                          {payment.due_date && (
                            <span className="flex items-center gap-1">
                              <AlertCircle className="w-3 h-3" />
                              Due: {new Date(payment.due_date).toLocaleDateString('en-IN')}
                            </span>
                          )}
                          {payment.paid_at && (
                            <span className="flex items-center gap-1 text-green-600">
                              <CheckCircle className="w-3 h-3" />
                              Paid: {new Date(payment.paid_at).toLocaleDateString('en-IN')}
                            </span>
                          )}
                        </div>
                      </div>

                      <div className="text-right ml-6">
                        <p className="text-3xl font-bold text-gray-900 mb-3">
                          ₹{payment.amount.toLocaleString()}
                        </p>

                        {payment.status === 'pending' ? (
                          <div className="space-y-2">
                            <button
                              onClick={() => {
                                setSelectedPayment(payment);
                                setShowPaymentModal(true);
                              }}
                              className="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-sm font-medium flex items-center justify-center gap-2"
                            >
                              <CreditCard className="w-4 h-4" />
                              Pay Now
                            </button>
                          </div>
                        ) : payment.status === 'success' ? (
                          <button
                            onClick={() => downloadReceipt(payment.id)}
                            className="w-full px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors text-sm font-medium flex items-center justify-center gap-2"
                          >
                            <Download className="w-4 h-4" />
                            Receipt
                          </button>
                        ) : null}
                      </div>
                    </div>
                  </div>
                ))
              )}
            </div>
          )}
        </div>
      </div>

      {/* Payment Gateway Selection Modal */}
      {showPaymentModal && selectedPayment && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg shadow-xl max-w-md w-full mx-4 p-6">
            <h3 className="text-xl font-bold text-gray-900 mb-4">Select Payment Gateway</h3>

            <div className="space-y-3 mb-6">
              <button
                onClick={() => setSelectedGateway('razorpay')}
                className={`w-full p-4 border-2 rounded-lg flex items-center justify-between transition-all ${selectedGateway === 'razorpay'
                    ? 'border-blue-500 bg-blue-50'
                    : 'border-gray-200 hover:border-gray-300'
                  }`}
              >
                <div className="flex items-center gap-3">
                  <Shield className="w-6 h-6 text-blue-600" />
                  <div className="text-left">
                    <p className="font-medium text-gray-900">Razorpay</p>
                    <p className="text-xs text-gray-500">UPI, Cards, Net Banking</p>
                  </div>
                </div>
                {selectedGateway === 'razorpay' && <CheckCircle className="w-5 h-5 text-blue-600" />}
              </button>

              <button
                onClick={() => setSelectedGateway('cashfree')}
                className={`w-full p-4 border-2 rounded-lg flex items-center justify-between transition-all ${selectedGateway === 'cashfree'
                    ? 'border-blue-500 bg-blue-50'
                    : 'border-gray-200 hover:border-gray-300'
                  }`}
              >
                <div className="flex items-center gap-3">
                  <Shield className="w-6 h-6 text-green-600" />
                  <div className="text-left">
                    <p className="font-medium text-gray-900">Cashfree</p>
                    <p className="text-xs text-gray-500">All payment methods</p>
                  </div>
                </div>
                {selectedGateway === 'cashfree' && <CheckCircle className="w-5 h-5 text-blue-600" />}
              </button>

              <button
                onClick={() => setSelectedGateway('paytm')}
                className={`w-full p-4 border-2 rounded-lg flex items-center justify-between transition-all ${selectedGateway === 'paytm'
                    ? 'border-blue-500 bg-blue-50'
                    : 'border-gray-200 hover:border-gray-300'
                  }`}
              >
                <div className="flex items-center gap-3">
                  <Shield className="w-6 h-6 text-indigo-600" />
                  <div className="text-left">
                    <p className="font-medium text-gray-900">Paytm</p>
                    <p className="text-xs text-gray-500">Wallet & more</p>
                  </div>
                </div>
                {selectedGateway === 'paytm' && <CheckCircle className="w-5 h-5 text-blue-600" />}
              </button>
            </div>

            <div className="bg-blue-50 border border-blue-200 rounded-lg p-3 mb-6">
              <div className="flex items-start gap-2">
                <Info className="w-4 h-4 text-blue-600 mt-0.5 flex-shrink-0" />
                <p className="text-xs text-blue-800">
                  Your payment is secured with 256-bit SSL encryption. All transactions are PCI-DSS compliant.
                </p>
              </div>
            </div>

            <div className="flex gap-3">
              <button
                onClick={() => setShowPaymentModal(false)}
                className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
              >
                Cancel
              </button>
              <button
                onClick={() => initiatePayment(selectedPayment)}
                className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors font-medium"
              >
                Proceed to Pay
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
