import React, { useState, useEffect, useRef } from 'react';
import { getUserVisitSchedules, VisitSchedule } from '../../api/visitSchedules';
import { Calendar, Eye, X, Video } from 'lucide-react';
import { LoadingDots } from '../LoadingDots';
import { toast } from 'sonner';
import { getUserId } from '../../utils/userUtils';

export const PropertyVisitManagement: React.FC = () => {
  const [visits, setVisits] = useState<VisitSchedule[]>([]);
  const [loading, setLoading] = useState(true);
  const [totalCount, setTotalCount] = useState(0);
  const [selectedVisit, setSelectedVisit] = useState<VisitSchedule | null>(null);
  const [activeTab, setActiveTab] = useState<'pending' | 'approved' | 'rejected'>('pending');


  const userId = getUserId();

  useEffect(() => {
    fetchVisits();
  }, [activeTab]);

  const fetchVisits = async () => {
    try {
      setLoading(true);
      
      if (!userId) {
        console.warn('No user ID found, cannot fetch visits');
        setVisits([]);
        setTotalCount(0);
        return;
      }
      
      let allVisits: VisitSchedule[] = [];
      let totalCount = 0;
      
      if (activeTab === 'pending') {
        // For pending tab, fetch both 'pending' and 'admin_approved' statuses
        try {
          const [pendingResponse, adminApprovedResponse] = await Promise.all([
            getUserVisitSchedules(userId, 1, 50, 'pending'),
            getUserVisitSchedules(userId, 1, 50, 'admin_approved')
          ]);
          
          allVisits = [...pendingResponse.visits, ...adminApprovedResponse.visits];
          totalCount = pendingResponse.total_count + adminApprovedResponse.total_count;
        } catch (error) {
          // If admin_approved fails, just get pending
          const response = await getUserVisitSchedules(userId, 1, 10, 'pending');
          allVisits = response.visits;
          totalCount = response.total_count;
        }
      } else if (activeTab === 'rejected') {
        // For rejected tab, fetch both seller and admin rejections
        try {
          const [sellerRejectedResponse, adminRejectedResponse] = await Promise.all([
            getUserVisitSchedules(userId, 1, 50, 'seller_rejected'),
            getUserVisitSchedules(userId, 1, 50, 'admin_rejected')
          ]);
          
          allVisits = [...sellerRejectedResponse.visits, ...adminRejectedResponse.visits];
          totalCount = sellerRejectedResponse.total_count + adminRejectedResponse.total_count;
        } catch (error) {
          // If admin_rejected fails, just get seller_rejected
          const response = await getUserVisitSchedules(userId, 1, 10, 'seller_rejected');
          allVisits = response.visits;
          totalCount = response.total_count;
        }
      } else {
        // For approved tab, just get seller_approved
        const response = await getUserVisitSchedules(userId, 1, 10, 'seller_approved');
        allVisits = response.visits;
        totalCount = response.total_count;
      }
      
      // Sort by created_at descending
      allVisits.sort((a, b) => new Date(b.created_at).getTime() - new Date(a.created_at).getTime());
      
      setVisits(allVisits);
      setTotalCount(totalCount);
    } catch (error) {
      console.error('Error fetching visits:', error);
      setVisits([]);
      setTotalCount(0);
    } finally {
      setLoading(false);
    }
  };



  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  if (loading) {
    return (
      <div className="flex justify-center py-12">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-semibold text-gray-900">Property Visit Requests</h3>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-xl border border-gray-200 p-1 shadow-sm">
        <div className="grid w-full grid-cols-3 gap-1">
          {[
            { key: 'pending', label: 'Pending' },
            { key: 'approved', label: 'Approved' },
            { key: 'rejected', label: 'Rejected' }
          ].map((tab) => {
            const isActive = activeTab === tab.key;
            
            return (
              <button
                key={tab.key}
                onClick={() => setActiveTab(tab.key as any)}
                className={`flex items-center justify-center gap-3 px-6 py-4 rounded-lg text-sm font-semibold transition-all duration-200 relative ${
                  isActive 
                    ? 'bg-blue-500 text-white shadow-lg' 
                    : 'text-gray-600 bg-transparent hover:bg-gray-50'
                }`}
              >
                <span>{tab.label}</span>
              </button>
            );
          })}
        </div>
      </div>

      {visits.length === 0 ? (
        <div className="text-center py-12">
          <Calendar className="mx-auto h-12 w-12 text-gray-400" />
          <h3 className="mt-2 text-sm font-medium text-gray-900">No visit requests</h3>
          <p className="mt-1 text-sm text-gray-500">Visit requests for your properties will appear here.</p>
        </div>
      ) : (
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50 border-b">
                <tr>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Property</th>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Visit Schedule</th>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Status</th>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Actions</th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {visits.map((visit) => (
                  <tr key={visit.id} className="hover:bg-gray-50 transition-colors">
                    <td className="px-6 py-4">
                      <div className="text-sm font-semibold text-gray-900">
                        {visit.property_title}
                      </div>
                      <div className="text-xs text-gray-500">{visit.property_city}</div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="text-sm text-gray-900">
                        {formatDate(visit.preferred_date)} at {visit.preferred_time}
                      </div>
                      <div className="text-xs text-gray-500">{visit.visit_mode}</div>
                    </td>
                    <td className="px-6 py-4">
                      <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                        visit.status === 'pending' ? 'bg-yellow-100 text-yellow-800' :
                        visit.status === 'admin_approved' ? 'bg-blue-100 text-blue-800' :
                        visit.status === 'seller_approved' ? 'bg-green-100 text-green-800' :
                        visit.status === 'seller_rejected' || visit.status === 'admin_rejected' ? 'bg-red-100 text-red-800' :
                        'bg-gray-100 text-gray-800'
                      }`}>
                        {visit.status === 'pending' ? 'Pending Admin' :
                         visit.status === 'admin_approved' ? 'Pending Seller' :
                         visit.status === 'seller_approved' ? 'Approved' :
                         visit.status === 'seller_rejected' ? 'Rejected by Seller' :
                         visit.status === 'admin_rejected' ? 'Rejected by Admin' :
                         visit.status}
                      </span>
                    </td>
                    <td className="px-6 py-4">
                      <div className="flex items-center space-x-2">
                        <button
                          onClick={() => setSelectedVisit(visit)}
                          className="inline-flex items-center px-3 py-1.5 text-xs font-medium text-white bg-blue-500 rounded-md hover:bg-blue-600 transition-all duration-200"
                        >
                          <Eye className="w-3 h-3 mr-1" />
                          View Details
                        </button>
                        {visit.video_call_enabled && visit.video_call_room_id && visit.status === 'seller_approved' && (
                          <a
                            href={`https://meet.jit.si/${visit.video_call_room_id}`}
                            target="_blank"
                            rel="noopener noreferrer"
                            className="inline-flex items-center px-3 py-1.5 text-xs font-medium text-white bg-green-500 rounded-md hover:bg-green-600 transition-all duration-200"
                          >
                            <Video className="w-3 h-3 mr-1" />
                            Video Call
                          </a>
                        )}
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}

      {/* Visit Details Modal */}
      {selectedVisit && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center p-4 z-50">
          <div className="bg-white rounded-2xl w-full max-w-3xl flex flex-col border border-gray-200 overflow-hidden font-inter" style={{ boxShadow: '0 25px 50px -12px rgba(0, 0, 0, 0.25)', maxHeight: '90vh' }}>
            <div className="px-6 py-4 border-b border-gray-100">
              <div className="flex items-center justify-between">
                <div>
                  <h2 className="text-xl font-bold text-gray-900 tracking-tight">Visit Request Details</h2>
                  <p className="text-gray-600 text-sm mt-1 font-medium">Request ID: #{selectedVisit.id}</p>
                </div>
                <button onClick={() => setSelectedVisit(null)} className="p-2 hover:text-gray-700 transition-colors">
                  <X className="w-5 h-5 text-gray-500" />
                </button>
              </div>
            </div>
            
            <div className="flex-1" style={{ minHeight: 0, overflowY: 'auto' }}>
              <div className="p-6 space-y-6">
                <div className="bg-blue-50 p-4 rounded-lg border border-blue-100">
                  <h3 className="text-lg font-semibold text-gray-900">{selectedVisit.property_title}</h3>
                  <p className="text-sm text-gray-600 mt-1">{selectedVisit.property_city}</p>
                </div>

                <div className="grid grid-cols-1 gap-6">
                  <div>
                    <h4 className="text-base font-semibold text-gray-900 mb-3">Visit Details</h4>
                    <div className="space-y-3">
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Preferred Date</label>
                        <p className="text-sm font-semibold text-gray-900">{formatDate(selectedVisit.preferred_date)}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Preferred Time</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.preferred_time}</p>
                      </div>
                      {selectedVisit.alternate_date && (
                        <>
                          <div>
                            <label className="block text-sm font-medium text-gray-500 mb-1">Alternate Date</label>
                            <p className="text-sm font-semibold text-gray-900">{formatDate(selectedVisit.alternate_date)}</p>
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-500 mb-1">Alternate Time</label>
                            <p className="text-sm font-semibold text-gray-900">{selectedVisit.alternate_time}</p>
                          </div>
                        </>
                      )}
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Visit Mode</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.visit_mode}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Number of Visitors</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.number_of_visitors}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Status</label>
                        <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                          selectedVisit.status === 'pending' ? 'bg-yellow-100 text-yellow-800' :
                          selectedVisit.status === 'admin_approved' ? 'bg-blue-100 text-blue-800' :
                          selectedVisit.status === 'seller_approved' ? 'bg-green-100 text-green-800' :
                          selectedVisit.status === 'seller_rejected' || selectedVisit.status === 'admin_rejected' ? 'bg-red-100 text-red-800' :
                          'bg-gray-100 text-gray-800'
                        }`}>
                          {selectedVisit.status === 'pending' ? 'Pending Admin Approval' :
                           selectedVisit.status === 'admin_approved' ? 'Pending Seller Approval' :
                           selectedVisit.status === 'seller_approved' ? 'Approved' :
                           selectedVisit.status === 'seller_rejected' ? 'Rejected by Seller' :
                           selectedVisit.status === 'admin_rejected' ? 'Rejected by Admin' :
                           selectedVisit.status}
                        </span>
                      </div>
                      {selectedVisit.video_call_enabled && (
                        <div>
                          <label className="block text-sm font-medium text-gray-500 mb-1">Video Call</label>
                          <div className="flex items-center gap-2">
                            <span className="text-sm font-semibold text-green-600">Available</span>
                            {selectedVisit.video_call_room_id && selectedVisit.status === 'seller_approved' && (
                              <a
                                href={`https://meet.jit.si/${selectedVisit.video_call_room_id}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="inline-flex items-center px-3 py-1 text-xs font-medium text-white bg-green-500 rounded hover:bg-green-600"
                              >
                                <Video className="w-3 h-3 mr-1" />
                                Join Video Call
                              </a>
                            )}
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                </div>

                {selectedVisit.comments && (
                  <div>
                    <h4 className="text-base font-semibold text-gray-900 mb-2">Comments</h4>
                    <p className="text-sm text-gray-700">{selectedVisit.comments}</p>
                  </div>
                )}
              </div>
            </div>
          </div>
        </div>
      )}
      

    </div>
  );
};