import React, { useState, useEffect } from 'react';
import { Loader } from 'lucide-react';
import { getRecommendedProperties, Property } from '../../api/properties';
import { PropertyCard } from '../PropertyCard';
import { toast } from 'sonner';

export function RecommendedProperties() {
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchRecommendedProperties();
  }, []);

  const fetchRecommendedProperties = async () => {
    try {
      setLoading(true);
      const recommendedProperties = await getRecommendedProperties();
      setProperties(recommendedProperties);
    } catch (error) {
      console.error('Error fetching recommended properties:', error);
      toast.error('Failed to load recommended properties');
    } finally {
      setLoading(false);
    }
  };

  const formatPrice = (price: number) => {
    if (!price || isNaN(price)) return '₹0';
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(1)} Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(0)} L`;
    return `₹${price.toLocaleString()}`;
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <Loader className="w-8 h-8 animate-spin text-blue-500" />
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">Recommended Properties</h2>
          <p className="text-gray-600">
            Based on your preferences, below are the recommended properties
          </p>
        </div>
      </div>

      {properties.length === 0 ? (
        <div className="text-center py-12">
          <p className="text-gray-500">No properties match your preferences. Try adjusting your criteria.</p>
        </div>
      ) : (
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 xl:grid-cols-5 gap-4">
          {properties.map((property) => (
            <PropertyCard
              key={property.id}
              id={property.id}
              title={property.title}
              location={`${property.locality}, ${property.city}`}
              beds={property.bedrooms}
              baths={property.bathrooms}
              area={property.built_up_area ? `${property.built_up_area} sq ft` : 'N/A'}
              price={formatPrice(property.sale_price)}
              priceNegotiable={property.price_negotiable}
              loanAvailable={property.loan_available}
              propertyData={{
                id: property.id.toString(),
                title: property.title,
                location: `${property.locality}, ${property.city}`,
                beds: property.bedrooms,
                baths: property.bathrooms,
                area: property.built_up_area ? `${property.built_up_area} sq ft` : 'N/A',
                price: formatPrice(property.sale_price),
                image: '',
                images: [],
                verified: true,
                riblScore: 'A+',
                description: property.description || '',
                amenities: property.amenities || [],
                nearbyPlaces: [],
                priceHistory: [],
                agent: {
                  name: '',
                  rating: 0,
                  experience: '',
                  phone: '',
                  image: ''
                },
                specifications: {
                  propertyType: property.property_type,
                  facing: '',
                  floor: '',
                  totalFloors: '',
                  age: '',
                  furnishing: property.furnishing || '',
                  parking: 0
                },
                legalInfo: {
                  approvals: [],
                  rera: '',
                  ownership: ''
                }
              }}
            />
          ))}
        </div>
      )}
    </div>
  );
}