import React, { useState, useEffect } from 'react';
import { DollarSign, TrendingUp, Home, BarChart3, MapPin, Calendar } from 'lucide-react';
import { rentalYieldApi, TopLocality, PropertyRentalAnalysis } from '../../api/rentalYield';
import { toast } from 'sonner';

export function RentalYieldDashboard() {
  const [topLocalities, setTopLocalities] = useState<TopLocality[]>([]);
  const [selectedCity, setSelectedCity] = useState<string>('');
  const [loading, setLoading] = useState(true);
  const [propertyAnalysis, setPropertyAnalysis] = useState<PropertyRentalAnalysis | null>(null);
  const [propertyId, setPropertyId] = useState<string>('');

  useEffect(() => {
    loadTopLocalities();
  }, [selectedCity]);

  const loadTopLocalities = async () => {
    try {
      setLoading(true);
      const data = await rentalYieldApi.getTopLocalities(selectedCity || undefined, 10);
      setTopLocalities(data);
    } catch (error) {
      console.error('Error loading rental yield data:', error);
      toast.error('Failed to load rental yield data');
    } finally {
      setLoading(false);
    }
  };

  const analyzeProperty = async () => {
    if (!propertyId) {
      toast.error('Please enter a property ID');
      return;
    }

    try {
      const analysis = await rentalYieldApi.getPropertyAnalysis(parseInt(propertyId));
      setPropertyAnalysis(analysis);
      toast.success('Property analysis completed');
    } catch (error) {
      console.error('Error analyzing property:', error);
      toast.error('Failed to analyze property');
    }
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-blue-600 to-indigo-600 rounded-xl p-6 text-white">
        <h1 className="text-3xl font-bold mb-2">Rental Yield Analysis</h1>
        <p className="text-blue-100">Real-time rental profitability insights from actual market data</p>
      </div>

      {/* Property Analysis Tool */}
      <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-200">
        <h2 className="text-xl font-bold text-gray-900 mb-4">Analyze Property Rental Potential</h2>
        <div className="flex gap-4">
          <input
            type="number"
            placeholder="Enter Property ID"
            value={propertyId}
            onChange={(e) => setPropertyId(e.target.value)}
            className="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          />
          <button
            onClick={analyzeProperty}
            className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Analyze
          </button>
        </div>

        {propertyAnalysis && (
          <div className="mt-6 grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            <div className="bg-green-50 p-4 rounded-lg border border-green-200">
              <div className="flex items-center gap-2 mb-2">
                <DollarSign className="w-5 h-5 text-green-600" />
                <span className="text-sm font-medium text-gray-600">Monthly Rent</span>
              </div>
              <p className="text-2xl font-bold text-green-700">
                ₹{propertyAnalysis.expected_monthly_rent.toLocaleString()}
              </p>
            </div>

            <div className="bg-blue-50 p-4 rounded-lg border border-blue-200">
              <div className="flex items-center gap-2 mb-2">
                <TrendingUp className="w-5 h-5 text-blue-600" />
                <span className="text-sm font-medium text-gray-600">Rental Yield</span>
              </div>
              <p className="text-2xl font-bold text-blue-700">
                {propertyAnalysis.rental_yield.toFixed(2)}%
              </p>
            </div>

            <div className="bg-purple-50 p-4 rounded-lg border border-purple-200">
              <div className="flex items-center gap-2 mb-2">
                <BarChart3 className="w-5 h-5 text-purple-600" />
                <span className="text-sm font-medium text-gray-600">Net Yield</span>
              </div>
              <p className="text-2xl font-bold text-purple-700">
                {propertyAnalysis.net_yield_after_costs.toFixed(2)}%
              </p>
            </div>

            <div className="bg-orange-50 p-4 rounded-lg border border-orange-200">
              <div className="flex items-center gap-2 mb-2">
                <Calendar className="w-5 h-5 text-orange-600" />
                <span className="text-sm font-medium text-gray-600">Break-Even</span>
              </div>
              <p className="text-2xl font-bold text-orange-700">
                {propertyAnalysis.break_even_years.toFixed(1)} years
              </p>
            </div>

            <div className="bg-indigo-50 p-4 rounded-lg border border-indigo-200">
              <div className="flex items-center gap-2 mb-2">
                <Home className="w-5 h-5 text-indigo-600" />
                <span className="text-sm font-medium text-gray-600">Demand Score</span>
              </div>
              <p className="text-2xl font-bold text-indigo-700">
                {propertyAnalysis.demand_score}/100
              </p>
            </div>

            <div className="bg-teal-50 p-4 rounded-lg border border-teal-200">
              <div className="flex items-center gap-2 mb-2">
                <TrendingUp className="w-5 h-5 text-teal-600" />
                <span className="text-sm font-medium text-gray-600">Occupancy Rate</span>
              </div>
              <p className="text-2xl font-bold text-teal-700">
                {propertyAnalysis.occupancy_rate.toFixed(1)}%
              </p>
            </div>

            <div className="bg-pink-50 p-4 rounded-lg border border-pink-200">
              <div className="flex items-center gap-2 mb-2">
                <DollarSign className="w-5 h-5 text-pink-600" />
                <span className="text-sm font-medium text-gray-600">Annual Income</span>
              </div>
              <p className="text-2xl font-bold text-pink-700">
                ₹{propertyAnalysis.annual_rental_income.toLocaleString()}
              </p>
            </div>

            <div className="bg-yellow-50 p-4 rounded-lg border border-yellow-200">
              <div className="flex items-center gap-2 mb-2">
                <MapPin className="w-5 h-5 text-yellow-600" />
                <span className="text-sm font-medium text-gray-600">Locality Avg</span>
              </div>
              <p className="text-2xl font-bold text-yellow-700">
                {propertyAnalysis.locality_avg_yield.toFixed(2)}%
              </p>
            </div>
          </div>
        )}
      </div>

      {/* Top Localities by Rental Yield */}
      <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-200">
        <div className="flex items-center justify-between mb-6">
          <h2 className="text-xl font-bold text-gray-900">Top Rental Yield Localities</h2>
          <select
            value={selectedCity}
            onChange={(e) => setSelectedCity(e.target.value)}
            className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
          >
            <option value="">All Cities</option>
            <option value="Bangalore">Bangalore</option>
            <option value="Mumbai">Mumbai</option>
            <option value="Delhi">Delhi</option>
            <option value="Pune">Pune</option>
            <option value="Hyderabad">Hyderabad</option>
          </select>
        </div>

        {loading ? (
          <div className="text-center py-8">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
            <p className="mt-4 text-gray-600">Loading rental yield data...</p>
          </div>
        ) : topLocalities.length === 0 ? (
          <div className="text-center py-8 text-gray-500">
            No rental yield data available. Please add properties with rental information.
          </div>
        ) : (
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead>
                <tr className="border-b border-gray-200">
                  <th className="text-left py-3 px-4 font-semibold text-gray-700">Rank</th>
                  <th className="text-left py-3 px-4 font-semibold text-gray-700">City</th>
                  <th className="text-left py-3 px-4 font-semibold text-gray-700">Locality</th>
                  <th className="text-right py-3 px-4 font-semibold text-gray-700">Avg Rent</th>
                  <th className="text-right py-3 px-4 font-semibold text-gray-700">Avg Price</th>
                  <th className="text-right py-3 px-4 font-semibold text-gray-700">Yield %</th>
                  <th className="text-right py-3 px-4 font-semibold text-gray-700">Properties</th>
                </tr>
              </thead>
              <tbody>
                {topLocalities.map((locality, index) => (
                  <tr key={index} className="border-b border-gray-100 hover:bg-gray-50">
                    <td className="py-3 px-4">
                      <span className="inline-flex items-center justify-center w-8 h-8 rounded-full bg-blue-100 text-blue-700 font-bold">
                        {index + 1}
                      </span>
                    </td>
                    <td className="py-3 px-4 text-gray-900">{locality.city}</td>
                    <td className="py-3 px-4 font-medium text-gray-900">{locality.locality}</td>
                    <td className="py-3 px-4 text-right text-gray-700">
                      ₹{locality.avg_monthly_rent.toLocaleString()}
                    </td>
                    <td className="py-3 px-4 text-right text-gray-700">
                      ₹{(locality.avg_property_price / 10000000).toFixed(2)}Cr
                    </td>
                    <td className="py-3 px-4 text-right">
                      <span className="inline-flex items-center px-3 py-1 rounded-full text-sm font-semibold bg-green-100 text-green-800">
                        {locality.rental_yield_percentage.toFixed(2)}%
                      </span>
                    </td>
                    <td className="py-3 px-4 text-right text-gray-600">
                      {locality.sample_size}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>

      {/* Info Cards */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <div className="bg-gradient-to-br from-green-50 to-emerald-50 p-6 rounded-xl border border-green-200">
          <DollarSign className="w-10 h-10 text-green-600 mb-3" />
          <h3 className="text-lg font-bold text-gray-900 mb-2">Real Market Data</h3>
          <p className="text-gray-600 text-sm">
            Analysis based on actual property listings and rental rates in your area
          </p>
        </div>

        <div className="bg-gradient-to-br from-blue-50 to-indigo-50 p-6 rounded-xl border border-blue-200">
          <BarChart3 className="w-10 h-10 text-blue-600 mb-3" />
          <h3 className="text-lg font-bold text-gray-900 mb-2">Dynamic Calculations</h3>
          <p className="text-gray-600 text-sm">
            Live rental yield calculations updated with latest market trends
          </p>
        </div>

        <div className="bg-gradient-to-br from-purple-50 to-pink-50 p-6 rounded-xl border border-purple-200">
          <TrendingUp className="w-10 h-10 text-purple-600 mb-3" />
          <h3 className="text-lg font-bold text-gray-900 mb-2">Investment Insights</h3>
          <p className="text-gray-600 text-sm">
            Comprehensive analysis including break-even and ROI projections
          </p>
        </div>
      </div>
    </div>
  );
}
