import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Badge } from '../ui/badge';

interface SafetyScoreCardProps {
  score: number;
  factors: {
    legal: number;
    price: number;
    location: number;
    builder: number;
    investment: number;
  };
}

const SafetyScoreCard: React.FC<SafetyScoreCardProps> = ({ score, factors }) => {
  const getScoreColor = (score: number) => {
    if (score >= 85) return 'text-green-600 bg-green-50 border-green-200';
    if (score >= 70) return 'text-yellow-600 bg-yellow-50 border-yellow-200';
    return 'text-red-600 bg-red-50 border-red-200';
  };

  const getFactorColor = (factor: number) => {
    if (factor >= 85) return 'bg-green-500';
    if (factor >= 70) return 'bg-yellow-500';
    return 'bg-red-500';
  };

  return (
    <Card className={`border-2 ${getScoreColor(score)}`}>
      <CardHeader className="pb-3">
        <CardTitle className="flex items-center justify-between">
          <div className="flex items-center space-x-2">
            <span className="text-2xl">🛡️</span>
            <span>Safety Score</span>
          </div>
          <Badge className={`text-lg px-3 py-1 ${getScoreColor(score)}`}>
            {score}/100
          </Badge>
        </CardTitle>
      </CardHeader>
      <CardContent>
        <div className="space-y-3">
          <div className="flex items-center justify-between">
            <span className="text-sm font-medium">Legal Safety</span>
            <div className="flex items-center space-x-2">
              <div className="w-16 bg-gray-200 rounded-full h-2">
                <div 
                  className={`h-2 rounded-full ${getFactorColor(factors.legal)}`}
                  style={{ width: `${factors.legal}%` }}
                ></div>
              </div>
              <span className="text-sm font-semibold w-8">{factors.legal}</span>
            </div>
          </div>
          
          <div className="flex items-center justify-between">
            <span className="text-sm font-medium">Price Safety</span>
            <div className="flex items-center space-x-2">
              <div className="w-16 bg-gray-200 rounded-full h-2">
                <div 
                  className={`h-2 rounded-full ${getFactorColor(factors.price)}`}
                  style={{ width: `${factors.price}%` }}
                ></div>
              </div>
              <span className="text-sm font-semibold w-8">{factors.price}</span>
            </div>
          </div>
          
          <div className="flex items-center justify-between">
            <span className="text-sm font-medium">Location Safety</span>
            <div className="flex items-center space-x-2">
              <div className="w-16 bg-gray-200 rounded-full h-2">
                <div 
                  className={`h-2 rounded-full ${getFactorColor(factors.location)}`}
                  style={{ width: `${factors.location}%` }}
                ></div>
              </div>
              <span className="text-sm font-semibold w-8">{factors.location}</span>
            </div>
          </div>
          
          <div className="flex items-center justify-between">
            <span className="text-sm font-medium">Builder Safety</span>
            <div className="flex items-center space-x-2">
              <div className="w-16 bg-gray-200 rounded-full h-2">
                <div 
                  className={`h-2 rounded-full ${getFactorColor(factors.builder)}`}
                  style={{ width: `${factors.builder}%` }}
                ></div>
              </div>
              <span className="text-sm font-semibold w-8">{factors.builder}</span>
            </div>
          </div>
          
          <div className="flex items-center justify-between">
            <span className="text-sm font-medium">Investment Safety</span>
            <div className="flex items-center space-x-2">
              <div className="w-16 bg-gray-200 rounded-full h-2">
                <div 
                  className={`h-2 rounded-full ${getFactorColor(factors.investment)}`}
                  style={{ width: `${factors.investment}%` }}
                ></div>
              </div>
              <span className="text-sm font-semibold w-8">{factors.investment}</span>
            </div>
          </div>
        </div>
        
        <div className="mt-4 p-3 bg-white rounded border">
          <div className="text-xs text-gray-600 text-center">
            {score >= 85 && "✅ Highly Safe Investment"}
            {score >= 70 && score < 85 && "⚠️ Moderately Safe Investment"}
            {score < 70 && "❌ High Risk Investment"}
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

export default SafetyScoreCard;