import React, { useState, useEffect } from 'react';
import { CreditCard, Trash2, Star, Plus, CheckCircle } from 'lucide-react';

interface SavedMethod {
  id: string;
  method_type: string;
  last_four?: string;
  card_network?: string;
  card_type?: string;
  expiry_month?: number;
  expiry_year?: number;
  upi_handle?: string;
  bank_name?: string;
  is_default: boolean;
  last_used_at?: string;
  created_at: string;
}

export function SavedPaymentMethods() {
  const [methods, setMethods] = useState<SavedMethod[]>([]);
  const [loading, setLoading] = useState(true);
  const [showAddForm, setShowAddForm] = useState(false);

  useEffect(() => {
    fetchSavedMethods();
  }, []);

  const fetchSavedMethods = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await fetch('/api/payments/methods', {
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });

      if (response.ok) {
        const data = await response.json();
        setMethods(data);
      }
    } catch (error) {
      console.error('Failed to fetch saved payment methods:', error);
    } finally {
      setLoading(false);
    }
  };

  const deleteMethod = async (methodId: string) => {
    if (!confirm('Are you sure you want to remove this payment method?')) {
      return;
    }

    try {
      const token = localStorage.getItem('token');
      const response = await fetch(`/api/payments/methods/${methodId}`, {
        method: 'DELETE',
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });

      if (response.ok) {
        setMethods(methods.filter(m => m.id !== methodId));
      } else {
        alert('Failed to delete payment method');
      }
    } catch (error) {
      console.error('Failed to delete payment method:', error);
      alert('Failed to delete payment method');
    }
  };

  const getCardIcon = (network?: string) => {
    const icons: Record<string, string> = {
      visa: '💳',
      mastercard: '💳',
      amex: '💳',
      rupay: '💳'
    };
    return icons[network?.toLowerCase() || ''] || '💳';
  };

  const getMethodDisplay = (method: SavedMethod) => {
    if (method.method_type === 'card') {
      return {
        icon: getCardIcon(method.card_network),
        title: `${method.card_network?.toUpperCase()} ${method.card_type || ''}`,
        subtitle: `•••• ${method.last_four}`,
        extra: method.expiry_month && method.expiry_year
          ? `Expires ${method.expiry_month}/${method.expiry_year}`
          : null
      };
    } else if (method.method_type === 'upi') {
      return {
        icon: '📱',
        title: 'UPI',
        subtitle: method.upi_handle || 'UPI Account',
        extra: null
      };
    } else if (method.method_type === 'netbanking') {
      return {
        icon: '🏦',
        title: method.bank_name || 'Net Banking',
        subtitle: 'Bank Account',
        extra: null
      };
    }
    return {
      icon: '💰',
      title: method.method_type,
      subtitle: '',
      extra: null
    };
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center p-12">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">Saved Payment Methods</h2>
          <p className="text-gray-600 mt-1">Manage your saved payment methods for faster checkout</p>
        </div>
        <button
          onClick={() => setShowAddForm(true)}
          className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors flex items-center gap-2"
        >
          <Plus className="w-4 h-4" />
          Add Method
        </button>
      </div>

      {/* Info Banner */}
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
        <div className="flex items-start gap-3">
          <CheckCircle className="w-5 h-5 text-blue-600 mt-0.5" />
          <div>
            <h3 className="font-semibold text-blue-900">Secure & Encrypted</h3>
            <p className="text-sm text-blue-700 mt-1">
              Your payment information is tokenized and encrypted. We never store your actual card details.
            </p>
          </div>
        </div>
      </div>

      {/* Saved Methods List */}
      {methods.length === 0 ? (
        <div className="text-center py-12 bg-gray-50 rounded-lg border-2 border-dashed border-gray-300">
          <CreditCard className="w-12 h-12 text-gray-400 mx-auto mb-3" />
          <h3 className="font-semibold text-gray-900 mb-2">No Saved Payment Methods</h3>
          <p className="text-gray-600 mb-4">Add a payment method to checkout faster next time</p>
          <button
            onClick={() => setShowAddForm(true)}
            className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors inline-flex items-center gap-2"
          >
            <Plus className="w-4 h-4" />
            Add Payment Method
          </button>
        </div>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          {methods.map((method) => {
            const display = getMethodDisplay(method);
            return (
              <div
                key={method.id}
                className={`bg-white border-2 rounded-lg p-5 hover:shadow-md transition-all ${method.is_default ? 'border-blue-500 bg-blue-50' : 'border-gray-200'
                  }`}
              >
                <div className="flex items-start justify-between mb-3">
                  <div className="flex items-center gap-3">
                    <div className="text-3xl">{display.icon}</div>
                    <div>
                      <h3 className="font-semibold text-gray-900 flex items-center gap-2">
                        {display.title}
                        {method.is_default && (
                          <span className="px-2 py-0.5 bg-blue-600 text-white text-xs rounded-full flex items-center gap-1">
                            <Star className="w-3 h-3 fill-current" />
                            Default
                          </span>
                        )}
                      </h3>
                      <p className="text-sm text-gray-600">{display.subtitle}</p>
                      {display.extra && (
                        <p className="text-xs text-gray-500 mt-1">{display.extra}</p>
                      )}
                    </div>
                  </div>
                  <button
                    onClick={() => deleteMethod(method.id)}
                    className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
                    title="Remove payment method"
                  >
                    <Trash2 className="w-4 h-4" />
                  </button>
                </div>

                {method.last_used_at && (
                  <div className="text-xs text-gray-500 mt-2 pt-2 border-t">
                    Last used: {new Date(method.last_used_at).toLocaleDateString('en-IN')}
                  </div>
                )}
              </div>
            );
          })}
        </div>
      )}

      {/* Add Method Form Modal */}
      {showAddForm && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg max-w-md w-full p-6">
            <h3 className="text-xl font-bold text-gray-900 mb-4">Add Payment Method</h3>
            <p className="text-gray-600 mb-6">
              To add a payment method, please make a payment first. Your payment details will be securely saved for future use.
            </p>
            <div className="flex gap-3">
              <button
                onClick={() => setShowAddForm(false)}
                className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
              >
                Close
              </button>
              <button
                onClick={() => {
                  setShowAddForm(false);
                  // Navigate to payments tab
                  window.location.hash = '#payments';
                }}
                className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
              >
                Make a Payment
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
