import { useState, useEffect } from 'react';
import { Bookmark } from 'lucide-react';
import { userDashboardAPI, SavedProperty } from '../../api/user/userDashboard';
import { PropertyCard } from '../PropertyCard';
import { getUserId } from '../../utils/userUtils';

export function SavedProperties() {
  const [savedProperties, setSavedProperties] = useState<SavedProperty[]>([]);
  const [loading, setLoading] = useState(true);


  const fetchSavedProperties = async () => {
    try {
      const userId = getUserId();
      if (!userId) {
        console.log('No user ID found');
        return;
      }

      console.log('Fetching saved properties for user:', userId);
      const savedData = await userDashboardAPI.getSavedProperties(userId);
      console.log('Saved properties response:', savedData);

        const propertiesWithDetails = await Promise.all(
          savedData.map(async (saved) => {
            try {
              const propertyResponse = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/properties/${saved.property_id}`);
              if (propertyResponse.ok) {
                const propertyData = await propertyResponse.json();
                return {
                  ...saved,
                  property: {
                    id: propertyData.id,
                    title: propertyData.title,
                    location: `${propertyData.locality}, ${propertyData.city}`,
                    beds: propertyData.bedrooms,
                    baths: propertyData.bathrooms,
                    area: `${propertyData.built_up_area} sq ft`,
                    price: `₹${propertyData.sale_price || propertyData.rent_price}`,
                    imageUrl:
                      propertyData.primary_image ||
                      propertyData.image_urls?.[0] ||
                      propertyData.photo ||
                      null,
                    isUrgent: propertyData.is_urgent,
                    isBidding: propertyData.is_bidding,
                    city: propertyData.city,
                    locality: propertyData.locality,
                    sale_price: propertyData.sale_price,
                    built_up_area: propertyData.built_up_area,
                  },
                };
              }
              return saved;
            } catch (error) {
              console.error(`Error fetching property ${saved.property_id}:`, error);
              return saved;
            }
          })
        );

      setSavedProperties(propertiesWithDetails);
      console.log('Final saved properties:', propertiesWithDetails);
    } catch (error) {
      console.error('Error fetching saved properties:', error);
      // Show user-friendly error message
      if (error instanceof Error && error.message.includes('Failed to fetch')) {
        console.error('API connection error - check if backend is running on correct port');
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchSavedProperties();

    // Listen for property state changes
    const handlePropertyStateChange = (event: Event) => {
      const customEvent = event as CustomEvent;
      if (customEvent.detail.type === 'saved') {
        fetchSavedProperties();
      }
    };

    window.addEventListener('propertyStateChange', handlePropertyStateChange);
    return () => window.removeEventListener('propertyStateChange', handlePropertyStateChange);
  }, []);



  const formatPrice = (price: string | undefined) => {
    if (!price) return 'Price on request';
    const numPrice = parseFloat(price.replace(/[^\d.]/g, ''));
    if (isNaN(numPrice)) return 'Price on request';
    if (numPrice >= 10000000) return `₹${(numPrice / 10000000).toFixed(1)} Cr`;
    if (numPrice >= 100000) return `₹${(numPrice / 100000).toFixed(0)} L`;
    return `₹${numPrice.toLocaleString()}`;
  };

  if (loading) {
    return (
      <div className="space-y-6">
        <h2 className="text-2xl font-bold text-gray-900">Saved Properties</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {[1, 2, 3].map((i) => (
            <div
              key={i}
              className="bg-white rounded-lg shadow-sm border border-gray-200 animate-pulse"
            >
              <div className="h-48 bg-gray-200 rounded-t-lg"></div>
              <div className="p-4 space-y-3">
                <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                <div className="h-3 bg-gray-200 rounded w-1/2"></div>
                <div className="h-3 bg-gray-200 rounded w-1/4"></div>
              </div>
            </div>
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h3 className="text-lg font-semibold text-gray-900">Saved Properties</h3>
        <span className="text-sm text-gray-500">
          {savedProperties.length} properties saved
        </span>
      </div>

      {savedProperties.length === 0 ? (
        <div className="text-center py-12">
          <Bookmark className="w-16 h-16 text-gray-300 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No saved properties</h3>
          <p className="text-gray-600">
            Start exploring properties and save your favorites here
          </p>
        </div>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {savedProperties.map((savedProp) => {
            const p = savedProp.property;
            if (!p) return null;
            return (
              <PropertyCard
                key={savedProp.id}
                id={p.id}
                title={p.title}
                location={p.location}
                beds={p.beds}
                baths={p.baths}
                area={p.area}
                price={formatPrice(p.price)}
                imageUrl={p.imageUrl?.startsWith('http') ? p.imageUrl : `${import.meta.env.VITE_API_BASE?.replace('/api', '') || 'http://localhost:8090'}${p.imageUrl}`}
                isUrgent={p.isUrgent}
                isBidding={p.isBidding}
              />
            );
          })}
        </div>
      )}
    </div>
  );
}
