import React, { useState, useEffect } from "react";
import { Calendar, Clock, MapPin, User, Phone, Mail, MessageSquare, CheckCircle, XCircle, AlertCircle, X } from "lucide-react";
import { getUserVisitSchedules, VisitSchedule } from '../../api/visitSchedules';
import { getUserId } from '../../utils/userUtils';

export function ScheduledVisits() {
  console.log('ScheduledVisits component is rendering - CHECK THIS LOG');
  const [visits, setVisits] = useState<VisitSchedule[]>([]);
  const [selectedVisit, setSelectedVisit] = useState<VisitSchedule | null>(null);
  const [loading, setLoading] = useState(true);
  const [selectedStatus, setSelectedStatus] = useState<string>('');

  useEffect(() => {
    if (selectedVisit) {
      document.body.style.overflow = 'hidden';
    } else {
      document.body.style.overflow = 'unset';
    }
    return () => {
      document.body.style.overflow = 'unset';
    };
  }, [selectedVisit]);

  useEffect(() => {
    console.log('ScheduledVisits: useEffect triggered, selectedStatus:', selectedStatus);
    fetchVisits();
  }, []);

  const fetchVisits = async () => {
    try {
      const userId = getUserId();
      if (!userId) {
        console.log('ScheduledVisits: No user ID found, user not logged in');
        return;
      }
      
      console.log('ScheduledVisits: Fetching visits for user ID:', userId);
      
      const response = await getUserVisitSchedules(userId, 1, 50);
      
      console.log('ScheduledVisits: API Response:', response);
      console.log('ScheduledVisits: All visits:', response.visits);
      
      // Temporarily show all visits for debugging
      setVisits(response.visits);
      
      // Filter for upcoming visits (approved and future dates)
      const today = new Date();
      today.setHours(0, 0, 0, 0);
      
      const upcomingVisits = response.visits.filter(visit => {
        const isApproved = visit.status === 'seller_approved';
        const visitDate = new Date(visit.preferred_date);
        const isFuture = visitDate >= today;
        console.log(`Visit ${visit.id}: status=${visit.status}, date=${visit.preferred_date}, isApproved=${isApproved}, isFuture=${isFuture}`);
        return isApproved && isFuture;
      });
      
      console.log('ScheduledVisits: Filtered upcoming visits:', upcomingVisits);
      
      // setVisits(upcomingVisits);
    } catch (error) {
      console.error('ScheduledVisits: Error fetching visit schedules:', error);
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "seller_approved":
      case "approved":
      case "confirmed":
        return "bg-green-100 text-green-800";
      case "pending":
      case "admin_approved":
      case "scheduled":
        return "bg-yellow-100 text-yellow-800";
      case "completed":
        return "bg-blue-100 text-blue-800";
      case "seller_rejected":
      case "admin_rejected":
      case "rejected":
      case "cancelled":
        return "bg-red-100 text-red-800";
      case "rescheduled":
        return "bg-purple-100 text-purple-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "seller_approved":
      case "approved":
      case "confirmed":
        return <CheckCircle className="w-4 h-4" />;
      case "pending":
      case "admin_approved":
      case "scheduled":
        return <AlertCircle className="w-4 h-4" />;
      case "completed":
        return <CheckCircle className="w-4 h-4" />;
      case "seller_rejected":
      case "admin_rejected":
      case "rejected":
      case "cancelled":
        return <XCircle className="w-4 h-4" />;
      case "rescheduled":
        return <Clock className="w-4 h-4" />;
      default:
        return <AlertCircle className="w-4 h-4" />;
    }
  };



  return (
    <div className="space-y-6">
      <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">Scheduled Visits</h2>
          <p className="text-gray-600">Manage your property visit appointments</p>
        </div>
        
      </div>

      {loading ? (
        <div className="flex justify-center py-12">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      ) : (
        <div>
          <div className="mb-4 p-4 bg-gray-100 rounded">
            <p>Debug: Total visits: {visits.length}</p>
            <p>Debug: Visits data: {JSON.stringify(visits, null, 2)}</p>
          </div>
          {visits.length === 0 ? (
            <div className="text-center py-12">
              <Calendar className="w-16 h-16 text-gray-300 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No upcoming visits scheduled</h3>
              <p className="text-gray-600">Your approved upcoming property visits will appear here</p>
            </div>
          ) : (
            <div className="bg-white rounded-xl shadow-sm border border-gray-200">
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gray-50 border-b">
                    <tr>
                      <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide w-1/4">Property</th>
                      <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide w-1/6">Visit Date</th>
                      <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide w-1/4">Location</th>
                      <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide w-1/6">Status</th>
                      <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide w-1/6">Actions</th>
                    </tr>
                  </thead>
                  <tbody className="bg-white divide-y divide-gray-200">
                    {visits.map((visit) => (
                      <tr key={visit.id} className="hover:bg-gray-50 transition-colors">
                        <td className="px-6 py-4 w-1/4">
                          <div className="text-sm font-semibold text-gray-900 truncate" title={visit.property_title}>
                            {visit.property_title}
                          </div>
                          <div className="text-xs text-gray-500">
                            {visit.number_of_visitors} visitor(s)
                          </div>
                        </td>
                        <td className="px-6 py-4 w-1/6">
                          <div className="text-sm text-gray-900">
                            {new Date(visit.preferred_date).toLocaleDateString('en-IN')}
                          </div>
                          <div className="text-xs text-gray-500">
                            {visit.preferred_time}
                          </div>
                        </td>
                        <td className="px-6 py-4 w-1/4">
                          <div className="text-sm text-gray-900 truncate" title={visit.property_address}>
                            {visit.property_address}
                          </div>
                          <div className="text-xs text-gray-500 truncate" title={visit.property_city}>
                            {visit.property_city}
                          </div>
                        </td>
                        <td className="px-6 py-4 w-1/6">
                          <span className={`px-2 py-1 text-xs font-semibold rounded-lg w-fit ${getStatusColor(visit.status)}`}>
                            {visit.status === 'admin_approved' ? 'Pending Seller' :
                             visit.status === 'seller_approved' ? 'Approved' :
                             visit.status === 'seller_rejected' ? 'Rejected by Seller' :
                             visit.status === 'admin_rejected' ? 'Rejected by Admin' :
                             visit.status.charAt(0).toUpperCase() + visit.status.slice(1)}
                          </span>
                        </td>
                        <td className="px-6 py-4 w-1/6">
                          <div className="flex items-center gap-2">
                            <button
                              onClick={() => setSelectedVisit(visit)}
                              className="inline-flex items-center px-3 py-1.5 text-xs font-medium text-white bg-green-500 rounded-md hover:bg-green-600 transition-all duration-200"
                            >
                              View
                            </button>
                            {visit.video_call_enabled && visit.video_call_room_id && visit.status === 'seller_approved' && (
                              <a
                                href={`https://meet.jit.si/${visit.video_call_room_id}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="inline-flex items-center px-3 py-1.5 text-xs font-medium text-white bg-blue-500 rounded-md hover:bg-blue-600 transition-all duration-200"
                              >
                                Video Call
                              </a>
                            )}
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </div>
          )}
        </div>
      )}
      {/* Visit Details Modal */}
      {selectedVisit && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center p-4 z-50">
          <div 
            className="bg-white rounded-2xl w-full max-w-3xl flex flex-col border border-gray-200 overflow-hidden font-inter"
            style={{
              boxShadow: '0 25px 50px -12px rgba(0, 0, 0, 0.25), 0 0 0 1px rgba(0, 0, 0, 0.05), 0 10px 15px -3px rgba(0, 0, 0, 0.1)',
              maxHeight: '90vh'
            }}
          >
            {/* Header */}
            <div className="px-6 py-4 border-b border-gray-100">
              <div className="flex items-center justify-between">
                <div>
                  <h2 className="text-xl font-bold text-gray-900 tracking-tight">Visit Details</h2>
                  <p className="text-gray-600 text-sm mt-1 font-medium">Visit ID: {selectedVisit.id}</p>
                </div>
                <button
                  onClick={() => setSelectedVisit(null)}
                  className="p-2 hover:text-gray-700 transition-colors"
                >
                  <X className="w-5 h-5 text-gray-500" />
                </button>
              </div>
            </div>
            
            {/* Content */}
            <div 
              className="flex-1" 
              style={{ 
                minHeight: 0,
                overflowY: 'auto',
                scrollbarWidth: 'thin',
                scrollbarColor: '#cbd5e1 #f1f5f9'
              }}
            >
              <div className="p-6 space-y-6">
                <div className="bg-blue-50 p-4 rounded-lg border border-blue-100">
                  <h3 className="text-lg font-semibold text-gray-900 tracking-tight">{selectedVisit.property_title}</h3>
                  <p className="text-sm text-gray-600 mt-1 font-medium">{selectedVisit.property_address}, {selectedVisit.property_city}</p>
                </div>

                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <h4 className="text-base font-semibold text-gray-900 mb-3 tracking-tight">Visit Information</h4>
                    <div className="space-y-3">
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Preferred Date</label>
                        <p className="text-sm font-semibold text-gray-900">{new Date(selectedVisit.preferred_date).toLocaleDateString('en-IN')}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Preferred Time</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.preferred_time}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Number of Visitors</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.number_of_visitors}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Visit Mode</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.visit_mode}</p>
                      </div>
                      {selectedVisit.alternate_date && (
                        <div>
                          <label className="block text-sm font-medium text-gray-500 mb-1">Alternate Date</label>
                          <p className="text-sm font-semibold text-gray-900">{new Date(selectedVisit.alternate_date).toLocaleDateString('en-IN')}</p>
                        </div>
                      )}
                      {selectedVisit.video_call_enabled && (
                        <div>
                          <label className="block text-sm font-medium text-gray-500 mb-1">Video Call</label>
                          <div className="flex items-center gap-2">
                            <span className="text-sm font-semibold text-green-600">Available</span>
                            {selectedVisit.video_call_room_id && selectedVisit.status === 'seller_approved' && (
                              <a
                                href={`https://meet.jit.si/${selectedVisit.video_call_room_id}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="inline-flex items-center px-3 py-1 text-xs font-medium text-white bg-blue-500 rounded hover:bg-blue-600"
                              >
                                Join Video Call
                              </a>
                            )}
                          </div>
                        </div>
                      )}
                    </div>
                  </div>

                  <div>
                    <h4 className="text-base font-semibold text-gray-900 mb-3 tracking-tight">Contact Information</h4>
                    <div className="space-y-3">
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Full Name</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.full_name}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Mobile</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.mobile}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Email</label>
                        <p className="text-sm font-semibold text-gray-900 break-words">{selectedVisit.email}</p>
                      </div>
                      {selectedVisit.comments && (
                        <div>
                          <label className="block text-sm font-medium text-gray-500 mb-1">Comments</label>
                          <p className="text-sm text-gray-700 leading-relaxed">{selectedVisit.comments}</p>
                        </div>
                      )}
                    </div>
                  </div>
                </div>

                <div className="flex justify-between items-center pt-4 border-t border-gray-200">
                  <span className="text-sm font-medium text-gray-500">Scheduled on {new Date(selectedVisit.created_at).toLocaleDateString('en-IN')}</span>
                  <span className={`px-3 py-1.5 text-xs font-semibold rounded-lg ${getStatusColor(selectedVisit.status)}`}>
                    {selectedVisit.status === 'admin_approved' ? 'Pending Seller Approval' :
                     selectedVisit.status === 'seller_approved' ? 'Approved' :
                     selectedVisit.status === 'seller_rejected' ? 'Rejected by Seller' :
                     selectedVisit.status === 'admin_rejected' ? 'Rejected by Admin' :
                     selectedVisit.status.charAt(0).toUpperCase() + selectedVisit.status.slice(1)}
                  </span>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}