import React, { useState, useEffect } from "react";
import { Search, MapPin, Clock, Eye, Download, Trash2 } from "lucide-react";
import { Button } from "../ui/button";
import { searchHistoryAPI } from "../../api/searchHistory";
import { useNavigate } from "react-router-dom";

export function SearchHistory() {
  const [searchHistory, setSearchHistory] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const navigate = useNavigate();

  useEffect(() => {
    loadSearchHistory();
  }, []);

  const loadSearchHistory = async () => {
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id || user.id || localStorage.getItem('user_id');
      
      console.log('🔍 Loading search history for user:', userId);
      
      if (userId && userId !== "anonymous" && !userId.startsWith("guest_")) {
        // Load from backend API
        const history = await searchHistoryAPI.getUserHistory(userId, 20);
        console.log('✅ Backend search history loaded:', history);
        setSearchHistory(history || []);
      } else {
        console.log('❌ User not logged in, checking localStorage');
        // Fallback to localStorage for guest users
        const localHistory = JSON.parse(localStorage.getItem('search_history') || '[]');
        console.log('📱 Local search history:', localHistory);
        
        if (localHistory.length === 0) {
          // Show message that user needs to search first
          setSearchHistory([]);
        } else {
          setSearchHistory(localHistory);
        }
      }
    } catch (error) {
      console.error('❌ Error loading search history:', error);
      // Try localStorage as fallback
      const localHistory = JSON.parse(localStorage.getItem('search_history') || '[]');
      console.log('📱 Fallback to local history:', localHistory);
      setSearchHistory(localHistory || []);
    } finally {
      setLoading(false);
    }
  };


  const handleSearchAgain = (query: string, filters?: any) => {
    const searchParams = new URLSearchParams();
    if (query) searchParams.set('search', query);
    if (filters?.city) searchParams.set('city', filters.city);
    navigate(`/properties?${searchParams.toString()}`);
  };

  const handleClearAll = async () => {
    setSearchHistory([]);
    localStorage.removeItem('search_history');
    
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.id || user.user_id;
      if (userId) {
        await searchHistoryAPI.clearAllHistory(userId);
      }
    } catch (error) {
      console.error('Error clearing search history:', error);
    }
  };

  const formatTimestamp = (timestamp: string) => {
    const date = new Date(timestamp);
    return date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
  };
  



  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-semibold text-gray-900">Your Recent Searches</h3>
        <Button variant="outline" size="sm" onClick={handleClearAll}>
          <Trash2 className="w-4 h-4 mr-2" />
          Clear All
        </Button>
      </div>



      {/* Recent Searches */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="p-6">
          <div className="space-y-4">
            {loading ? (
              <div className="flex items-center justify-center py-8">
                <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-[#0056D2]"></div>
                <span className="ml-2 text-gray-500">Loading search history...</span>
              </div>
            ) : searchHistory.length > 0 ? (
              searchHistory.map((search, index) => (
                <div key={search.id || index} className="flex items-center gap-4 p-4 border rounded-lg hover:bg-gray-50">
                  <div className="flex-1">
                    <h4 className="font-semibold">{search.query || 'Filter Search'}</h4>
                    <p className="text-sm text-gray-600 flex items-center">
                      <MapPin className="w-3 h-3 mr-1" />
                      {search.location || 'All Cities'}
                    </p>
                    <div className="flex gap-2 mt-1">
                      {search.filters?.property_type && search.filters.property_type.length > 0 && (
                        <span className="text-xs bg-blue-100 px-2 py-1 rounded">
                          {search.filters.property_type.join(', ')}
                        </span>
                      )}
                      {search.filters?.bedrooms && (
                        <span className="text-xs bg-green-100 px-2 py-1 rounded">
                          {search.filters.bedrooms} BHK
                        </span>
                      )}
                      {search.filters?.city && (
                        <span className="text-xs bg-purple-100 px-2 py-1 rounded">
                          {search.filters.city}
                        </span>
                      )}
                    </div>
                  </div>
                  <div className="flex flex-col items-end gap-2">
                    <div className="text-sm text-gray-500">
                      <Clock className="w-4 h-4 inline mr-1" />
                      {formatTimestamp(search.timestamp)}
                    </div>
                    <Button 
                      size="sm" 
                      variant="outline"
                      onClick={() => handleSearchAgain(search.query, search.filters)}
                    >
                      <Search className="w-4 h-4 mr-1" />
                      Search Again
                    </Button>
                  </div>
                </div>
              ))
            ) : (
              <div className="text-center py-8">
                <Search className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                <p className="text-gray-500">No search history found</p>
                <p className="text-gray-400 text-sm">Start searching for properties on the main search page to see your history here</p>
                <div className="mt-4">
                  <Button 
                    onClick={() => navigate('/properties')}
                    className="bg-[#0056D2] text-white hover:bg-[#0056D2]/90"
                  >
                    Go to Property Search
                  </Button>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>



    </div>
  );
}