import React, { useState } from "react";
import { LifeBuoy, MessageSquare, Phone, Mail, FileText, Search, Send, Clock, CheckCircle } from "lucide-react";
import { Button } from "../ui/button";

export function SupportCenter() {
  const [activeTab, setActiveTab] = useState("faq");
  const [searchQuery, setSearchQuery] = useState("");
  const [newTicket, setNewTicket] = useState({
    subject: "",
    category: "",
    message: ""
  });

  const faqs = [
    {
      id: 1,
      question: "How do I schedule a property visit?",
      answer: "You can schedule a property visit by clicking on the 'Schedule Visit' button on any property listing. Choose your preferred date and time, and our agent will confirm the appointment."
    },
    {
      id: 2,
      question: "What documents do I need for property purchase?",
      answer: "You'll need ID proof, address proof, income documents, bank statements, and PAN card. Our team will guide you through the complete documentation process."
    },
    {
      id: 3,
      question: "How can I get a home loan?",
      answer: "We have partnerships with leading banks and financial institutions. Our loan experts can help you get pre-approved loans with competitive interest rates."
    },
    {
      id: 4,
      question: "Is there any brokerage fee?",
      answer: "Our brokerage structure is transparent and competitive. The exact fee depends on the property value and type. Contact our team for detailed information."
    }
  ];

  const [supportTickets, setSupportTickets] = useState([]);
  
  const loadUserTickets = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('http://localhost:8000/api/user-tickets/', {
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });
      
      if (response.ok) {
        const tickets = await response.json();
        const formattedTickets = tickets.map(ticket => ({
          id: `T${ticket.id.toString().padStart(3, '0')}`,
          subject: ticket.subject,
          status: ticket.status === 'open' ? 'open' : 'resolved',
          createdDate: ticket.created_at.split('T')[0],
          lastUpdate: ticket.updated_at.split('T')[0]
        }));
        setSupportTickets(formattedTickets);
      } else {
        console.error('Failed to load tickets:', response.status);
        setSupportTickets([]);
      }
    } catch (error) {
      console.error('Error loading tickets:', error);
      setSupportTickets([]);
    }
  };
  
  React.useEffect(() => {
    loadUserTickets();
  }, []);

  const handleTicketSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    try {
      const token = localStorage.getItem('access_token');
      
      const response = await fetch('http://localhost:8000/api/create-support-ticket/', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({
          subject: newTicket.subject,
          category: newTicket.category,
          message: newTicket.message
        })
      });
      
      if (response.ok) {
        alert('Ticket submitted successfully!');
        setNewTicket({ subject: "", category: "", message: "" });
        setActiveTab("tickets");
        loadUserTickets();
      } else {
        const errorText = await response.text();
        alert(`Failed to submit ticket: ${response.status} - ${errorText}`);
      }
    } catch (error) {
      console.error('Error submitting ticket:', error);
      alert(`Network error: ${error.message}. Please ensure Django server is running.`);
    }
  };

  return (
    <div className="space-y-6">
      {/* Contact Options */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 text-center">
          <Phone className="w-8 h-8 text-[#0056D2] mx-auto mb-3" />
          <h3 className="font-semibold text-gray-900 mb-2">Call Us</h3>
          <p className="text-sm text-gray-600 mb-3">Speak with our experts</p>
          <Button size="sm" className="bg-[#0056D2] hover:bg-[#0041A3]">
            +91 1800-123-4567
          </Button>
        </div>

        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 text-center">
          <Mail className="w-8 h-8 text-[#0056D2] mx-auto mb-3" />
          <h3 className="font-semibold text-gray-900 mb-2">Email Support</h3>
          <p className="text-sm text-gray-600 mb-3">Get help via email</p>
          <Button size="sm" className="bg-[#0056D2] hover:bg-[#0041A3]">
            support@nalindia.com
          </Button>
        </div>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6">
            <button
              onClick={() => setActiveTab("faq")}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === "faq"
                  ? "border-[#0056D2] text-[#0056D2]"
                  : "border-transparent text-gray-500 hover:text-gray-700"
              }`}
            >
              FAQ
            </button>
            <button
              onClick={() => setActiveTab("tickets")}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === "tickets"
                  ? "border-[#0056D2] text-[#0056D2]"
                  : "border-transparent text-gray-500 hover:text-gray-700"
              }`}
            >
              Support Tickets
            </button>
            <button
              onClick={() => setActiveTab("new-ticket")}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === "new-ticket"
                  ? "border-[#0056D2] text-[#0056D2]"
                  : "border-transparent text-gray-500 hover:text-gray-700"
              }`}
            >
              Create Ticket
            </button>
          </nav>
        </div>

        <div className="p-6">
          {/* FAQ Tab */}
          {activeTab === "faq" && (
            <div className="space-y-4">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400" />
                <input
                  type="text"
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  placeholder="Search frequently asked questions..."
                  className="w-full pl-10 pr-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                />
              </div>

              <div className="space-y-4">
                {faqs
                  .filter(faq => 
                    searchQuery === "" || 
                    faq.question.toLowerCase().includes(searchQuery.toLowerCase().split(' ')[0])
                  )
                  .map((faq) => (
                    <div key={faq.id} className="border border-gray-200 rounded-lg p-4">
                      <h3 className="font-semibold text-gray-900 mb-2">{faq.question}</h3>
                      <p className="text-gray-700 text-sm">{faq.answer}</p>
                    </div>
                  ))
                }
              </div>
            </div>
          )}

          {/* Support Tickets Tab */}
          {activeTab === "tickets" && (
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <h3 className="text-lg font-semibold text-gray-900">Your Support Tickets</h3>
                <Button 
                  size="sm" 
                  onClick={() => setActiveTab("new-ticket")}
                  className="bg-[#0056D2] hover:bg-[#0041A3]"
                >
                  Create New Ticket
                </Button>
              </div>

              <div className="space-y-3">
                {supportTickets.map((ticket) => (
                  <div key={ticket.id} className="border border-gray-200 rounded-lg p-4">
                    <div className="flex items-center justify-between mb-2">
                      <div className="flex items-center gap-3">
                        <span className="font-medium text-gray-900">#{ticket.id}</span>
                        <span className={`inline-flex items-center gap-1 px-2 py-1 rounded-full text-xs font-medium ${
                          ticket.status === 'open' 
                            ? 'bg-yellow-100 text-yellow-800' 
                            : 'bg-green-100 text-green-800'
                        }`}>
                          {ticket.status === 'open' ? <Clock className="w-3 h-3" /> : <CheckCircle className="w-3 h-3" />}
                          {ticket.status.charAt(0).toUpperCase() + ticket.status.slice(1)}
                        </span>
                      </div>
                      <Button size="sm" variant="outline">
                        View Details
                      </Button>
                    </div>
                    <h4 className="font-medium text-gray-900 mb-1">{ticket.subject}</h4>
                    <div className="text-sm text-gray-600">
                      <span>Created: {ticket.createdDate}</span>
                      <span className="mx-2">•</span>
                      <span>Last Update: {ticket.lastUpdate}</span>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}

          {/* Create Ticket Tab */}
          {activeTab === "new-ticket" && (
            <form onSubmit={handleTicketSubmit} className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Subject</label>
                <input
                  type="text"
                  value={newTicket.subject}
                  onChange={(e) => setNewTicket(prev => ({ ...prev, subject: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                  placeholder="Brief description of your issue"
                  required
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Category</label>
                <select
                  value={newTicket.category}
                  onChange={(e) => setNewTicket(prev => ({ ...prev, category: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                  required
                >
                  <option value="">Select a category</option>
                  <option value="property-inquiry">Property Inquiry</option>
                  <option value="technical-issue">Technical Issue</option>
                  <option value="payment-billing">Payment & Billing</option>
                  <option value="documentation">Documentation</option>
                  <option value="other">Other</option>
                </select>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Message</label>
                <textarea
                  value={newTicket.message}
                  onChange={(e) => setNewTicket(prev => ({ ...prev, message: e.target.value }))}
                  rows={6}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                  placeholder="Describe your issue in detail..."
                  required
                />
              </div>

              <div className="flex justify-end">
                <Button type="submit" className="bg-[#0056D2] hover:bg-[#0041A3]">
                  <Send className="w-4 h-4 mr-2" />
                  Submit Ticket
                </Button>
              </div>
            </form>
          )}
        </div>
      </div>

      {/* Knowledge Base */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Knowledge Base</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          <div className="p-4 border border-gray-200 rounded-lg hover:border-[#0056D2] cursor-pointer transition-colors">
            <FileText className="w-6 h-6 text-[#0056D2] mb-2" />
            <h4 className="font-medium text-gray-900 mb-1">Property Buying Guide</h4>
            <p className="text-sm text-gray-600">Complete guide to buying property in India</p>
          </div>
          
          <div className="p-4 border border-gray-200 rounded-lg hover:border-[#0056D2] cursor-pointer transition-colors">
            <FileText className="w-6 h-6 text-[#0056D2] mb-2" />
            <h4 className="font-medium text-gray-900 mb-1">Loan Process</h4>
            <p className="text-sm text-gray-600">Step-by-step home loan application process</p>
          </div>
          
          <div className="p-4 border border-gray-200 rounded-lg hover:border-[#0056D2] cursor-pointer transition-colors">
            <FileText className="w-6 h-6 text-[#0056D2] mb-2" />
            <h4 className="font-medium text-gray-900 mb-1">Legal Documentation</h4>
            <p className="text-sm text-gray-600">Important legal documents for property purchase</p>
          </div>
        </div>
      </div>
    </div>
  );
}