import React, { useState, useEffect } from "react";
import { Home, Calendar, MapPin, FileText, RefreshCw } from "lucide-react";

interface Tenancy {
  id: string;
  property_id: number;
  tenancy_type: "RENT" | "LEASE";
  start_date: string;
  status: "ACTIVE" | "ENDED";
  created_at: string;
  property: {
    title: string;
    address: string;
    monthly_rent: string;
  };
}

export function TenantView() {
  const [tenancies, setTenancies] = useState<Tenancy[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchTenancies();
  }, []);

  const fetchTenancies = async () => {
    try {
      const user = JSON.parse(localStorage.getItem("user") || "{}");
      
      console.log('🔍 User data from localStorage:', user);
      
      // Check if user ID exists (try both id and user_id)
      const userId = user.id || user.user_id;
      if (!userId) {
        console.warn('❌ No user ID found in localStorage');
        setLoading(false);
        return;
      }
      
      console.log('📞 Fetching tenancies for user ID:', userId);
      
      const API_BASE =
        import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
      const url = `${API_BASE}/property-management/tenancies/tenant/${userId}`;
      
      console.log('🌐 API URL:', url);
      
      const response = await fetch(url, {
        headers: {
          Authorization: `Bearer ${localStorage.getItem("access_token") || localStorage.getItem("token")}`,
          'Content-Type': 'application/json'
        },
      });

      console.log('📡 Response status:', response.status);
      
      if (response.ok) {
        const data = await response.json();
        console.log('✅ Tenancies data received:', data);
        setTenancies(data.tenancies || data || []);
      } else {
        const errorText = await response.text();
        console.error('❌ Failed to fetch tenancies:', response.status, errorText);
      }
    } catch (error) {
      console.error("💥 Error fetching tenancies:", error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse">
          <div className="h-8 bg-gray-200 rounded w-1/4 mb-6"></div>
          <div className="space-y-4">
            {[1, 2].map((i) => (
              <div key={i} className="h-48 bg-gray-200 rounded-lg"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  if (tenancies.length === 0) {
    return (
      <div className="p-6">
        <h2 className="text-2xl font-bold mb-6 text-gray-900">My Rentals</h2>
        <div className="text-center py-12">
          <Home className="mx-auto h-12 w-12 text-gray-400 mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">
            No Active Rentals
          </h3>
          <p className="text-gray-600">
            You don't have any active rental properties yet. Once a property
            owner converts your rental enquiry, it will appear here.
          </p>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      <div className="flex items-center justify-between mb-6">
        <h2 className="text-2xl font-bold text-gray-900">My Rentals</h2>
        <div className="flex items-center gap-4">
          <button
            onClick={fetchTenancies}
            className="flex items-center px-3 py-2 text-sm bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            <RefreshCw className="h-4 w-4 mr-2" />
            Refresh
          </button>
          <span className="text-sm text-gray-500">
            {tenancies.length} active rental{tenancies.length !== 1 ? "s" : ""}
          </span>
        </div>
      </div>

      <div className="grid gap-6">
        {tenancies.map((tenancy) => (
          <TenancyCard key={tenancy.id} tenancy={tenancy} />
        ))}
      </div>
    </div>
  );
}

interface TenancyCardProps {
  tenancy: Tenancy;
}

function TenancyCard({ tenancy }: TenancyCardProps) {
  const startDate = new Date(tenancy.start_date);
  const monthsActive = Math.floor(
    (Date.now() - startDate.getTime()) / (1000 * 60 * 60 * 24 * 30),
  );

  return (
    <div className="bg-white border rounded-lg shadow-sm hover:shadow-md transition-shadow">
      <div className="p-6">
        <div className="flex justify-between items-start mb-4">
          <div className="flex-1">
            <h3 className="text-xl font-semibold text-gray-900 mb-2">
              {tenancy.property.title}
            </h3>
            <div className="flex items-center text-gray-600 mb-2">
              <MapPin className="h-4 w-4 mr-2" />
              <span className="text-sm">{tenancy.property.address}</span>
            </div>
          </div>

          <span
            className={`px-3 py-1 rounded-full text-sm font-medium ${
              tenancy.status === "ACTIVE"
                ? "bg-green-100 text-green-800"
                : "bg-gray-100 text-gray-800"
            }`}
          >
            {tenancy.status}
          </span>
        </div>

        <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
          <div className="bg-gray-50 p-4 rounded-lg">
            <div className="flex items-center mb-2">
              <Calendar className="h-4 w-4 text-gray-500 mr-2" />
              <span className="text-sm font-medium text-gray-700">
                Start Date
              </span>
            </div>
            <p className="text-lg font-semibold text-gray-900">
              {startDate.toLocaleDateString()}
            </p>
          </div>

          <div className="bg-gray-50 p-4 rounded-lg">
            <div className="flex items-center mb-2">
              <Home className="h-4 w-4 text-gray-500 mr-2" />
              <span className="text-sm font-medium text-gray-700">
                Tenancy Type
              </span>
            </div>
            <p className="text-lg font-semibold text-gray-900">
              {tenancy.tenancy_type}
            </p>
          </div>

          <div className="bg-gray-50 p-4 rounded-lg">
            <div className="flex items-center mb-2">
              <span className="text-sm font-medium text-gray-700">
                Monthly Rent
              </span>
            </div>
            <p className="text-lg font-semibold text-gray-900">
              {tenancy.property.monthly_rent || "N/A"}
            </p>
          </div>

          <div className="bg-gray-50 p-4 rounded-lg">
            <div className="flex items-center mb-2">
              <span className="text-sm font-medium text-gray-700">
                Duration
              </span>
            </div>
            <p className="text-lg font-semibold text-gray-900">
              {monthsActive} month{monthsActive !== 1 ? "s" : ""}
            </p>
          </div>
        </div>

        <div className="flex gap-3">
          <button
            onClick={() =>
              window.open(`/property/${tenancy.property_id}`, "_blank")
            }
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            <Home className="h-4 w-4 mr-2" />
            View Property
          </button>

          <button className="flex items-center px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors">
            <FileText className="h-4 w-4 mr-2" />
            Rental Agreement
          </button>
        </div>
      </div>
    </div>
  );
}
