import React from 'react';
import { Card, CardContent } from '../ui/card';
import { Badge } from '../ui/badge';

interface TrustIndicatorsProps {
  indicators: {
    reraApproved: boolean;
    clearTitle: boolean;
    verifiedSeller: boolean;
    bankApproved: boolean;
    legalVerified: boolean;
    builderRating: number;
    platformVerified: boolean;
  };
}

const TrustIndicators: React.FC<TrustIndicatorsProps> = ({ indicators }) => {
  const trustItems = [
    {
      key: 'reraApproved',
      label: 'RERA Approved',
      icon: '🏛️',
      status: indicators.reraApproved,
      description: 'Registered with Real Estate Regulatory Authority'
    },
    {
      key: 'clearTitle',
      label: 'Clear Title',
      icon: '📋',
      status: indicators.clearTitle,
      description: 'Property has clear legal title'
    },
    {
      key: 'verifiedSeller',
      label: 'Verified Seller',
      icon: '✅',
      status: indicators.verifiedSeller,
      description: 'Seller identity verified by platform'
    },
    {
      key: 'bankApproved',
      label: 'Bank Approved',
      icon: '🏦',
      status: indicators.bankApproved,
      description: 'Pre-approved for home loans'
    },
    {
      key: 'legalVerified',
      label: 'Legal Verified',
      icon: '⚖️',
      status: indicators.legalVerified,
      description: 'Legal documents verified by experts'
    },
    {
      key: 'platformVerified',
      label: 'Platform Verified',
      icon: '🛡️',
      status: indicators.platformVerified,
      description: 'Verified by NAL quality standards'
    }
  ];

  const verifiedCount = trustItems.filter(item => item.status).length;
  const trustPercentage = (verifiedCount / trustItems.length) * 100;

  const getTrustColor = () => {
    if (trustPercentage >= 80) return 'text-green-600 bg-green-50';
    if (trustPercentage >= 60) return 'text-yellow-600 bg-yellow-50';
    return 'text-red-600 bg-red-50';
  };

  return (
    <Card>
      <CardContent className="p-4">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold flex items-center space-x-2">
            <span>🔒</span>
            <span>Trust Indicators</span>
          </h3>
          <Badge className={`${getTrustColor()} px-3 py-1`}>
            {verifiedCount}/{trustItems.length} Verified
          </Badge>
        </div>

        <div className="space-y-3">
          {trustItems.map((item) => (
            <div key={item.key} className="flex items-center justify-between p-2 rounded-lg hover:bg-gray-50">
              <div className="flex items-center space-x-3">
                <span className="text-lg">{item.icon}</span>
                <div>
                  <div className="font-medium text-sm">{item.label}</div>
                  <div className="text-xs text-gray-500">{item.description}</div>
                </div>
              </div>
              <div>
                {item.status ? (
                  <Badge className="bg-green-100 text-green-800 text-xs">
                    ✓ Verified
                  </Badge>
                ) : (
                  <Badge className="bg-red-100 text-red-800 text-xs">
                    ✗ Not Verified
                  </Badge>
                )}
              </div>
            </div>
          ))}
        </div>

        {/* Builder Rating */}
        <div className="mt-4 p-3 bg-gray-50 rounded-lg">
          <div className="flex items-center justify-between">
            <span className="text-sm font-medium">Builder Rating</span>
            <div className="flex items-center space-x-2">
              <div className="flex">
                {[1, 2, 3, 4, 5].map((star) => (
                  <span
                    key={star}
                    className={`text-lg ${
                      star <= indicators.builderRating ? 'text-yellow-400' : 'text-gray-300'
                    }`}
                  >
                    ⭐
                  </span>
                ))}
              </div>
              <span className="text-sm font-semibold">{indicators.builderRating}/5</span>
            </div>
          </div>
        </div>

        {/* Trust Score Summary */}
        <div className="mt-4 p-3 bg-blue-50 rounded-lg border border-blue-200">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-semibold text-blue-800">Overall Trust Score</span>
            <span className="text-lg font-bold text-blue-600">{Math.round(trustPercentage)}%</span>
          </div>
          <div className="w-full bg-blue-200 rounded-full h-2">
            <div 
              className="bg-blue-600 h-2 rounded-full transition-all duration-500"
              style={{ width: `${trustPercentage}%` }}
            ></div>
          </div>
          <div className="text-xs text-blue-700 mt-1 text-center">
            {trustPercentage >= 80 && "Highly Trustworthy"}
            {trustPercentage >= 60 && trustPercentage < 80 && "Moderately Trustworthy"}
            {trustPercentage < 60 && "Requires Caution"}
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

export default TrustIndicators;