import React, { useState, useEffect } from 'react';
import { QrCode, Smartphone, Clock, CheckCircle, XCircle, RefreshCw } from 'lucide-react';

interface UPIPaymentProps {
  orderId: string;
  amount: number;
  onSuccess: () => void;
  onCancel: () => void;
}

interface UPIQRData {
  order_id: string;
  qr_code_data: string;
  upi_intent_url: string;
  amount: number;
  expires_at: string;
}

export function UPIPayment({ orderId, amount, onSuccess, onCancel }: UPIPaymentProps) {
  const [qrData, setQrData] = useState<UPIQRData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [paymentStatus, setPaymentStatus] = useState<'pending' | 'checking' | 'success' | 'failed'>('pending');
  const [timeRemaining, setTimeRemaining] = useState(900); // 15 minutes in seconds

  useEffect(() => {
    generateQR();
  }, [orderId]);

  useEffect(() => {
    if (qrData && timeRemaining > 0) {
      const timer = setInterval(() => {
        setTimeRemaining(prev => prev - 1);
      }, 1000);
      return () => clearInterval(timer);
    }
  }, [qrData, timeRemaining]);

  useEffect(() => {
    if (paymentStatus === 'pending' && qrData) {
      const pollInterval = setInterval(() => {
        checkPaymentStatus();
      }, 3000); // Check every 3 seconds

      return () => clearInterval(pollInterval);
    }
  }, [paymentStatus, qrData]);

  const generateQR = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');

      const response = await fetch('/api/payments/upi/qr', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          order_id: orderId,
          amount: amount,
          payment_gateway: 'razorpay',
          expiry_minutes: 15
        })
      });

      if (!response.ok) {
        throw new Error('Failed to generate UPI QR code');
      }

      const data = await response.json();
      setQrData(data);
      setError(null);
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to generate QR code');
    } finally {
      setLoading(false);
    }
  };

  const checkPaymentStatus = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await fetch(`/api/payments/${orderId}`, {
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });

      if (response.ok) {
        const payment = await response.json();
        if (payment.status === 'success') {
          setPaymentStatus('success');
          setTimeout(() => onSuccess(), 2000);
        } else if (payment.status === 'failed') {
          setPaymentStatus('failed');
        }
      }
    } catch (err) {
      console.error('Failed to check payment status:', err);
    }
  };

  const handleUPIIntent = () => {
    if (qrData?.upi_intent_url) {
      window.location.href = qrData.upi_intent_url;
    }
  };

  const formatTime = (seconds: number) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center p-12">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-6 bg-red-50 border border-red-200 rounded-lg">
        <div className="flex items-center gap-3 mb-4">
          <XCircle className="w-6 h-6 text-red-600" />
          <h3 className="font-semibold text-red-900">Failed to Generate QR Code</h3>
        </div>
        <p className="text-red-700 mb-4">{error}</p>
        <div className="flex gap-3">
          <button
            onClick={generateQR}
            className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors flex items-center gap-2"
          >
            <RefreshCw className="w-4 h-4" />
            Retry
          </button>
          <button
            onClick={onCancel}
            className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
          >
            Cancel
          </button>
        </div>
      </div>
    );
  }

  if (paymentStatus === 'success') {
    return (
      <div className="p-8 bg-green-50 border border-green-200 rounded-lg text-center">
        <CheckCircle className="w-16 h-16 text-green-600 mx-auto mb-4" />
        <h3 className="text-xl font-bold text-green-900 mb-2">Payment Successful!</h3>
        <p className="text-green-700">Your payment has been received and confirmed.</p>
      </div>
    );
  }

  if (paymentStatus === 'failed') {
    return (
      <div className="p-8 bg-red-50 border border-red-200 rounded-lg text-center">
        <XCircle className="w-16 h-16 text-red-600 mx-auto mb-4" />
        <h3 className="text-xl font-bold text-red-900 mb-2">Payment Failed</h3>
        <p className="text-red-700 mb-4">The payment could not be completed. Please try again.</p>
        <button
          onClick={onCancel}
          className="px-6 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors"
        >
          Close
        </button>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="text-center">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Pay with UPI</h2>
        <p className="text-3xl font-bold text-blue-600">₹{amount.toLocaleString('en-IN')}</p>
      </div>

      {/* Timer */}
      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
        <div className="flex items-center justify-center gap-2 text-yellow-800">
          <Clock className="w-5 h-5" />
          <span className="font-semibold">
            QR Code expires in: {formatTime(timeRemaining)}
          </span>
        </div>
      </div>

      {/* QR Code */}
      {qrData && (
        <div className="bg-white border-2 border-gray-200 rounded-lg p-6">
          <div className="text-center mb-4">
            <h3 className="font-semibold text-gray-900 mb-2">Scan QR Code</h3>
            <p className="text-sm text-gray-600">Use any UPI app to scan and pay</p>
          </div>

          <div className="flex justify-center mb-6">
            <div className="bg-white p-4 rounded-lg border-4 border-blue-500 shadow-lg">
              <img
                src={qrData.qr_code_data}
                alt="UPI QR Code"
                className="w-64 h-64"
              />
            </div>
          </div>

          {/* UPI Apps */}
          <div className="text-center mb-4">
            <p className="text-sm text-gray-600 mb-3">Supported UPI Apps</p>
            <div className="flex justify-center gap-4">
              <div className="text-center">
                <div className="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center mb-1">
                  <span className="text-xs font-bold text-purple-700">PhonePe</span>
                </div>
              </div>
              <div className="text-center">
                <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mb-1">
                  <span className="text-xs font-bold text-blue-700">Paytm</span>
                </div>
              </div>
              <div className="text-center">
                <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mb-1">
                  <span className="text-xs font-bold text-green-700">GPay</span>
                </div>
              </div>
            </div>
          </div>

          {/* Mobile Intent Button */}
          <div className="border-t pt-4">
            <button
              onClick={handleUPIIntent}
              className="w-full px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors flex items-center justify-center gap-2 font-semibold"
            >
              <Smartphone className="w-5 h-5" />
              Pay with UPI App
            </button>
            <p className="text-xs text-gray-500 text-center mt-2">
              Opens your UPI app directly (mobile only)
            </p>
          </div>
        </div>
      )}

      {/* Status */}
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
        <div className="flex items-center gap-3">
          <div className="animate-pulse">
            <div className="w-3 h-3 bg-blue-600 rounded-full"></div>
          </div>
          <span className="text-blue-800 font-medium">
            Waiting for payment confirmation...
          </span>
        </div>
      </div>

      {/* Cancel Button */}
      <div className="text-center">
        <button
          onClick={onCancel}
          className="px-6 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
        >
          Cancel Payment
        </button>
      </div>
    </div>
  );
}
