import React, { useState, useEffect } from 'react';
import { calculatePrice } from '../../hooks/usePromotions';
import { PriceCalculation } from '../../types/promotions';

interface PricingPlan {
  id: number;
  name: string;
  price: number;
  billing_cycle: string;
  is_recommended: boolean;
  limits_json: Record<string, any>;
}

interface Props {
  userId: number;
  plans: PricingPlan[];
}

export const UserPricingWithCoupons: React.FC<Props> = ({ userId, plans }) => {
  const [selectedPlan, setSelectedPlan] = useState<number | null>(null);
  const [couponCode, setCouponCode] = useState('');
  const [priceCalc, setPriceCalc] = useState<PriceCalculation | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');

  const handleCalculatePrice = async (planId: number, originalPrice: number) => {
    setLoading(true);
    setError('');
    try {
      const result = await calculatePrice(planId, userId, originalPrice, couponCode || undefined);
      setPriceCalc(result);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  const handlePlanSelect = (plan: PricingPlan) => {
    setSelectedPlan(plan.id);
    handleCalculatePrice(plan.id, plan.price);
  };

  const handleApplyCoupon = () => {
    if (selectedPlan) {
      const plan = plans.find(p => p.id === selectedPlan);
      if (plan) handleCalculatePrice(plan.id, plan.price);
    }
  };

  return (
    <div className="p-6">
      <h1 className="text-3xl font-bold mb-8 text-center">Choose Your Plan</h1>

      <div className="grid md:grid-cols-3 gap-6 mb-8">
        {plans.map((plan) => (
          <div
            key={plan.id}
            className={`bg-white rounded-lg shadow-lg p-6 cursor-pointer transition-all ${
              selectedPlan === plan.id ? 'ring-2 ring-blue-600' : ''
            } ${plan.is_recommended ? 'border-2 border-blue-600' : ''}`}
            onClick={() => handlePlanSelect(plan)}
          >
            {plan.is_recommended && (
              <div className="bg-blue-600 text-white text-xs font-bold px-3 py-1 rounded-full inline-block mb-3">
                RECOMMENDED
              </div>
            )}
            <h3 className="text-xl font-bold mb-2">{plan.name}</h3>
            <div className="text-3xl font-bold mb-4">
              ₹{plan.price}
              <span className="text-sm text-gray-500 font-normal">/{plan.billing_cycle}</span>
            </div>
            <ul className="space-y-2 text-sm">
              {Object.entries(plan.limits_json).map(([key, value]) => (
                <li key={key} className="flex items-center">
                  <svg className="w-4 h-4 mr-2 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                    <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                  </svg>
                  {key.replace(/_/g, ' ')}: {typeof value === 'boolean' ? (value ? 'Yes' : 'No') : value}
                </li>
              ))}
            </ul>
          </div>
        ))}
      </div>

      {selectedPlan && (
        <div className="max-w-2xl mx-auto bg-white rounded-lg shadow-lg p-6">
          <h2 className="text-xl font-bold mb-4">Apply Coupon Code</h2>
          
          <div className="flex gap-2 mb-6">
            <input
              type="text"
              value={couponCode}
              onChange={(e) => setCouponCode(e.target.value.toUpperCase())}
              placeholder="Enter coupon code"
              className="flex-1 border rounded px-4 py-2 uppercase"
            />
            <button
              onClick={handleApplyCoupon}
              disabled={loading}
              className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 disabled:bg-gray-400"
            >
              {loading ? 'Applying...' : 'Apply'}
            </button>
          </div>

          {error && (
            <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded mb-4">
              {error}
            </div>
          )}

          {priceCalc && (
            <div className="border-t pt-4">
              <div className="space-y-3">
                <div className="flex justify-between text-lg">
                  <span>Original Price:</span>
                  <span>₹{priceCalc.original_price.toFixed(2)}</span>
                </div>
                
                {priceCalc.discount_amount > 0 && (
                  <>
                    <div className="flex justify-between text-lg text-green-600">
                      <span>
                        Discount ({priceCalc.discount_source === 'coupon' ? 'Coupon' : 
                                  priceCalc.discount_source === 'scholarship' ? 'Scholarship' : 
                                  'Auto Discount'}):
                      </span>
                      <span>-₹{priceCalc.discount_amount.toFixed(2)}</span>
                    </div>
                    
                    {priceCalc.discount_details && (
                      <div className="text-sm text-gray-600 bg-green-50 p-3 rounded">
                        {priceCalc.discount_source === 'coupon' && (
                          <p>✓ Coupon "{priceCalc.discount_details.coupon_code}" applied successfully!</p>
                        )}
                        {priceCalc.discount_source === 'scholarship' && (
                          <p>✓ Scholarship benefit applied!</p>
                        )}
                        {priceCalc.discount_source === 'auto_discount' && (
                          <p>✓ {priceCalc.discount_details.rule_name} applied automatically!</p>
                        )}
                      </div>
                    )}
                  </>
                )}
                
                <div className="flex justify-between text-2xl font-bold border-t pt-3">
                  <span>Final Price:</span>
                  <span className="text-blue-600">₹{priceCalc.final_price.toFixed(2)}</span>
                </div>
                
                {priceCalc.discount_amount > 0 && (
                  <div className="text-center text-green-600 font-semibold">
                    You save ₹{priceCalc.discount_amount.toFixed(2)}!
                  </div>
                )}
              </div>

              <button className="w-full bg-blue-600 text-white py-3 rounded-lg mt-6 hover:bg-blue-700 font-semibold">
                Subscribe Now
              </button>
            </div>
          )}
        </div>
      )}
    </div>
  );
};
