import React, { useState } from 'react';
import { useActiveScholarships, useMyScholarshipApplications, useMyActiveScholarship } from '../../hooks/usePromotions';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:8090';

const getAuthHeaders = () => {
  const token = localStorage.getItem('access_token');
  return {
    'Content-Type': 'application/json',
    'Authorization': token ? `Bearer ${token}` : ''
  };
};

interface Props {
  userId: number;
  userType: string;
}

export const UserScholarships: React.FC<Props> = ({ userId, userType }) => {
  const { programs, loading: programsLoading } = useActiveScholarships();
  const { applications, refetch: refetchApps } = useMyScholarshipApplications(userId);
  const { scholarship: activeScholarship } = useMyActiveScholarship(userId);
  const [showModal, setShowModal] = useState(false);
  const [selectedProgram, setSelectedProgram] = useState<any>(null);
  const [applicationData, setApplicationData] = useState<Record<string, any>>({});

  const handleApply = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      await fetch(`${API_URL}/api/user/scholarships/apply/`, {
        method: 'POST',
        headers: getAuthHeaders(),
        body: JSON.stringify({
          program_id: selectedProgram.id,
          user_id: userId,
          user_type: userType,
          application_data: applicationData
        })
      });
      setShowModal(false);
      setApplicationData({});
      refetchApps();
      alert('Application submitted successfully!');
    } catch (error) {
      console.error('Error submitting application:', error);
    }
  };

  const openApplicationModal = (program: any) => {
    setSelectedProgram(program);
    setShowModal(true);
  };

  if (programsLoading) return <div className="p-6">Loading...</div>;

  return (
    <div className="p-6">
      <h1 className="text-2xl font-bold mb-6">Scholarship Programs</h1>

      {activeScholarship && (
        <div className="bg-green-50 border border-green-200 rounded-lg p-4 mb-6">
          <div className="flex items-center gap-2 mb-2">
            <svg className="w-5 h-5 text-green-600" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
            </svg>
            <h3 className="font-semibold text-green-800">Active Scholarship</h3>
          </div>
          <p className="text-green-700">
            You have an active scholarship: {activeScholarship.program_name}
            {activeScholarship.benefit_type === 'free' && ' (Free access)'}
            {activeScholarship.benefit_type === 'percentage' && ` (${activeScholarship.benefit_value}% discount)`}
            {activeScholarship.benefit_type === 'fixed' && ` (₹${activeScholarship.benefit_value} discount)`}
          </p>
          {activeScholarship.expires_at && (
            <p className="text-sm text-green-600 mt-1">
              Valid until: {new Date(activeScholarship.expires_at).toLocaleDateString()}
            </p>
          )}
        </div>
      )}

      <div className="grid md:grid-cols-2 gap-6 mb-8">
        {programs.map((program) => (
          <div key={program.id} className="bg-white rounded-lg shadow-lg p-6">
            <div className="flex items-start justify-between mb-3">
              <div>
                <h3 className="text-lg font-bold">{program.name}</h3>
                <span className="text-xs bg-purple-100 text-purple-800 px-2 py-1 rounded font-mono">
                  {program.code}
                </span>
              </div>
              <div className="text-right">
                {program.benefit_type === 'free' && (
                  <div className="text-2xl font-bold text-green-600">FREE</div>
                )}
                {program.benefit_type === 'percentage' && (
                  <div className="text-2xl font-bold text-blue-600">{program.benefit_value}% OFF</div>
                )}
                {program.benefit_type === 'fixed' && (
                  <div className="text-2xl font-bold text-blue-600">₹{program.benefit_value} OFF</div>
                )}
              </div>
            </div>
            
            <p className="text-gray-600 text-sm mb-4">{program.description}</p>
            
            <div className="space-y-2 text-sm mb-4">
              <div className="flex justify-between">
                <span className="text-gray-600">Duration:</span>
                <span className="font-medium">{program.duration_months} months</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Spots Available:</span>
                <span className="font-medium">
                  {program.max_beneficiaries 
                    ? `${program.max_beneficiaries - program.current_beneficiaries} / ${program.max_beneficiaries}`
                    : 'Unlimited'}
                </span>
              </div>
            </div>

            <div className="border-t pt-4">
              <p className="text-xs text-gray-500 mb-3">Eligibility Criteria:</p>
              <div className="text-xs bg-gray-50 p-3 rounded">
                {Object.entries(program.eligibility_criteria).map(([key, value]) => (
                  <div key={key} className="flex justify-between mb-1">
                    <span className="text-gray-600">{key.replace(/_/g, ' ')}:</span>
                    <span className="font-medium">{String(value)}</span>
                  </div>
                ))}
              </div>
            </div>

            <button
              onClick={() => openApplicationModal(program)}
              className="w-full mt-4 bg-blue-600 text-white py-2 rounded hover:bg-blue-700"
              disabled={applications.some(app => app.program_id === program.id && app.status === 'pending')}
            >
              {applications.some(app => app.program_id === program.id && app.status === 'pending')
                ? 'Application Pending'
                : 'Apply Now'}
            </button>
          </div>
        ))}
      </div>

      {applications.length > 0 && (
        <div className="bg-white rounded-lg shadow p-6">
          <h2 className="text-xl font-bold mb-4">My Applications</h2>
          <div className="space-y-3">
            {applications.map((app) => (
              <div key={app.id} className="flex justify-between items-center border-b pb-3">
                <div>
                  <p className="font-medium">{app.program_name}</p>
                  <p className="text-sm text-gray-600">
                    Applied: {app.created_at ? new Date(app.created_at).toLocaleDateString() : '-'}
                  </p>
                </div>
                <span className={`px-3 py-1 rounded text-sm ${
                  app.status === 'approved' ? 'bg-green-100 text-green-800' :
                  app.status === 'rejected' ? 'bg-red-100 text-red-800' :
                  'bg-yellow-100 text-yellow-800'
                }`}>
                  {app.status}
                </span>
              </div>
            ))}
          </div>
        </div>
      )}

      {showModal && selectedProgram && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 w-full max-w-lg">
            <h2 className="text-xl font-bold mb-4">Apply for {selectedProgram.name}</h2>
            <form onSubmit={handleApply} className="space-y-4">
              <div>
                <label className="block text-sm font-medium mb-1">Why do you qualify?</label>
                <textarea
                  value={applicationData.reason || ''}
                  onChange={(e) => setApplicationData({ ...applicationData, reason: e.target.value })}
                  className="w-full border rounded px-3 py-2"
                  rows={4}
                  required
                  placeholder="Explain why you meet the eligibility criteria..."
                />
              </div>
              <div>
                <label className="block text-sm font-medium mb-1">Additional Information</label>
                <textarea
                  value={applicationData.additional_info || ''}
                  onChange={(e) => setApplicationData({ ...applicationData, additional_info: e.target.value })}
                  className="w-full border rounded px-3 py-2"
                  rows={3}
                  placeholder="Any additional information to support your application..."
                />
              </div>
              <div className="flex justify-end space-x-2 pt-4">
                <button
                  type="button"
                  onClick={() => { setShowModal(false); setApplicationData({}); }}
                  className="px-4 py-2 border rounded hover:bg-gray-50"
                >
                  Cancel
                </button>
                <button type="submit" className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700">
                  Submit Application
                </button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
};
