import { useState, useEffect } from 'react';
import { Heart } from 'lucide-react';
import { wishlistAPI, WishlistProperty } from '../../api/user/wishlistApi';
import { PropertyCard } from '../PropertyCard';
import { getUserId } from '../../utils/userUtils';

export function WishlistManagement() {
  const [wishlistProperties, setWishlistProperties] = useState<WishlistProperty[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchWishlistProperties = async () => {
      try {
        const userId = getUserId();
        if (!userId) return;
        
        const wishlistData = await wishlistAPI.getWishlist(userId);
        
        // Fetch full property details for each wishlist property
        const propertiesWithDetails = await Promise.all(
          wishlistData.map(async (wishlist) => {
            try {
              const propertyResponse = await fetch(`http://localhost:8090/api/properties/${wishlist.property_id}`);
              if (propertyResponse.ok) {
                const propertyData = await propertyResponse.json();
                return {
                  ...wishlist,
                  property: {
                    id: propertyData.id,
                    title: propertyData.title,
                    location: `${propertyData.locality}, ${propertyData.city}`,
                    beds: propertyData.bedrooms,
                    baths: propertyData.bathrooms,
                    area: `${propertyData.built_up_area} sq ft`,
                    price: `₹${propertyData.sale_price || propertyData.rent_price}`,
                    imageUrl: propertyData.primary_image || propertyData.image_urls?.[0] || propertyData.photo || null,
                    isUrgent: propertyData.is_urgent,
                    isBidding: propertyData.is_bidding
                  }
                };
              }
              return wishlist;
            } catch (error) {
              console.error(`Error fetching property ${wishlist.property_id}:`, error);
              return wishlist;
            }
          })
        );
        
        setWishlistProperties(propertiesWithDetails);
      } catch (error) {
        console.error('Error fetching wishlist properties:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchWishlistProperties();
  }, []);

  if (loading) {
    return (
      <div className="space-y-6">
        <h2 className="text-2xl font-bold text-gray-900">Wishlist/Favorites</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {[1, 2, 3].map((i) => (
            <div key={i} className="bg-white rounded-lg shadow-sm border border-gray-200 animate-pulse">
              <div className="h-48 bg-gray-200 rounded-t-lg"></div>
              <div className="p-4 space-y-3">
                <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                <div className="h-3 bg-gray-200 rounded w-1/2"></div>
                <div className="h-3 bg-gray-200 rounded w-1/4"></div>
              </div>
            </div>
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div>
        <h2 className="text-2xl font-bold text-gray-900">Wishlist/Favorites</h2>
        <p className="text-gray-600">Properties you've added to your wishlist</p>
      </div>

      {wishlistProperties.length === 0 ? (
        <div className="text-center py-12">
          <Heart className="w-16 h-16 text-gray-300 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No wishlist properties</h3>
          <p className="text-gray-600">Start exploring properties and add your favorites to wishlist</p>
        </div>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {wishlistProperties.map((wishlistProp) => (
            <PropertyCard
              key={wishlistProp.id}
              id={wishlistProp.property_id}
              title={wishlistProp.property?.title || `Property #${wishlistProp.property_id}`}
              location={wishlistProp.property?.location || 'Location not available'}
              beds={wishlistProp.property?.beds || 0}
              baths={wishlistProp.property?.baths || 0}
              area={wishlistProp.property?.area || 'N/A'}
              price={wishlistProp.property?.price || 'Price on request'}
              imageUrl={wishlistProp.property?.imageUrl}
              isUrgent={wishlistProp.property?.isUrgent}
              isBidding={wishlistProp.property?.isBidding}
              isFavorited={true}
            />
          ))}
        </div>
      )}
    </div>
  );
}