import React, { useState } from 'react';
import { CheckCircle, Upload, ArrowRight, ArrowLeft, AlertCircle } from 'lucide-react';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Label } from '../ui/label';
import { Card } from '../ui/card';
import { safeToast } from '../../utils/toastUtils';
import axios from 'axios';

const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export const BuyerKYCForm: React.FC = () => {
  const [currentStep, setCurrentStep] = useState(1);
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState({
    full_name: '',
    pan_number: '',
    date_of_birth: '',
    phone_number: '',
    email: '',
    address: '',
    city: '',
    state: '',
    pincode: '',
    aadhaar_number: '',
    passport_number: '',
    pan_document_url: '',
    aadhaar_document_url: '',
    passport_document_url: '',
    photo_url: '',
    bank_statement_url: '',
    income_proof_url: ''
  });

  const getToken = () => localStorage.getItem('token');

  const validateStep1 = () => {
    const { full_name, pan_number, date_of_birth, phone_number, email } = formData;
    if (!full_name || full_name.length < 2) {
      safeToast.error('Full name is required');
      return false;
    }
    if (!/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/.test(pan_number.toUpperCase())) {
      safeToast.error('Invalid PAN format (e.g., ABCDE1234F)');
      return false;
    }
    if (!date_of_birth) {
      safeToast.error('Date of birth is required');
      return false;
    }
    if (!/^[0-9]{10}$/.test(phone_number)) {
      safeToast.error('Invalid phone number (10 digits)');
      return false;
    }
    if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
      safeToast.error('Invalid email format');
      return false;
    }
    return true;
  };

  const validateStep2 = () => {
    const { address, city, state, pincode } = formData;
    if (!address || address.length < 10) {
      safeToast.error('Complete address is required');
      return false;
    }
    if (!city || !state) {
      safeToast.error('City and state are required');
      return false;
    }
    if (!/^[0-9]{6}$/.test(pincode)) {
      safeToast.error('Invalid pincode (6 digits)');
      return false;
    }
    return true;
  };

  const handleNext = () => {
    if (currentStep === 1 && !validateStep1()) return;
    if (currentStep === 2 && !validateStep2()) return;
    setCurrentStep(currentStep + 1);
  };

  const uploadFile = async (file: File, type: string) => {
    console.log('🔄 Starting upload:', { filename: file.name, type, size: file.size });
    
    const formDataUpload = new FormData();
    formDataUpload.append('file', file);
    formDataUpload.append('doc_type', type);
    
    // Get user_id from localStorage
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const userId = user.user_id || 'demo-user-id';
    formDataUpload.append('user_id', userId);
    
    const token = getToken();
    console.log('🔑 Token exists:', !!token);
    console.log('👤 User ID:', userId);
    console.log('🌐 API URL:', `${API_BASE_URL}/buyer-kyc/upload-document`);
    
    try {
      const response = await axios.post(
        `${API_BASE_URL}/buyer-kyc/upload-document`,
        formDataUpload,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'multipart/form-data'
          }
        }
      );
      
      console.log('✅ Upload response:', response.data);
      return response.data.success ? 'uploaded' : null;
    } catch (error: any) {
      console.error('❌ Upload error:', {
        message: error.message,
        status: error.response?.status,
        data: error.response?.data,
        url: error.config?.url
      });
      throw error;
    }
  };

  const handleFileChange = async (type: string, e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) {
      console.log('❌ No file selected');
      return;
    }

    console.log('📁 File selected:', { name: file.name, type: file.type, size: file.size });

    // Validate file size (5MB limit)
    if (file.size > 5 * 1024 * 1024) {
      console.log('❌ File too large:', file.size);
      safeToast.error('File size must be less than 5MB');
      return;
    }

    // Validate file type
    const allowedTypes = ['application/pdf', 'image/jpeg', 'image/png'];
    if (!allowedTypes.includes(file.type)) {
      console.log('❌ Invalid file type:', file.type);
      safeToast.error('Only PDF, JPG, and PNG files are allowed');
      return;
    }

    console.log('✅ File validation passed');
    setLoading(true);
    
    try {
      console.log('🚀 Starting upload process...');
      const result = await uploadFile(file, type);
      
      if (result) {
        console.log('✅ Upload successful, updating form data');
        setFormData(prev => ({ ...prev, [`${type}_url`]: 'uploaded' }));
        safeToast.success(`${type.replace('_', ' ')} uploaded successfully`);
      } else {
        console.log('❌ Upload returned no result');
        safeToast.error('Upload failed - no response');
      }
    } catch (error: any) {
      console.error('❌ Upload failed:', error);
      
      let errorMsg = 'Upload failed';
      if (error.response?.data?.detail) {
        // Handle validation errors array
        if (Array.isArray(error.response.data.detail)) {
          errorMsg = error.response.data.detail.map((err: any) => err.msg || err).join(', ');
        } else if (typeof error.response.data.detail === 'string') {
          errorMsg = error.response.data.detail;
        } else {
          errorMsg = 'Validation error occurred';
        }
      } else if (error.message) {
        errorMsg = error.message;
      }
      
      safeToast.error(errorMsg);
    } finally {
      setLoading(false);
      console.log('🏁 Upload process completed');
    }
  };

  const submitKYC = async () => {
    if (!formData.pan_document_url) {
      safeToast.error('PAN document is mandatory');
      return;
    }

    setLoading(true);
    try {
      const token = getToken();
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id || 'demo-user-id';
      
      // Prepare submit data with user_id and document URLs as 'uploaded'
      const submitData = {
        ...formData,
        user_id: userId,
        date_of_birth: formData.date_of_birth + 'T00:00:00.000Z',
        pan_document_url: formData.pan_document_url || 'uploaded',
        aadhaar_document_url: formData.aadhaar_document_url || '',
        passport_document_url: formData.passport_document_url || '',
        photo_url: formData.photo_url || '',
        bank_statement_url: formData.bank_statement_url || '',
        income_proof_url: formData.income_proof_url || ''
      };
      
      console.log('📤 Submitting KYC data:', submitData);
      
      await axios.post(`${API_BASE_URL}/buyer-kyc/submit`, submitData, {
        headers: { 
          Authorization: `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      safeToast.success('KYC submitted successfully!');
      setTimeout(() => window.location.href = '/buyer-dashboard', 2000);
    } catch (error: any) {
      let errorMsg = 'Submission failed';
      if (error.response?.data?.detail) {
        if (Array.isArray(error.response.data.detail)) {
          errorMsg = error.response.data.detail.map((err: any) => err.msg || err).join(', ');
        } else {
          errorMsg = error.response.data.detail;
        }
      }
      safeToast.error(errorMsg);
    } finally {
      setLoading(false);
    }
  };

  const steps = [
    { number: 1, title: 'Basic Info' },
    { number: 2, title: 'Address' },
    { number: 3, title: 'Documents' },
    { number: 4, title: 'Upload' }
  ];

  return (
    <div className="max-w-4xl mx-auto p-6">
      <h1 className="text-3xl font-bold mb-2">Complete Your KYC Verification</h1>
      <p className="text-gray-600 mb-8">Required to schedule visits and make inquiries</p>

      {/* Progress Indicator */}
      <div className="flex justify-between mb-12">
        {steps.map((step, idx) => (
          <div key={step.number} className="flex items-center flex-1">
            <div className="flex flex-col items-center flex-1">
              <div className={`w-10 h-10 rounded-full flex items-center justify-center font-semibold
                ${currentStep > step.number ? 'bg-green-500 text-white' : 
                  currentStep === step.number ? 'bg-blue-500 text-white ring-4 ring-blue-200' : 
                  'bg-gray-200 text-gray-600'}`}>
                {currentStep > step.number ? <CheckCircle className="w-6 h-6" /> : step.number}
              </div>
              <span className="text-sm mt-2 font-medium">{step.title}</span>
            </div>
            {idx < steps.length - 1 && (
              <div className={`h-1 flex-1 mx-2 ${currentStep > step.number ? 'bg-green-500' : 'bg-gray-200'}`} />
            )}
          </div>
        ))}
      </div>

      <Card className="p-6">
        {/* Step 1: Basic Info */}
        {currentStep === 1 && (
          <div className="space-y-4">
            <h2 className="text-xl font-semibold mb-4">Basic Identity Information</h2>
            <div>
              <Label>Full Name *</Label>
              <Input value={formData.full_name} onChange={(e) => setFormData(prev => ({ ...prev, full_name: e.target.value }))} placeholder="Enter your full name" />
            </div>
            <div>
              <Label>PAN Number *</Label>
              <Input value={formData.pan_number} onChange={(e) => setFormData(prev => ({ ...prev, pan_number: e.target.value.toUpperCase() }))} placeholder="ABCDE1234F" maxLength={10} />
            </div>
            <div>
              <Label>Date of Birth *</Label>
              <Input type="date" value={formData.date_of_birth} onChange={(e) => setFormData(prev => ({ ...prev, date_of_birth: e.target.value }))} />
            </div>
            <div>
              <Label>Phone Number *</Label>
              <Input value={formData.phone_number} onChange={(e) => setFormData(prev => ({ ...prev, phone_number: e.target.value }))} placeholder="9876543210" maxLength={10} />
            </div>
            <div>
              <Label>Email *</Label>
              <Input type="email" value={formData.email} onChange={(e) => setFormData(prev => ({ ...prev, email: e.target.value }))} placeholder="your@email.com" />
            </div>
          </div>
        )}

        {/* Step 2: Address */}
        {currentStep === 2 && (
          <div className="space-y-4">
            <h2 className="text-xl font-semibold mb-4">Address Information</h2>
            <div>
              <Label>Complete Address *</Label>
              <Input value={formData.address} onChange={(e) => setFormData(prev => ({ ...prev, address: e.target.value }))} placeholder="House No, Street, Area" />
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label>City *</Label>
                <Input value={formData.city} onChange={(e) => setFormData(prev => ({ ...prev, city: e.target.value }))} placeholder="Mumbai" />
              </div>
              <div>
                <Label>State *</Label>
                <Input value={formData.state} onChange={(e) => setFormData(prev => ({ ...prev, state: e.target.value }))} placeholder="Maharashtra" />
              </div>
            </div>
            <div>
              <Label>Pincode *</Label>
              <Input value={formData.pincode} onChange={(e) => setFormData(prev => ({ ...prev, pincode: e.target.value }))} placeholder="400001" maxLength={6} />
            </div>
          </div>
        )}

        {/* Step 3: Additional Documents */}
        {currentStep === 3 && (
          <div className="space-y-4">
            <h2 className="text-xl font-semibold mb-4">Additional Identity Documents</h2>
            <div>
              <Label>Aadhaar Number (Optional)</Label>
              <Input value={formData.aadhaar_number} onChange={(e) => setFormData(prev => ({ ...prev, aadhaar_number: e.target.value }))} placeholder="123456789012" maxLength={12} />
            </div>
            <div>
              <Label>Passport Number (Optional)</Label>
              <Input value={formData.passport_number} onChange={(e) => setFormData(prev => ({ ...prev, passport_number: e.target.value }))} placeholder="A1234567" />
            </div>
          </div>
        )}

        {/* Step 4: Document Upload */}
        {currentStep === 4 && (
          <div className="space-y-6">
            <h2 className="text-xl font-semibold mb-4">Upload Documents</h2>
            <div className="space-y-4">
              <div className="border rounded-lg p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base font-semibold">PAN Card * (Required)</Label>
                    <p className="text-sm text-gray-500">PDF, JPG, or PNG</p>
                  </div>
                  <div className="flex items-center gap-2">
                    {formData.pan_document_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                    <Input type="file" accept=".pdf,.jpg,.jpeg,.png" onChange={(e) => handleFileChange('pan_document', e)} className="w-auto" />
                  </div>
                </div>
              </div>

              <div className="border rounded-lg p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base font-semibold">Your Photo (Recommended)</Label>
                    <p className="text-sm text-gray-500">JPG or PNG</p>
                  </div>
                  <div className="flex items-center gap-2">
                    {formData.photo_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                    <Input type="file" accept=".jpg,.jpeg,.png" onChange={(e) => handleFileChange('photo', e)} className="w-auto" />
                  </div>
                </div>
              </div>

              {formData.aadhaar_number && (
                <div className="border rounded-lg p-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-base font-semibold">Aadhaar Card (Optional)</Label>
                      <p className="text-sm text-gray-500">PDF, JPG, or PNG</p>
                    </div>
                    <div className="flex items-center gap-2">
                      {formData.aadhaar_document_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                      <Input type="file" accept=".pdf,.jpg,.jpeg,.png" onChange={(e) => handleFileChange('aadhaar_document', e)} className="w-auto" />
                    </div>
                  </div>
                </div>
              )}
            </div>

            {!formData.pan_document_url && (
              <div className="flex items-center gap-2 text-amber-600 bg-amber-50 p-3 rounded">
                <AlertCircle className="w-5 h-5" />
                <span className="text-sm">PAN document upload is mandatory to submit KYC</span>
              </div>
            )}
            
            <div className="text-sm text-gray-500 mt-4">
              <p>• Documents are securely stored in our database</p>
              <p>• Maximum file size: 5MB per document</p>
              <p>• Supported formats: PDF, JPG, PNG</p>
            </div>
          </div>
        )}

        {/* Navigation Buttons */}
        <div className="flex justify-between mt-8">
          {currentStep > 1 && (
            <Button variant="outline" onClick={() => setCurrentStep(currentStep - 1)} disabled={loading} className="border-[#0056D2] text-[#0056D2] hover:bg-[#0056D2] hover:text-white">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Previous
            </Button>
          )}

          <div className="ml-auto">
            {currentStep < 4 ? (
              <Button onClick={handleNext} disabled={loading} className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white">
                Next
                <ArrowRight className="w-4 h-4 ml-2" />
              </Button>
            ) : (
              <Button onClick={submitKYC} disabled={loading || !formData.pan_document_url} className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white">
                {loading ? 'Submitting...' : 'Submit for Review'}
              </Button>
            )}
          </div>
        </div>
      </Card>
    </div>
  );
};
