import React, { useEffect, useState } from 'react';
import { Shield, CheckCircle, Clock, AlertCircle, XCircle } from 'lucide-react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8000';

export const BuyerKYCProfileSection: React.FC = () => {
  const [kycStatus, setKycStatus] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const navigate = useNavigate();

  useEffect(() => {
    fetchKYCStatus();
  }, []);

  const fetchKYCStatus = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await axios.get(`${API_BASE_URL}/buyer-kyc/check-verification`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setKycStatus(response.data);
    } catch (error) {
      console.error('Error fetching KYC status:', error);
    } finally {
      setLoading(false);
    }
  };

  const getStatusConfig = (status: string) => {
    const configs = {
      approved: {
        icon: CheckCircle,
        color: 'text-green-600',
        bgColor: 'bg-green-50',
        borderColor: 'border-green-200',
        text: 'Verified',
        badge: 'bg-green-100 text-green-800'
      },
      submitted: {
        icon: Clock,
        color: 'text-blue-600',
        bgColor: 'bg-blue-50',
        borderColor: 'border-blue-200',
        text: 'Under Review',
        badge: 'bg-blue-100 text-blue-800'
      },
      under_review: {
        icon: Clock,
        color: 'text-yellow-600',
        bgColor: 'bg-yellow-50',
        borderColor: 'border-yellow-200',
        text: 'Under Review',
        badge: 'bg-yellow-100 text-yellow-800'
      },
      rejected: {
        icon: XCircle,
        color: 'text-red-600',
        bgColor: 'bg-red-50',
        borderColor: 'border-red-200',
        text: 'Rejected',
        badge: 'bg-red-100 text-red-800'
      },
      resubmission_required: {
        icon: AlertCircle,
        color: 'text-orange-600',
        bgColor: 'bg-orange-50',
        borderColor: 'border-orange-200',
        text: 'Action Required',
        badge: 'bg-orange-100 text-orange-800'
      },
      not_started: {
        icon: Shield,
        color: 'text-gray-600',
        bgColor: 'bg-gray-50',
        borderColor: 'border-gray-200',
        text: 'Not Started',
        badge: 'bg-gray-100 text-gray-800'
      }
    };
    return configs[status as keyof typeof configs] || configs.not_started;
  };

  if (loading) {
    return (
      <div className="bg-white rounded-2xl shadow-sm border border-gray-200 p-6">
        <div className="flex items-center justify-center py-8">
          <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600"></div>
          <span className="ml-2 text-gray-600">Loading KYC status...</span>
        </div>
      </div>
    );
  }

  const status = kycStatus?.status || 'not_started';
  const config = getStatusConfig(status);
  const Icon = config.icon;

  return (
    <div className="bg-white rounded-2xl shadow-sm border border-gray-200 p-6">
      <div className="flex items-center justify-between mb-6">
        <h3 className="text-lg font-semibold text-gray-900 flex items-center gap-2">
          <Shield className="h-5 w-5 text-blue-600" />
          Buyer KYC Verification
        </h3>
        <button
          onClick={() => navigate('/buyer-kyc')}
          className="text-blue-600 hover:text-blue-800 text-sm font-medium"
        >
          {kycStatus?.is_verified ? 'View Details' : 'Complete KYC'} →
        </button>
      </div>

      <div className="space-y-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-3">
            <Icon className={`h-6 w-6 ${config.color}`} />
            <div>
              <p className="font-medium text-gray-900">Verification Status</p>
              <p className="text-sm text-gray-600">
                {kycStatus?.is_verified
                  ? 'Your identity has been successfully verified'
                  : 'Complete KYC to schedule visits and make inquiries'}
              </p>
            </div>
          </div>
          <span className={`inline-flex items-center px-3 py-1 rounded-full text-sm font-medium ${config.badge}`}>
            {config.text}
          </span>
        </div>

        {kycStatus?.is_verified ? (
          <div className={`${config.bgColor} border ${config.borderColor} rounded-lg p-4`}>
            <div className="flex items-center gap-2">
              <CheckCircle className="h-5 w-5 text-green-600" />
              <p className="text-green-800 font-medium">KYC Verification Complete</p>
            </div>
            <p className="text-green-700 text-sm mt-1">
              You can now schedule property visits and make inquiries.
            </p>
          </div>
        ) : (
          <div className={`${config.bgColor} border ${config.borderColor} rounded-lg p-4`}>
            <p className="text-sm font-medium text-gray-900 mb-2">Why complete KYC?</p>
            <ul className="text-sm text-gray-600 space-y-1">
              <li>• Schedule property visits</li>
              <li>• Make inquiries to sellers</li>
              <li>• Participate in bidding</li>
              <li>• Build trust with sellers</li>
            </ul>
            <button
              onClick={() => navigate('/buyer-kyc')}
              className="mt-4 w-full bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors"
            >
              Start KYC Verification
            </button>
          </div>
        )}
      </div>
    </div>
  );
};
