import React, { useEffect, useState } from 'react';
import { AlertCircle, CheckCircle } from 'lucide-react';
import { Button } from '../ui/button';
import { Card } from '../ui/card';
import axios from 'axios';
import { useNavigate } from 'react-router-dom';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8000';

interface KYCCheckProps {
  onVerified?: () => void;
  showModal?: boolean;
}

export const KYCVerificationCheck: React.FC<KYCCheckProps> = ({ onVerified, showModal = true }) => {
  const [kycStatus, setKycStatus] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const navigate = useNavigate();

  useEffect(() => {
    checkKYCStatus();
  }, []);

  const checkKYCStatus = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await axios.get(`${API_BASE_URL}/buyer-kyc/check-verification`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      setKycStatus(response.data);
      
      if (response.data.is_verified && onVerified) {
        onVerified();
      }
    } catch (error) {
      console.error('Error checking KYC status:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) return null;

  if (kycStatus?.is_verified) {
    return null; // User is verified, no need to show anything
  }

  if (!showModal) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <Card className="max-w-md w-full p-6">
        <div className="text-center">
          <div className="mx-auto w-16 h-16 bg-amber-100 rounded-full flex items-center justify-center mb-4">
            <AlertCircle className="w-8 h-8 text-amber-600" />
          </div>
          
          <h2 className="text-2xl font-bold mb-2">KYC Verification Required</h2>
          <p className="text-gray-600 mb-6">
            {kycStatus?.message || 'Please complete your KYC verification to access this feature'}
          </p>

          {kycStatus?.status && kycStatus.status !== 'not_started' && (
            <div className="mb-4 p-3 bg-blue-50 rounded-lg">
              <p className="text-sm text-blue-700">
                Current Status: <span className="font-semibold capitalize">{kycStatus.status.replace('_', ' ')}</span>
              </p>
            </div>
          )}

          <div className="space-y-3">
            <Button 
              onClick={() => navigate('/buyer-kyc')}
              className="w-full"
            >
              Complete KYC Verification
            </Button>
            
            <Button 
              variant="outline"
              onClick={() => navigate('/buyer-dashboard')}
              className="w-full"
            >
              Go to Dashboard
            </Button>
          </div>
        </div>
      </Card>
    </div>
  );
};
