import React, { useState, useEffect } from "react";

// Add CSS for animations
const chatbotStyles = `
  @keyframes pulse {
    0%, 100% { transform: scale(1); }
    50% { transform: scale(1.05); }
  }
  @keyframes bounce {
    0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
    40% { transform: translateY(-10px); }
    60% { transform: translateY(-5px); }
  }
  @keyframes slideUp {
    from { transform: translateY(100%); opacity: 0; }
    to { transform: translateY(0); opacity: 1; }
  }
`;

function Chatbot() {
  const [isOpen, setIsOpen] = useState(false);
  const [messages, setMessages] = useState([]);
  const [input, setInput] = useState("");
  const [typing, setTyping] = useState(false);
  const [conversationId, setConversationId] = useState(null);

  useEffect(() => {
    const handleOpenChatbot = () => {
      setIsOpen(true);
    };

    window.addEventListener('openChatbot', handleOpenChatbot);
    return () => window.removeEventListener('openChatbot', handleOpenChatbot);
  }, []);

  const toggleChat = () => setIsOpen(!isOpen);

  const sendMessage = async () => {
    if (!input.trim()) return;
    
    const userMsg = input;
    setMessages((prev) => [...prev, { from: "user", text: userMsg }]);
    setInput("");
    setTyping(true);

    try {
      const response = await fetch('/api/chat/chat', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          message: userMsg,
          conversation_id: conversationId,
          user_id: null, // Add user ID if available
          user_email: null // Add user email if available
        })
      });

      const data = await response.json();
      
      if (data.success) {
        // Set conversation ID for future messages
        if (!conversationId) {
          setConversationId(data.conversation_id);
        }
        
        setMessages((prev) => [...prev, { 
          from: "bot", 
          text: data.response,
          intent: data.intent,
          confidence: data.confidence
        }]);
        
        // Handle special actions
        if (data.action === 'redirect_to_listing_form') {
          // Could redirect to listing form
          console.log('Redirect to listing form');
        }
      } else {
        setMessages((prev) => [...prev, { 
          from: "bot", 
          text: "I'm having trouble processing your request. Please try again."
        }]);
      }
    } catch (error) {
      console.error('Chat error:', error);
      setMessages((prev) => [...prev, { 
        from: "bot", 
        text: "I'm experiencing technical difficulties. Please try again later."
      }]);
    } finally {
      setTyping(false);
    }
  };

  return (
    <>
      <style>{chatbotStyles}</style>
      {/* Simple Chatbot Icon */}
      <div
        style={{
          position: "fixed",
          bottom: "24px",
          right: "24px",
          background: "#3B82F6",
          color: "white",
          borderRadius: "50%",
          width: "60px",
          height: "60px",
          display: "flex",
          alignItems: "center",
          justifyContent: "center",
          cursor: "pointer",
          boxShadow: "0 4px 16px rgba(59, 130, 246, 0.25)",
          transition: "all 0.3s cubic-bezier(0.4, 0, 0.2, 1)",
          zIndex: 9998,
          border: "2px solid rgba(255, 255, 255, 0.1)",
          animation: "pulse 2s infinite"
        }}
        onClick={toggleChat}
        onMouseEnter={(e) => {
          e.currentTarget.style.transform = "scale(1.1)";
          e.currentTarget.style.boxShadow = "0 6px 24px rgba(59, 130, 246, 0.35)";
          e.currentTarget.style.animation = "bounce 0.6s ease";
        }}
        onMouseLeave={(e) => {
          e.currentTarget.style.transform = "scale(1)";
          e.currentTarget.style.boxShadow = "0 4px 16px rgba(59, 130, 246, 0.25)";
          e.currentTarget.style.animation = "pulse 2s infinite";
        }}
      >
        <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
          <path d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"/>
        </svg>
      </div>

      {/* Chat Panel */}
      {isOpen && (
        <div
          style={{
            position: "fixed",
            bottom: "90px",
            right: "24px",
            width: "320px",
            height: "420px",
            background: "white",
            border: "1px solid #e5e7eb",
            borderRadius: "16px",
            display: "flex",
            flexDirection: "column",
            boxShadow: "0 10px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04)",
            zIndex: 9999,
            animation: "slideUp 0.3s ease-out"
          }}
        >
          {/* Header with Close Button */}
          <div
            style={{
              background: "#3B82F6",
              color: "white",
              padding: "16px",
              borderTopLeftRadius: "12px",
              borderTopRightRadius: "12px",
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
              fontSize: "16px",
              fontWeight: "500"
            }}
          >
            <span style={{ display: "flex", alignItems: "center", gap: "8px" }}>
              <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                <path d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"/>
              </svg>
              NAL Assistant
            </span>
            <button
              onClick={toggleChat}
              style={{
                background: "rgba(255, 255, 255, 0.2)",
                border: "none",
                color: "white",
                width: "24px",
                height: "24px",
                borderRadius: "4px",
                cursor: "pointer",
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
                fontSize: "12px",
                transition: "background-color 0.2s ease"
              }}
              onMouseEnter={(e) => e.currentTarget.style.background = "rgba(255, 255, 255, 0.3)"}
              onMouseLeave={(e) => e.currentTarget.style.background = "rgba(255, 255, 255, 0.2)"}
            >
              <svg width="12" height="12" viewBox="0 0 24 24" fill="currentColor">
                <path d="M18.3 5.71a.996.996 0 0 0-1.41 0L12 10.59 7.11 5.7A.996.996 0 1 0 5.7 7.11L10.59 12 5.7 16.89a.996.996 0 1 0 1.41 1.41L12 13.41l4.89 4.89a.996.996 0 0 0 1.41-1.41L13.41 12l4.89-4.89c.38-.38.38-1.02 0-1.4z"/>
              </svg>
            </button>
          </div>

          <div style={{ flex: 1, padding: "10px", overflowY: "auto", maxHeight: "300px" }}>
            {messages.length === 0 && (
              <div style={{ color: "#666", fontSize: "14px", textAlign: "center", marginTop: "20px" }}>
                <div style={{ display: "flex", alignItems: "center", justifyContent: "center", marginBottom: "12px" }}>
                  <svg width="24" height="24" viewBox="0 0 24 24" fill="#3B82F6">
                    <path d="M12 2C13.1 2 14 2.9 14 4C14 5.1 13.1 6 12 6C10.9 6 10 5.1 10 4C10 2.9 10.9 2 12 2ZM21 9V7L15 1H5C3.89 1 3 1.89 3 3V21C3 22.11 3.89 23 5 23H11V21H5V3H13V9H21Z"/>
                  </svg>
                </div>
                Hi! I'm your NAL assistant. I can help you with:
                <br />• Property search
                <br />• Agent information  
                <br />• Platform features
                <br />• Pricing details
                <br /><br />What would you like to know?
              </div>
            )}
            {messages.map((msg, i) => (
              <div key={i} style={{ 
                marginBottom: "12px", 
                textAlign: msg.from === "user" ? "right" : "left",
                padding: "8px 12px",
                backgroundColor: msg.from === "user" ? "#3B82F6" : "#f8fafc",
                color: msg.from === "user" ? "white" : "#374151",
                borderRadius: "12px",
                maxWidth: "80%",
                marginLeft: msg.from === "user" ? "auto" : "0",
                marginRight: msg.from === "user" ? "0" : "auto",
                fontSize: "14px",
                lineHeight: "1.4",
                whiteSpace: "pre-wrap"
              }}>
                {msg.text}
                {/* Removed intent display as requested */}
              </div>
            ))}
            {typing && (
              <div style={{ 
                padding: "8px 12px", 
                backgroundColor: "#f8fafc", 
                borderRadius: "12px", 
                maxWidth: "80%",
                fontSize: "14px",
                fontStyle: "italic",
                color: "#6b7280",
                display: "flex",
                alignItems: "center",
                gap: "8px"
              }}>
                <div style={{ display: "flex", gap: "2px" }}>
                  <div style={{ width: "4px", height: "4px", borderRadius: "50%", backgroundColor: "#3B82F6", animation: "pulse 1.5s infinite" }}></div>
                  <div style={{ width: "4px", height: "4px", borderRadius: "50%", backgroundColor: "#3B82F6", animation: "pulse 1.5s infinite 0.2s" }}></div>
                  <div style={{ width: "4px", height: "4px", borderRadius: "50%", backgroundColor: "#3B82F6", animation: "pulse 1.5s infinite 0.4s" }}></div>
                </div>
                NAL assistant is typing...
              </div>
            )}
          </div>

          <div style={{ display: "flex", borderTop: "1px solid #e5e7eb", borderBottomLeftRadius: "16px", borderBottomRightRadius: "16px" }}>
            <input
              type="text"
              value={input}
              onChange={(e) => setInput(e.target.value)}
              onKeyPress={(e) => e.key === "Enter" && sendMessage()}
              placeholder="Type your message..."
              style={{ 
                flex: 1, 
                border: "none", 
                padding: "12px 16px", 
                fontSize: "14px",
                borderBottomLeftRadius: "16px",
                outline: "none"
              }}
            />
            <button 
              onClick={sendMessage} 
              style={{ 
                padding: "12px 16px", 
                background: "#3B82F6", 
                color: "white", 
                border: "none",
                borderBottomRightRadius: "16px",
                cursor: "pointer",
                fontSize: "16px",
                transition: "background-color 0.2s ease"
              }}
              onMouseEnter={(e) => e.currentTarget.style.background = "#2563EB"}
              onMouseLeave={(e) => e.currentTarget.style.background = "#3B82F6"}
            >
              ➤
            </button>
          </div>
        </div>
      )}
    </>
  );
}

export default Chatbot;
