import React, { useState, useEffect } from 'react';
import { MapPin, Clock, Car, Train, User } from 'lucide-react';

interface CommuteFilterProps {
   startingLocation: string;
   endingLocation: string;
   transportMode: 'driving' | 'transit' | 'walking';
   calculatedTime: number | null;
   showOnlyForSale?: boolean;
   onStartingLocationChange: (location: string) => void;
   onEndingLocationChange: (location: string) => void;
   onTransportModeChange: (mode: 'driving' | 'transit' | 'walking') => void;
   onCalculatedTimeChange: (time: number | null) => void;
   onShowOnlyForSaleChange?: (show: boolean) => void;
   availablePropertiesCount?: number;
   onPropertiesCountChange?: (count: number) => void;
}

export const CommuteFilter: React.FC<CommuteFilterProps> = ({
   startingLocation,
   endingLocation,
   transportMode,
   calculatedTime,
   showOnlyForSale = false,
   onStartingLocationChange,
   onEndingLocationChange,
   onTransportModeChange,
   onCalculatedTimeChange,
   onShowOnlyForSaleChange,
   availablePropertiesCount = 0,
   onPropertiesCountChange
}) => {
  console.log('CommuteFilter render with calculatedTime:', calculatedTime);

  const [locationSuggestions, setLocationSuggestions] = useState<string[]>([]);
  const [showSuggestions, setShowSuggestions] = useState(false);
  const [endingLocationSuggestions, setEndingLocationSuggestions] = useState<string[]>([]);
  const [showEndingSuggestions, setShowEndingSuggestions] = useState(false);
  const [isCalculating, setIsCalculating] = useState(false);

  // Calculate travel time - prioritize working fallback for now
  const calculateTravelTime = async (start: string, end: string, mode: string) => {
    console.log('🗺️ calculateTravelTime called with:', { start, end, mode });

    if (!start || !end) {
      console.log('❌ Missing start or end location');
      onCalculatedTimeChange(null);
      return;
    }

    setIsCalculating(true);

    try {
      // For now, use our custom calculation which works reliably
      console.log('🔄 Using enhanced fallback calculation');
      await calculateFallbackTime(start, end, mode);

      // Count properties available in the destination area
      if (calculatedTime && onPropertiesCountChange) {
        console.log('🏠 Counting properties for commute area...');
        await countPropertiesInCommuteArea(start, end, calculatedTime);
      }

      // TODO: Re-enable Google Maps API when key is properly configured
      /*
      // Try Google Maps Directions API first for most accurate results
      if (window.google && window.google.maps && window.google.maps.DirectionsService) {
        console.log('✅ Using Google Maps Directions API');

        const directionsService = new window.google.maps.DirectionsService();
        const request = {
          origin: start,
          destination: end,
          travelMode: window.google.maps.TravelMode[mode.toUpperCase()],
          drivingOptions: mode === 'driving' ? {
            departureTime: new Date(),
            trafficModel: window.google.maps.TrafficModel.BEST_GUESS
          } : undefined,
          transitOptions: mode === 'transit' ? {
            departureTime: new Date(),
            modes: [window.google.maps.TransitMode.BUS, window.google.maps.TransitMode.TRAIN, window.google.maps.TransitMode.SUBWAY]
          } : undefined,
          unitSystem: window.google.maps.UnitSystem.METRIC,
        };

        const result = await directionsService.route(request);
        if (result.status === 'OK' && result.routes[0] && result.routes[0].legs[0]) {
          const duration = result.routes[0].legs[0].duration.value;
          const minutes = Math.ceil(duration / 60);
          console.log('🎯 Google Maps result:', minutes, 'minutes');
          onCalculatedTimeChange(minutes);
          setIsCalculating(false);
          return;
        }
      }
      */

    } catch (error) {
      console.error('❌ Error calculating travel time:', error);
      onCalculatedTimeChange(30); // fallback
    } finally {
      setIsCalculating(false);
    }
  };

  // Fallback calculation using approximate distances and speeds
  const calculateFallbackTime = async (start: string, end: string, mode: string) => {
    console.log('🔄 Using fallback calculation for:', { start, end, mode });

    try {
      const startLower = start.toLowerCase();
      const endLower = end.toLowerCase();

      // Add some test cases to verify different results
      if (startLower.includes('electronic') && endLower.includes('whitefield')) {
        console.log('🎯 Test case: Electronic City to Whitefield');
        const testDistance = 28;
        const testSpeed = mode === 'driving' ? 25 : mode === 'transit' ? 20 : 5;
        const testMinutes = Math.ceil((testDistance / testSpeed) * 60);
        console.log(`Test result: ${testDistance}km at ${testSpeed}km/h = ${testMinutes} minutes`);
        onCalculatedTimeChange(testMinutes);
        return;
      }

      // Accurate distance matrix for Bangalore locations (in km) - based on actual road distances
      const distanceMatrix: { [key: string]: { [key: string]: number } } = {
        'electronic city': {
          'whitefield': 28,
          'koramangala': 22,
          'indiranagar': 26,
          'hsr layout': 16,
          'marathahalli': 20,
          'jayanagar': 24,
          'jp nagar': 20,
          'btm layout': 18,
          'bannerghatta road': 10,
          'sarjapur road': 12,
          'rajajinagar': 28,
          'malleshwaram': 26,
          'basavanagudi': 22,
          'ulsoor': 24
        },
        'whitefield': {
          'electronic city': 28,
          'koramangala': 24,
          'indiranagar': 22,
          'hsr layout': 18,
          'marathahalli': 10,
          'jayanagar': 26,
          'jp nagar': 22,
          'btm layout': 20,
          'bannerghatta road': 24,
          'sarjapur road': 14,
          'rajajinagar': 20,
          'malleshwaram': 18,
          'basavanagudi': 24,
          'ulsoor': 20
        },
        'koramangala': {
          'electronic city': 22,
          'whitefield': 24,
          'indiranagar': 10,
          'hsr layout': 12,
          'marathahalli': 18,
          'jayanagar': 8,
          'jp nagar': 10,
          'btm layout': 6,
          'bannerghatta road': 14,
          'sarjapur road': 18,
          'rajajinagar': 12,
          'malleshwaram': 10,
          'basavanagudi': 6,
          'ulsoor': 8
        },
        'indiranagar': {
          'electronic city': 26,
          'whitefield': 22,
          'koramangala': 10,
          'hsr layout': 14,
          'marathahalli': 16,
          'jayanagar': 12,
          'jp nagar': 14,
          'btm layout': 10,
          'bannerghatta road': 20,
          'sarjapur road': 20,
          'rajajinagar': 8,
          'malleshwaram': 6,
          'basavanagudi': 10,
          'ulsoor': 4
        },
        'hsr layout': {
          'electronic city': 16,
          'whitefield': 18,
          'koramangala': 12,
          'indiranagar': 14,
          'marathahalli': 10,
          'jayanagar': 14,
          'jp nagar': 10,
          'btm layout': 8,
          'bannerghatta road': 10,
          'sarjapur road': 10,
          'rajajinagar': 16,
          'malleshwaram': 14,
          'basavanagudi': 12,
          'ulsoor': 12
        },
        'marathahalli': {
          'electronic city': 20,
          'whitefield': 10,
          'koramangala': 18,
          'indiranagar': 16,
          'hsr layout': 10,
          'jayanagar': 22,
          'jp nagar': 18,
          'btm layout': 16,
          'bannerghatta road': 18,
          'sarjapur road': 8,
          'rajajinagar': 18,
          'malleshwaram': 16,
          'basavanagudi': 20,
          'ulsoor': 18
        },
        'jayanagar': {
          'electronic city': 24,
          'whitefield': 26,
          'koramangala': 8,
          'indiranagar': 12,
          'hsr layout': 14,
          'marathahalli': 22,
          'jp nagar': 6,
          'btm layout': 4,
          'bannerghatta road': 12,
          'sarjapur road': 20,
          'rajajinagar': 14,
          'malleshwaram': 12,
          'basavanagudi': 4,
          'ulsoor': 10
        },
        'jp nagar': {
          'electronic city': 20,
          'whitefield': 22,
          'koramangala': 10,
          'indiranagar': 14,
          'hsr layout': 10,
          'marathahalli': 18,
          'jayanagar': 6,
          'btm layout': 4,
          'bannerghatta road': 10,
          'sarjapur road': 16,
          'rajajinagar': 16,
          'malleshwaram': 14,
          'basavanagudi': 6,
          'ulsoor': 12
        },
        'btm layout': {
          'electronic city': 18,
          'whitefield': 20,
          'koramangala': 6,
          'indiranagar': 10,
          'hsr layout': 8,
          'marathahalli': 16,
          'jayanagar': 4,
          'jp nagar': 4,
          'bannerghatta road': 8,
          'sarjapur road': 14,
          'rajajinagar': 12,
          'malleshwaram': 10,
          'basavanagudi': 2,
          'ulsoor': 8
        },
        'bannerghatta road': {
          'electronic city': 10,
          'whitefield': 24,
          'koramangala': 14,
          'indiranagar': 20,
          'hsr layout': 10,
          'marathahalli': 18,
          'jayanagar': 12,
          'jp nagar': 10,
          'btm layout': 8,
          'sarjapur road': 8,
          'rajajinagar': 22,
          'malleshwaram': 20,
          'basavanagudi': 10,
          'ulsoor': 18
        },
        'sarjapur road': {
          'electronic city': 12,
          'whitefield': 14,
          'koramangala': 18,
          'indiranagar': 20,
          'hsr layout': 10,
          'marathahalli': 8,
          'jayanagar': 20,
          'jp nagar': 16,
          'btm layout': 14,
          'bannerghatta road': 8,
          'rajajinagar': 22,
          'malleshwaram': 20,
          'basavanagudi': 16,
          'ulsoor': 18
        },
        'rajajinagar': {
          'electronic city': 28,
          'whitefield': 20,
          'koramangala': 12,
          'indiranagar': 8,
          'hsr layout': 16,
          'marathahalli': 18,
          'jayanagar': 14,
          'jp nagar': 16,
          'btm layout': 12,
          'bannerghatta road': 22,
          'sarjapur road': 22,
          'malleshwaram': 4,
          'basavanagudi': 10,
          'ulsoor': 6
        },
        'malleshwaram': {
          'electronic city': 26,
          'whitefield': 18,
          'koramangala': 10,
          'indiranagar': 6,
          'hsr layout': 14,
          'marathahalli': 16,
          'jayanagar': 12,
          'jp nagar': 14,
          'btm layout': 10,
          'bannerghatta road': 20,
          'sarjapur road': 20,
          'rajajinagar': 4,
          'basavanagudi': 8,
          'ulsoor': 4
        },
        'basavanagudi': {
          'electronic city': 22,
          'whitefield': 24,
          'koramangala': 6,
          'indiranagar': 10,
          'hsr layout': 12,
          'marathahalli': 20,
          'jayanagar': 4,
          'jp nagar': 6,
          'btm layout': 2,
          'bannerghatta road': 10,
          'sarjapur road': 16,
          'rajajinagar': 10,
          'malleshwaram': 8,
          'ulsoor': 8
        },
        'ulsoor': {
          'electronic city': 24,
          'whitefield': 20,
          'koramangala': 8,
          'indiranagar': 4,
          'hsr layout': 12,
          'marathahalli': 18,
          'jayanagar': 10,
          'jp nagar': 12,
          'btm layout': 8,
          'bannerghatta road': 18,
          'sarjapur road': 18,
          'rajajinagar': 6,
          'malleshwaram': 4,
          'basavanagudi': 8
        }
      };

      // Find matching locations - improved matching logic
      let distance = 15; // default 15km
      let foundMatch = false;

      // Normalize location names for better matching
      const normalizeLocation = (location: string) => {
        return location.toLowerCase()
          .replace(/, bangalore/g, '')
          .replace(/ bangalore/g, '')
          .replace(/, mumbai/g, '')
          .replace(/ mumbai/g, '')
          .replace(/, delhi/g, '')
          .replace(/ delhi/g, '')
          .replace(/, chennai/g, '')
          .replace(/ chennai/g, '')
          .replace(/, hyderabad/g, '')
          .replace(/ hyderabad/g, '')
          .replace(/, pune/g, '')
          .replace(/ pune/g, '')
          .replace(/, kolkata/g, '')
          .replace(/ kolkata/g, '')
          .replace(/, maharashtra/g, '')
          .replace(/, karnataka/g, '')
          .replace(/, tamil nadu/g, '')
          .replace(/, telangana/g, '')
          .replace(/, delhi ncr/g, '')
          .trim();
      };

      const normalizedStart = normalizeLocation(startLower);
      const normalizedEnd = normalizeLocation(endLower);

      console.log('Normalized locations:', { normalizedStart, normalizedEnd });

      // Check for direct matches in distance matrix
      console.log('🔍 Checking distance matrix for:', normalizedStart, '→', normalizedEnd);
      for (const [startKey, destinations] of Object.entries(distanceMatrix)) {
        const startMatch = normalizedStart.includes(startKey) || startKey.includes(normalizedStart);
        console.log(`Checking start: "${startKey}" against "${normalizedStart}" = ${startMatch}`);
        if (startMatch) {
          for (const [endKey, dist] of Object.entries(destinations)) {
            const endMatch = normalizedEnd.includes(endKey) || endKey.includes(normalizedEnd);
            console.log(`Checking end: "${endKey}" against "${normalizedEnd}" = ${endMatch}, distance = ${dist}km`);
            if (endMatch) {
              distance = dist;
              foundMatch = true;
              console.log(`✅ Direct match found: ${startKey} to ${endKey} = ${dist}km`);
              break;
            }
          }
          if (foundMatch) break;
        }
      }

      // If no direct match, try reverse lookup
      if (!foundMatch) {
        for (const [startKey, destinations] of Object.entries(distanceMatrix)) {
          if (normalizedEnd.includes(startKey) || startKey.includes(normalizedEnd)) {
            for (const [endKey, dist] of Object.entries(destinations)) {
              if (normalizedStart.includes(endKey) || endKey.includes(normalizedStart)) {
                distance = dist;
                foundMatch = true;
                console.log(`Reverse match found: ${endKey} to ${startKey} = ${dist}km`);
                break;
              }
            }
            if (foundMatch) break;
          }
        }
      }

      // If still no match, check for city-level distances
      if (!foundMatch) {
        // Define city areas for better matching
        const cityAreas = {
          bangalore: ['bangalore', 'bengaluru', 'electronic', 'whitefield', 'koramangala', 'indiranagar', 'hsr', 'marathahalli', 'jayanagar', 'jp nagar', 'btm', 'sarjapur', 'bannerghatta'],
          mumbai: ['mumbai', 'bandra', 'andheri', 'powai', 'juhu', 'malad', 'borivali', 'thane', 'navi mumbai', 'worli', 'lower parel'],
          delhi: ['delhi', 'connaught place', 'karol bagh', 'lajpat nagar', 'south extension', 'vasant kunj', 'dwarka', 'rohini', 'janakpuri'],
          chennai: ['chennai', 't nagar', 'anna nagar', 'adyar', 'velachery', 'omr', 'porur', 'tambaram', 'chrompet'],
          hyderabad: ['hyderabad', 'banjara hills', 'jubilee hills', 'gachibowli', 'hitech city', 'kondapur', 'madhapur', 'kukatpally'],
          pune: ['pune', 'koregaon park', 'viman nagar', 'kharadi', 'wakad', 'hinjewadi', 'baner', 'aundh'],
          kolkata: ['kolkata', 'salt lake', 'new town', 'park street', 'ballygunge', 'alipore']
        };

        // Determine cities for start and end locations
        let startCity = 'unknown';
        let endCity = 'unknown';

        for (const [city, areas] of Object.entries(cityAreas)) {
          if (areas.some(area => normalizedStart.includes(area))) {
            startCity = city;
            break;
          }
        }

        for (const [city, areas] of Object.entries(cityAreas)) {
          if (areas.some(area => normalizedEnd.includes(area))) {
            endCity = city;
            break;
          }
        }

        console.log('Detected cities:', { startCity, endCity });

        if (startCity === endCity && startCity !== 'unknown') {
          // Same city - use intra-city distance
          const intraCityDistances = {
            bangalore: 12,
            mumbai: 15,
            delhi: 18,
            chennai: 14,
            hyderabad: 13,
            pune: 11,
            kolkata: 10
          };
          distance = intraCityDistances[startCity as keyof typeof intraCityDistances] || 12;
          console.log(`Both locations in ${startCity}, using intra-city distance: ${distance}km`);
        } else if (startCity !== 'unknown' && endCity !== 'unknown') {
          // Different cities - use inter-city distance
          const interCityDistances: { [key: string]: { [key: string]: number } } = {
            bangalore: { mumbai: 850, delhi: 1750, chennai: 350, hyderabad: 500, pune: 700, kolkata: 1560 },
            mumbai: { delhi: 1150, chennai: 1030, hyderabad: 630, pune: 120, kolkata: 1660 },
            delhi: { chennai: 1750, hyderabad: 1250, pune: 1150, kolkata: 1300 },
            chennai: { hyderabad: 510, pune: 930, kolkata: 1360 },
            hyderabad: { pune: 500, kolkata: 1090 },
            pune: { kolkata: 1580 }
          };

          // Make the matrix symmetric
          const symmetricDistances = { ...interCityDistances };
          for (const [from, destinations] of Object.entries(interCityDistances)) {
            for (const [to, dist] of Object.entries(destinations)) {
              if (!symmetricDistances[to]) symmetricDistances[to] = {};
              symmetricDistances[to][from] = dist;
            }
          }

          distance = symmetricDistances[startCity]?.[endCity] || 300; // Default 300km for unknown inter-city
          console.log(`Locations in different cities (${startCity} to ${endCity}), using inter-city distance: ${distance}km`);
        } else {
          // Unknown cities or one unknown
          distance = startCity === 'unknown' && endCity === 'unknown' ? 25 : 50;
          console.log('Unknown city combination, using default distance:', distance, 'km');
        }
      }

      // Realistic speed calculations for Indian traffic conditions
      let speed: number;

      if (mode === 'driving') {
        // Indian driving speeds accounting for traffic, signals, and road conditions
        if (distance > 100) {
          speed = 70; // Long distance highways
        } else if (distance > 30) {
          speed = 35; // City outskirts and arterial roads
        } else if (distance > 10) {
          speed = 22; // Inner city roads with moderate traffic
        } else {
          speed = 15; // Central business districts and congested areas
        }
      } else if (mode === 'transit') {
        // Public transport speeds including waiting times and transfers
        if (distance > 100) {
          speed = 40; // Long distance trains/buses
        } else if (distance > 30) {
          speed = 22; // Metro/suburban trains with transfers
        } else if (distance > 10) {
          speed = 16; // City buses with stops
        } else {
          speed = 12; // Local buses in congested areas
        }
      } else if (mode === 'walking') {
        speed = 4.5; // Realistic walking speed including crossings
      } else {
        speed = 18; // Default
      }

      // Adjust for rush hour (severe Indian traffic conditions)
      const currentHour = new Date().getHours();
      const isRushHour = (currentHour >= 8 && currentHour <= 11) || (currentHour >= 17 && currentHour <= 20);
      const isPeakRushHour = (currentHour >= 9 && currentHour <= 10) || (currentHour >= 18 && currentHour <= 19);

      if (mode === 'driving') {
        if (isPeakRushHour && distance <= 30) {
          speed *= 0.4; // 60% slower during peak rush hour in city
          console.log('Peak rush hour detected in city, reducing driving speed by 60%');
        } else if (isRushHour && distance <= 30) {
          speed *= 0.55; // 45% slower during regular rush hour
          console.log('Rush hour detected in city, reducing driving speed by 45%');
        }
      } else if (mode === 'transit') {
        if (isRushHour && distance <= 30) {
          speed *= 0.75; // 25% slower during rush hour due to crowding
          console.log('Rush hour detected, reducing transit speed by 25%');
        }
      }

      // Additional factors for very short distances (traffic signals, parking, etc.)
      if (mode === 'driving' && distance < 5) {
        speed *= 0.8; // Additional 20% reduction for very short trips
      }

      const hours = distance / speed;
      const minutes = Math.ceil(hours * 60);

      console.log(`Calculation result - distance: ${distance}km, speed: ${speed}km/h, time: ${minutes} minutes`);
      console.log('Calling onCalculatedTimeChange with:', minutes);

      onCalculatedTimeChange(minutes);

    } catch (error) {
      console.error('Fallback calculation failed:', error);
      onCalculatedTimeChange(30); // fallback to 30 minutes
    }
  };

  // Haversine formula to calculate distance between two points
  const getDistanceFromLatLonInKm = (lat1: number, lon1: number, lat2: number, lon2: number) => {
    const R = 6371; // Radius of the earth in km
    const dLat = deg2rad(lat2 - lat1);
    const dLon = deg2rad(lon2 - lon1);
    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos(deg2rad(lat1)) * Math.cos(deg2rad(lat2)) *
      Math.sin(dLon / 2) * Math.sin(dLon / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    const d = R * c; // Distance in km
    return d;
  };

  const deg2rad = (deg: number) => {
    return deg * (Math.PI / 180);
  };

  // Count properties available for sale within commute area
  const countPropertiesInCommuteArea = async (start: string, end: string, maxTime: number) => {
    console.log('🏠 Counting properties for commute:', { start, end, maxTime });

    try {
      // For now, we'll use a simple approach - count sell properties in the destination area
      // In a real implementation, this would use the commute time to filter properties within the travel radius
      const endLocationNormalized = end.toLowerCase();

      // Extract city from end location
      let city = 'Bangalore'; // default
      if (endLocationNormalized.includes('mumbai')) city = 'Mumbai';
      else if (endLocationNormalized.includes('delhi')) city = 'Delhi';
      else if (endLocationNormalized.includes('chennai')) city = 'Chennai';
      else if (endLocationNormalized.includes('hyderabad')) city = 'Hyderabad';
      else if (endLocationNormalized.includes('pune')) city = 'Pune';

      console.log('🏠 Searching properties in city:', city);

      // Fetch properties with sell listing type in the destination city
      const response = await fetch(`/api/properties/?listing_type=sell&city=${encodeURIComponent(city)}&limit=100`, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });

      if (response.ok) {
        const result = await response.json();
        const properties = result.properties || [];
        const count = properties.length;

        console.log(`✅ Found ${count} sell properties in ${city}`);
        if (onPropertiesCountChange) {
          onPropertiesCountChange(count);
        }
        return count;
      } else {
        console.warn('⚠️ Failed to fetch properties for counting');
        if (onPropertiesCountChange) {
          onPropertiesCountChange(0);
        }
        return 0;
      }
    } catch (error) {
      console.error('❌ Error counting properties:', error);
      if (onPropertiesCountChange) {
        onPropertiesCountChange(0);
      }
      return 0;
    }
  };

  // Recalculate when locations or transport mode change
  useEffect(() => {
    console.log('🚗 useEffect triggered with:', { startingLocation, endingLocation, transportMode, calculatedTime });
    if (startingLocation && endingLocation) {
      console.log('📍 Calling calculateTravelTime for:', startingLocation, '→', endingLocation, 'via', transportMode);
      // Immediate call without delay for responsiveness
      calculateTravelTime(startingLocation, endingLocation, transportMode);
    } else {
      console.log('❌ Clearing calculated time - missing locations');
      onCalculatedTimeChange(null);
      if (onPropertiesCountChange) {
        onPropertiesCountChange(0);
      }
    }
  }, [startingLocation, endingLocation, transportMode]);

  // Common starting locations for suggestions
  const commonLocations = [
    'Electronic City, Bangalore',
    'Whitefield, Bangalore',
    'Koramangala, Bangalore',
    'Indiranagar, Bangalore',
    'HSR Layout, Bangalore',
    'Marathahalli, Bangalore',
    'Jayanagar, Bangalore',
    'JP Nagar, Bangalore',
    'BTM Layout, Bangalore',
    'Bannerghatta Road, Bangalore',
    'Sarjapur Road, Bangalore',
    'Rajajinagar, Bangalore',
    'Malleshwaram, Bangalore',
    'Basavanagudi, Bangalore',
    'Ulsoor, Bangalore',
    'Bandra Kurla Complex, Mumbai',
    'Andheri, Mumbai',
    'Powai, Mumbai',
    'Juhu, Mumbai',
    'Malad, Mumbai',
    'Borivali, Mumbai',
    'Hitech City, Hyderabad',
    'DLF Cyber City, Gurgaon',
    'Salt Lake, Kolkata',
    'Rajiv Gandhi Infotech Park, Pune',
    'Phoenix Market City, Chennai',
    'Connaught Place, Delhi',
    'Infocity, Gandhinagar',
    'Technopark, Trivandrum'
  ];

  const handleLocationInputChange = (value: string) => {
    console.log('Starting location changed to:', value);
    onStartingLocationChange(value);
    if (value.length > 2) {
      const filtered = commonLocations.filter(place =>
        place.toLowerCase().includes(value.toLowerCase())
      );
      setLocationSuggestions(filtered);
      setShowSuggestions(true);
    } else {
      setShowSuggestions(false);
    }
  };

  const handleEndingLocationInputChange = (value: string) => {
    console.log('Ending location changed to:', value);
    onEndingLocationChange(value);
    if (value.length > 2) {
      const filtered = commonLocations.filter(place =>
        place.toLowerCase().includes(value.toLowerCase())
      );
      setEndingLocationSuggestions(filtered);
      setShowEndingSuggestions(true);
    } else {
      setShowEndingSuggestions(false);
    }
  };

  const handleSuggestionSelect = (suggestion: string) => {
    console.log('Starting location selected:', suggestion);
    onStartingLocationChange(suggestion);
    setShowSuggestions(false);
  };

  const handleEndingSuggestionSelect = (suggestion: string) => {
    console.log('Ending location selected:', suggestion);
    onEndingLocationChange(suggestion);
    setShowEndingSuggestions(false);
  };

  const transportOptions = [
    { value: 'driving' as const, label: 'Driving', icon: Car, color: 'text-blue-600' },
    { value: 'transit' as const, label: 'Public Transport', icon: Train, color: 'text-green-600' },
    { value: 'walking' as const, label: 'Walking', icon: User, color: 'text-orange-600' }
  ];

  return (
    <div className="mb-6">
      <h4 className="text-md font-medium text-gray-800 mb-4 flex items-center">
        <Clock className="w-4 h-4 mr-2 text-[#0056D2]" />
        Commute Time Filter
      </h4>

      <div className="space-y-4">
        {/* Starting Location Input */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Starting Location
          </label>
          <div className="relative">
            <input
              type="text"
              value={startingLocation}
              onChange={(e) => handleLocationInputChange(e.target.value)}
              placeholder="Enter your starting location (home, office, etc.)"
              className="w-full px-3 py-2 pl-10 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
            />
            <MapPin className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400" />

            {/* Location Suggestions */}
            {showSuggestions && locationSuggestions.length > 0 && (
              <div className="absolute z-10 w-full mt-1 bg-white border border-gray-200 rounded-lg shadow-lg max-h-40 overflow-y-auto">
                {locationSuggestions.map((suggestion, index) => (
                  <div
                    key={index}
                    onClick={() => handleSuggestionSelect(suggestion)}
                    className="px-3 py-2 hover:bg-gray-50 cursor-pointer text-sm border-b border-gray-100 last:border-b-0"
                  >
                    <MapPin className="w-3 h-3 inline mr-2 text-gray-400" />
                    {suggestion}
                  </div>
                ))}
              </div>
            )}
          </div>
        </div>

        {/* Ending Location Input */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Ending Location
          </label>
          <div className="relative">
            <input
              type="text"
              value={endingLocation}
              onChange={(e) => handleEndingLocationInputChange(e.target.value)}
              placeholder="Enter your ending location (work, destination, etc.)"
              className="w-full px-3 py-2 pl-10 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
            />
            <MapPin className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400" />

            {/* Ending Location Suggestions */}
            {showEndingSuggestions && endingLocationSuggestions.length > 0 && (
              <div className="absolute z-10 w-full mt-1 bg-white border border-gray-200 rounded-lg shadow-lg max-h-40 overflow-y-auto">
                {endingLocationSuggestions.map((suggestion, index) => (
                  <div
                    key={index}
                    onClick={() => handleEndingSuggestionSelect(suggestion)}
                    className="px-3 py-2 hover:bg-gray-50 cursor-pointer text-sm border-b border-gray-100 last:border-b-0"
                  >
                    <MapPin className="w-3 h-3 inline mr-2 text-gray-400" />
                    {suggestion}
                  </div>
                ))}
              </div>
            )}
          </div>
        </div>

        {/* Transport Mode Selection */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Transport Mode
          </label>
          <div className="grid grid-cols-3 gap-2">
            {transportOptions.map((option) => {
              const Icon = option.icon;
              return (
                <button
                  key={option.value}
                  onClick={() => onTransportModeChange(option.value)}
                  className={`flex items-center justify-center px-3 py-2 border rounded-lg text-sm font-medium transition-colors ${
                    transportMode === option.value
                      ? 'border-[#0056D2] bg-[#0056D2] text-white'
                      : 'border-gray-300 text-gray-700 hover:border-[#0056D2] hover:bg-[#0056D2]/5'
                  }`}
                >
                  <Icon className={`w-4 h-4 mr-2 ${transportMode === option.value ? 'text-white' : option.color}`} />
                  {option.label}
                </button>
              );
            })}
          </div>
        </div>

        {/* Show Only Properties for Sale */}
        {onShowOnlyForSaleChange && (
          <div>
            <label className="flex items-center text-sm">
              <input
                type="checkbox"
                checked={showOnlyForSale}
                onChange={(e) => onShowOnlyForSaleChange(e.target.checked)}
                className="mr-2 w-4 h-4 rounded text-[#0056D2] focus:ring-[#0056D2]"
              />
              Show only properties ready to sell
            </label>
          </div>
        )}

        {/* Calculated Travel Time Display */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Commute Analysis
          </label>
          <div className="space-y-3">
            {/* Travel Time */}
            <div className="flex items-center space-x-2">
              <div className="flex-1 px-3 py-2 bg-gray-50 border border-gray-300 rounded-lg text-sm">
                {isCalculating ? (
                  <div className="flex items-center">
                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-[#0056D2] mr-2"></div>
                    Calculating...
                  </div>
                ) : calculatedTime !== null ? (
                  <div>
                    <div className="font-medium flex items-center">
                      <Clock className="w-4 h-4 mr-1 text-blue-600" />
                      {calculatedTime} minutes by {transportOptions.find(opt => opt.value === transportMode)?.label.toLowerCase()}
                    </div>
                    <div className="text-xs text-gray-500">
                      Estimated travel time
                    </div>
                  </div>
                ) : (
                  'Enter locations to calculate'
                )}
              </div>
              <button
                onClick={() => {
                  console.log('🔄 Manual calculation triggered');
                  calculateTravelTime(startingLocation, endingLocation, transportMode);
                }}
                className="px-2 py-1 bg-blue-500 text-white rounded text-xs hover:bg-blue-600"
                disabled={!startingLocation || !endingLocation}
              >
                Test
              </button>
            </div>

            {/* Properties Available */}
            {calculatedTime !== null && availablePropertiesCount !== undefined && (
              <div className="px-3 py-2 bg-green-50 border border-green-200 rounded-lg text-sm">
                <div className="font-medium flex items-center text-green-800">
                  <div className="w-4 h-4 bg-green-500 rounded-full mr-2 flex items-center justify-center">
                    <span className="text-white text-xs font-bold">{availablePropertiesCount}</span>
                  </div>
                  Properties ready to sell in {endingLocation || 'destination area'}
                </div>
                <div className="text-xs text-green-600 mt-1">
                  {availablePropertiesCount > 0
                    ? `${availablePropertiesCount} verified propert${availablePropertiesCount === 1 ? 'y' : 'ies'} available within your commute time`
                    : 'No properties currently available in this area'
                  }
                </div>
              </div>
            )}
          </div>

          {startingLocation && endingLocation && calculatedTime === null && !isCalculating && (
            <p className="text-xs text-gray-500 mt-1">Travel time will be calculated automatically.</p>
          )}
        </div>

        {/* Commute Time Info */}
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-3">
          <div className="flex items-start">
            <Clock className="w-4 h-4 text-blue-600 mt-0.5 mr-2 flex-shrink-0" />
            <div className="text-sm text-blue-800">
              <p className="font-medium">Commute Filter Active</p>
              <p className="text-blue-600">
                {calculatedTime ? (
                  `Travel time from ${startingLocation || 'starting location'} to ${endingLocation || 'ending location'} is ${calculatedTime} minutes by ${transportOptions.find(opt => opt.value === transportMode)?.label.toLowerCase()}.`
                ) : (
                  `Enter starting and ending locations to calculate travel time.`
                )}
              </p>
              <p className="text-blue-600 mt-1">
                Showing properties ready to sell within the calculated commute time.
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};