import React, { useState, useEffect } from 'react';
import { X, TrendingUp, TrendingDown, Minus, BarChart3, MapPin, Home, AlertCircle, CheckCircle, Info } from 'lucide-react';
import { competitorAnalyticsApi, CompetitorAnalyticsRequest, CompetitorAnalyticsResponse } from '../../api/competitorAnalytics';
import { toast } from 'sonner';
import { IndianRupee } from "lucide-react";
interface CompetitorAnalyticsModalProps {
  isOpen: boolean;
  onClose: () => void;
  propertyData?: {
    id?: number;
    city: string;
    locality: string;
    property_type: string;
    listing_type: string;
    property_condition?: string;
    bedrooms?: number;
    built_up_area?: number;
    sale_price?: number;
    listed_date?: string;
  };
}

export const CompetitorAnalyticsModal: React.FC<CompetitorAnalyticsModalProps> = ({
  isOpen,
  onClose,
  propertyData
}) => {
  const [isLoading, setIsLoading] = useState(false);
  const [analytics, setAnalytics] = useState<CompetitorAnalyticsResponse | null>(null);
  const [formData, setFormData] = useState({
    city: propertyData?.city || '',
    locality: propertyData?.locality || '',
    property_type: propertyData?.property_type || 'apartment',
    listing_type: propertyData?.listing_type || 'sell',
    property_condition: propertyData?.property_condition || 'resale',
    bhk: propertyData?.bedrooms || 2,
    builtup_area_sqft: propertyData?.built_up_area || 1000,
    price: propertyData?.sale_price || 5000000,
    listed_date: propertyData?.listed_date || new Date().toISOString().split('T')[0]
  });

  useEffect(() => {
    if (propertyData) {
      setFormData(prev => ({
        ...prev,
        city: propertyData.city || prev.city,
        locality: propertyData.locality || prev.locality,
        property_type: propertyData.property_type || prev.property_type,
        listing_type: propertyData.listing_type || prev.listing_type,
        property_condition: propertyData.property_condition || prev.property_condition,
        bhk: propertyData.bedrooms || prev.bhk,
        builtup_area_sqft: propertyData.built_up_area || prev.builtup_area_sqft,
        price: propertyData.sale_price || prev.price,
        listed_date: propertyData.listed_date || prev.listed_date
      }));
    }
  }, [propertyData]);

  const handleAnalyze = async () => {
    if (!formData.city || !formData.locality || !formData.builtup_area_sqft || !formData.price) {
      toast.error('Please fill all required fields');
      return;
    }

    setIsLoading(true);
    try {
      const request: CompetitorAnalyticsRequest = {
        property_id: propertyData?.id,
        city: formData.city,
        locality: formData.locality,
        property_type: formData.property_type,
        listing_type: formData.listing_type,
        property_condition: formData.property_condition,
        bhk: formData.bhk,
        builtup_area_sqft: formData.builtup_area_sqft,
        price: formData.price,
        listed_date: formData.listed_date
      };

      const result = await competitorAnalyticsApi.analyzeProperty(request);
      setAnalytics(result);
      if (result.verdict === 'insufficient_data') {
        toast.info('Analysis completed - limited market data available');
      } else {
        toast.success('Analytics completed successfully');
      }
    } catch (error: any) {
      console.error('Analytics error:', error);
      if (error.response?.status === 500) {
        toast.error('Server error. Analytics service temporarily unavailable.');
      } else {
        toast.error('Failed to analyze property. Please try again.');
      }
    } finally {
      setIsLoading(false);
    }
  };

  const getVerdictIcon = (verdict: string) => {
    switch (verdict) {
      case 'Underpriced':
        return <TrendingDown className="w-5 h-5 text-green-600" />;
      case 'Overpriced':
        return <TrendingUp className="w-5 h-5 text-red-600" />;
      case 'insufficient_data':
        return <AlertCircle className="w-5 h-5 text-gray-600" />;
      default:
        return <Minus className="w-5 h-5 text-blue-600" />;
    }
  };

  const getVerdictColor = (verdict: string) => {
    switch (verdict) {
      case 'Underpriced':
        return 'text-green-600 bg-green-50 border-green-200';
      case 'Overpriced':
        return 'text-red-600 bg-red-50 border-red-200';
      case 'insufficient_data':
        return 'text-gray-600 bg-gray-50 border-gray-200';
      default:
        return 'text-blue-600 bg-blue-50 border-blue-200';
    }
  };

  const getConfidenceBadge = (confidence: string) => {
    switch (confidence) {
      case 'High':
        return (
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
            <CheckCircle className="w-3 h-3 mr-1" />
            High Confidence
          </span>
        );
      case 'Medium':
        return (
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
            <Info className="w-3 h-3 mr-1" />
            Medium Confidence
          </span>
        );
      case 'Low':
        return (
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-orange-100 text-orange-800">
            <Info className="w-3 h-3 mr-1" />
            Low Confidence
          </span>
        );
      default:
        return (
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
            <AlertCircle className="w-3 h-3 mr-1" />
            Insufficient Data
          </span>
        );
    }
  };

  const formatCurrency = (value: number) => {
    return `₹${value.toLocaleString('en-IN')}`;
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-white bg-opacity-95 z-50 overflow-y-auto">
      <div className="min-h-screen flex items-center justify-center p-4">
        <div className="bg-white rounded-lg shadow-xl max-w-5xl w-full my-8">
          {/* Header */}
          <div className="flex items-center justify-between p-6 border-b border-gray-200 bg-white rounded-t-lg">
          <div className="flex items-center space-x-3">
            <BarChart3 className="w-6 h-6 text-blue-600" />
            <div>
              <h2 className="text-xl font-semibold text-gray-900">Competitor Analytics</h2>
              <p className="text-sm text-gray-500">Seller-focused market pricing insights</p>
            </div>
          </div>
          <button
            onClick={onClose}
            className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
          >
            <X className="w-5 h-5 text-gray-500" />
          </button>
        </div>

        <div className="p-6">
          {!analytics ? (
            /* Input Form */
            <div className="space-y-6">
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <div className="flex items-start space-x-3">
                  <Info className="w-5 h-5 text-blue-600 mt-0.5" />
                  <div>
                    <h3 className="font-medium text-blue-900">Property Details</h3>
                    <p className="text-sm text-blue-700 mt-1">
                      Enter your property details to get accurate competitor pricing analysis. 
                      All comparisons are based on similar properties in your locality.
                    </p>
                  </div>
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    City *
                  </label>
                  <input
                    type="text"
                    value={formData.city}
                    onChange={(e) => setFormData({ ...formData, city: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    placeholder="e.g., Bengaluru"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Locality *
                  </label>
                  <input
                    type="text"
                    value={formData.locality}
                    onChange={(e) => setFormData({ ...formData, locality: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    placeholder="e.g., Whitefield"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Property Type *
                  </label>
                  <select
                    value={formData.property_type}
                    onChange={(e) => setFormData({ ...formData, property_type: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option value="apartment">Apartment</option>
                    <option value="villa">Villa</option>
                    <option value="house">House</option>
                    <option value="plot">Plot</option>
                    <option value="builder-floor">Builder Floor</option>
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Listing Type *
                  </label>
                  <select
                    value={formData.listing_type}
                    onChange={(e) => setFormData({ ...formData, listing_type: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option value="sell">Sale</option>
                    <option value="rent">Rent</option>
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Property Condition
                  </label>
                  <select
                    value={formData.property_condition}
                    onChange={(e) => setFormData({ ...formData, property_condition: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option value="new">New/Under Construction</option>
                    <option value="resale">Resale</option>
                    <option value="ready-to-move">Ready to Move</option>
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    BHK
                  </label>
                  <input
                    type="number"
                    value={formData.bhk}
                    onChange={(e) => setFormData({ ...formData, bhk: parseInt(e.target.value) || 0 })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    min="1"
                    max="10"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Built-up Area (sq ft) *
                  </label>
                  <input
                    type="number"
                    value={formData.builtup_area_sqft}
                    onChange={(e) => setFormData({ ...formData, builtup_area_sqft: parseInt(e.target.value) || 0 })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    min="100"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Price (₹) *
                  </label>
                  <input
                    type="number"
                    value={formData.price}
                    onChange={(e) => setFormData({ ...formData, price: parseInt(e.target.value) || 0 })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    min="100000"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Listed Date
                  </label>
                  <input
                    type="date"
                    value={formData.listed_date}
                    onChange={(e) => setFormData({ ...formData, listed_date: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                </div>
              </div>

              {/* Action Buttons */}
              <div className="mt-6 pt-4 border-t border-gray-200">
                <div className="flex justify-end space-x-3">
                  <button
                    onClick={onClose}
                    className="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors"
                  >
                    Cancel
                  </button>
                  <button
                    onClick={handleAnalyze}
                    disabled={isLoading}
                    className="px-8 py-3 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center space-x-2 font-semibold"
                  >
                    {isLoading && (
                      <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-gray-700"></div>
                    )}
                    <span>{isLoading ? 'Analyzing...' : 'Analyze Property'}</span>
                  </button>
                </div>
              </div>
            </div>
          ) : (
            /* Analytics Results */
            <div className="space-y-6">
              {/* Fallback Warning */}
              {analytics.fallback_used && (
                <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                  <div className="flex items-start space-x-3">
                    <AlertCircle className="w-5 h-5 text-yellow-600 mt-0.5" />
                    <div>
                      <h3 className="font-medium text-yellow-900">Limited Locality Data</h3>
                      <p className="text-sm text-yellow-700 mt-1">
                        {analytics.fallback_reason || 'Analysis expanded to nearby localities due to limited comparable properties in your immediate area.'}
                      </p>
                    </div>
                  </div>
                </div>
              )}

              {/* Property Summary */}
              <div className="bg-gray-50 rounded-lg p-4">
                <div className="flex items-center space-x-2 mb-3">
                  <Home className="w-5 h-5 text-gray-600" />
                  <h3 className="font-medium text-gray-900">Your Property</h3>
                  {getConfidenceBadge(analytics.confidence_level)}
                </div>
                <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                  <div>
                    <span className="text-gray-600">Location:</span>
                    <p className="font-medium">{analytics.locality}, {analytics.city}</p>
                  </div>
                  <div>
                    <span className="text-gray-600">Area:</span>
                    <p className="font-medium">{formData.builtup_area_sqft.toLocaleString()} sq ft</p>
                  </div>
                  <div>
                    <span className="text-gray-600">Total Price:</span>
                    <p className="font-medium">{formatCurrency(formData.price)}</p>
                  </div>
                  <div>
                    <span className="text-gray-600">Your Rate:</span>
                    <p className="font-medium text-blue-600">{formatCurrency(analytics.seller_price_per_sqft)}/sq ft</p>
                    <p className="text-xs text-gray-500">({formData.price} ÷ {formData.builtup_area_sqft})</p>
                  </div>
                </div>
              </div>

              {/* Price Per Sqft Calculation Summary */}
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 max-h-96 overflow-y-auto">
                <div className="flex items-center justify-between mb-3">
                  <div className="flex items-center space-x-2">
                    <BarChart3 className="w-5 h-5 text-blue-600" />
                    <h3 className="font-medium text-blue-900">Price per Square Feet Analysis</h3>
                  </div>
                  <button
                    onClick={() => setAnalytics(null)}
                    className="p-1 hover:bg-blue-100 rounded transition-colors"
                    title="Close"
                  >
                    <X className="w-4 h-4 text-blue-700" />
                  </button>
                </div>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="bg-white rounded-lg p-3 border border-blue-100">
                    <h4 className="text-sm font-medium text-gray-700 mb-2">Your Calculation</h4>
                    <div className="text-lg font-semibold text-blue-600">
                      {formatCurrency(analytics.seller_price_per_sqft)}/sq ft
                    </div>
                    <p className="text-xs text-gray-500 mt-1">
                      {formatCurrency(formData.price)} (Total Price) ÷ {formData.builtup_area_sqft.toLocaleString()} sq ft (Area)
                    </p>
                  </div>
                  <div className="bg-white rounded-lg p-3 border border-blue-100">
                    <h4 className="text-sm font-medium text-gray-700 mb-2">Market Comparison</h4>
                    <div className="space-y-1 text-sm">
                      <div className="flex justify-between">
                        <span className="text-gray-600">Median:</span>
                        <span className="font-medium">{formatCurrency(analytics.locality_median_price_per_sqft)}/sq ft</span>
                      </div>
                      <div className="flex justify-between">
                        <span className="text-gray-600">Average:</span>
                        <span className="font-medium">{formatCurrency(analytics.locality_average_price_per_sqft)}/sq ft</span>
                      </div>
                      <div className="flex justify-between">
                        <span className="text-gray-600">Range:</span>
                        <span className="font-medium">{formatCurrency(analytics.price_range_per_sqft.min)} - {formatCurrency(analytics.price_range_per_sqft.max)}/sq ft</span>
                      </div>
                    </div>
                  </div>
                </div>
              </div>

              {/* Verdict */}
              <div className={`border rounded-lg p-4 ${getVerdictColor(analytics.verdict)}`}>
                <div className="flex items-center space-x-3 mb-2">
                  {getVerdictIcon(analytics.verdict)}
                  <h3 className="font-semibold text-lg">{analytics.verdict}</h3>
                  <span className="text-sm font-medium bg-white bg-opacity-50 px-2 py-1 rounded">
                    {analytics.difference_percentage > 0 ? '+' : ''}{analytics.difference_percentage.toFixed(1)}% vs median
                  </span>
                </div>
                <p className="text-sm font-medium">{analytics.seller_insight}</p>
              </div>

              {/* Market Benchmarks */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <div className="bg-white border border-gray-200 rounded-lg p-4">
                  <div className="flex items-center space-x-2 mb-2">
                    <BarChart3 className="w-4 h-4 text-blue-600" />
                    <h4 className="font-medium text-gray-900">Your Rate</h4>
                  </div>
                  <p className="text-2xl font-bold text-gray-900">
                    {formatCurrency(analytics.seller_price_per_sqft)}
                  </p>
                  <p className="text-sm text-gray-600">per sq ft</p>
                </div>

                <div className="bg-white border border-gray-200 rounded-lg p-4">
                  <div className="flex items-center space-x-2 mb-2">
                    <Minus className="w-4 h-4 text-green-600" />
                    <h4 className="font-medium text-gray-900">Market Median</h4>
                  </div>
                  <p className="text-2xl font-bold text-green-600">
                    {formatCurrency(analytics.locality_median_price_per_sqft)}
                  </p>
                  <p className="text-sm text-gray-600">per sq ft</p>
                </div>

                <div className="bg-white border border-gray-200 rounded-lg p-4">
                  <div className="flex items-center space-x-2 mb-2">
                   <IndianRupee className="w-4 h-4 text-blue-600" />
                    <h4 className="font-medium text-gray-900">Market Average</h4>
                  </div>
                  <p className="text-2xl font-bold text-blue-600">
                    {formatCurrency(analytics.locality_average_price_per_sqft)}
                  </p>
                  <p className="text-sm text-gray-600">per sq ft</p>
                </div>

                <div className="bg-white border border-gray-200 rounded-lg p-4">
                  <div className="flex items-center space-x-2 mb-2">
                    <MapPin className="w-4 h-4 text-purple-600" />
                    <h4 className="font-medium text-gray-900">Price Range</h4>
                  </div>
                  <p className="text-lg font-bold text-purple-600">
                    {formatCurrency(analytics.price_range_per_sqft.min)} - {formatCurrency(analytics.price_range_per_sqft.max)}
                  </p>
                  <p className="text-sm text-gray-600">per sq ft</p>
                </div>
              </div>

              {/* Comparable Properties Info */}
              <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
                <div className="flex items-center space-x-2 mb-3">
                  <CheckCircle className="w-5 h-5 text-green-600" />
                  <h4 className="font-medium text-gray-900">Competitor Analysis Summary</h4>
                </div>
                <div className="space-y-3">
                  <p className="text-sm text-gray-700">
                    Analysis based on <strong className="text-blue-600">{analytics.comparable_property_count}</strong> comparable properties 
                    in <strong>{analytics.locality}</strong>, {analytics.city}.
                  </p>
                  <div className="bg-white rounded-lg p-3 border border-gray-200">
                    <h5 className="text-sm font-medium text-gray-700 mb-2">Competitor Price per Sqft Average</h5>
                    <div className="grid grid-cols-3 gap-3">
                      <div className="text-center">
                        <p className="text-xs text-gray-500">Median</p>
                        <p className="text-lg font-bold text-green-600">{formatCurrency(analytics.locality_median_price_per_sqft)}</p>
                      </div>
                      <div className="text-center">
                        <p className="text-xs text-gray-500">Average</p>
                        <p className="text-lg font-bold text-blue-600">{formatCurrency(analytics.locality_average_price_per_sqft)}</p>
                      </div>
                      <div className="text-center">
                        <p className="text-xs text-gray-500">Your Rate</p>
                        <p className="text-lg font-bold text-gray-900">{formatCurrency(analytics.seller_price_per_sqft)}</p>
                      </div>
                    </div>
                  </div>
                  <p className="text-xs text-gray-500">
                    * Outliers (top/bottom 10%) have been excluded for reliable market benchmarks.
                  </p>
                </div>
              </div>

              {/* Action Buttons */}
              <div className="flex justify-end space-x-3 pt-4 border-t border-gray-200">
                <button
                  onClick={() => setAnalytics(null)}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                >
                  Analyze Again
                </button>
                <button
                  onClick={onClose}
                  className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                >
                  Close
                </button>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
    </div>
  );
};
