import React, { useState, useEffect } from 'react';
import { Shield, CheckCircle, XCircle, AlertCircle, FileText } from 'lucide-react';
import axios from 'axios';

interface ConsentForm {
  id: number;
  title: string;
  type: string;
  content: string;
  version: string;
  is_required: boolean;
  created_at: string;
}

interface UserConsent {
  id: number;
  consent_form_id: number;
  accepted: boolean;
  accepted_at: string;
}

export function UserCompliance() {
  const [consentForms, setConsentForms] = useState<ConsentForm[]>([]);
  const [userConsents, setUserConsents] = useState<UserConsent[]>([]);
  const [loading, setLoading] = useState(false);
  const [selectedForm, setSelectedForm] = useState<ConsentForm | null>(null);

  const API_BASE = 'http://localhost:8090/api';

  useEffect(() => {
    fetchConsentForms();
    fetchUserConsents();
  }, []);

  const fetchConsentForms = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('access_token');
      const response = await axios.get(`${API_BASE}/compliance/consent-forms`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setConsentForms(response.data || []);
    } catch (error) {
      console.error('Failed to fetch consent forms:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchUserConsents = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const response = await axios.get(`${API_BASE}/compliance/my-consents`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setUserConsents(response.data || []);
    } catch (error) {
      console.error('Failed to fetch user consents:', error);
    }
  };

  const handleAcceptReject = async (consentFormId: number, accepted: boolean) => {
    try {
      const token = localStorage.getItem('access_token');
      await axios.post(
        `${API_BASE}/compliance/accept-consent`,
        { consent_form_id: consentFormId, accepted },
        { headers: { Authorization: `Bearer ${token}` } }
      );
      setSelectedForm(null);
      await fetchUserConsents();
      await fetchConsentForms();
    } catch (error) {
      console.error('Failed to save consent:', error);
    }
  };

  const getConsentStatus = (formId: number) => {
    return userConsents.find(c => c.consent_form_id === formId);
  };

  const getStatusBadge = (formId: number) => {
    const consent = getConsentStatus(formId);
    if (!consent) {
      return <span className="px-3 py-1 text-xs font-semibold bg-yellow-100 text-yellow-700 rounded-full">Pending</span>;
    }
    if (consent.accepted) {
      return <span className="px-3 py-1 text-xs font-semibold bg-green-100 text-green-700 rounded-full flex items-center gap-1"><CheckCircle className="w-3 h-3" />Accepted</span>;
    }
    return <span className="px-3 py-1 text-xs font-semibold bg-red-100 text-red-700 rounded-full flex items-center gap-1"><XCircle className="w-3 h-3" />Rejected</span>;
  };

  return (
    <div className="space-y-6 p-6 bg-gray-50 min-h-screen">
      <div className="bg-gradient-to-r from-blue-600 to-indigo-600 rounded-xl shadow-lg p-8">
        <div className="flex items-center gap-3">
          <Shield className="w-10 h-10" style={{ color: '#ffffff' }} />
          <div>
            <h1 className="text-3xl font-bold" style={{ color: '#000000' }}>Compliance Rules</h1>
            <p className="mt-1" style={{ color: '#1f2937' }}>Review and manage your consent preferences</p>
          </div>
        </div>
      </div>

      {loading ? (
        <div className="flex justify-center py-12">
          <div className="animate-spin rounded-full h-12 w-12 border-4 border-blue-600 border-t-transparent"></div>
        </div>
      ) : (
        <div className="grid gap-4">
          {consentForms.length === 0 ? (
            <div className="bg-white rounded-xl shadow-md p-12 text-center">
              <FileText className="w-16 h-16 text-gray-300 mx-auto mb-4" />
              <p className="text-gray-500 text-lg">No consent forms available</p>
            </div>
          ) : (
            consentForms.map((form) => {
              const consent = getConsentStatus(form.id);
              return (
                <div key={form.id} className="bg-white rounded-xl shadow-md hover:shadow-lg transition-shadow p-6 border border-gray-200">
                  <div className="flex items-start justify-between gap-4">
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center gap-3 mb-3 flex-wrap">
                        <FileText className="w-6 h-6 text-blue-600 flex-shrink-0" />
                        <h3 className="text-xl font-bold" style={{ color: '#111827' }}>{form.title}</h3>
                        {form.is_required && (
                          <span className="px-2 py-1 text-xs font-semibold bg-red-100 text-red-700 rounded-full">Required</span>
                        )}
                        <span className="px-2 py-1 text-xs font-semibold bg-gray-100 text-gray-700 rounded-full">v{form.version}</span>
                      </div>
                      <p className="text-gray-600 line-clamp-2 mb-3" style={{ color: '#4b5563' }}>{form.content}</p>
                      <div className="flex items-center gap-4 flex-wrap">
                        {getStatusBadge(form.id)}
                        {consent && (
                          <span className="text-sm" style={{ color: '#6b7280' }}>
                            {consent.accepted ? 'Accepted' : 'Rejected'} on {new Date(consent.accepted_at).toLocaleDateString()}
                          </span>
                        )}
                      </div>
                    </div>
                    <div className="flex gap-2 flex-shrink-0">
                      {!consent ? (
                        <>
                          <button
                            onClick={() => handleAcceptReject(form.id, true)}
                            className="px-4 py-2 rounded-lg font-semibold flex items-center gap-2"
                            style={{ backgroundColor: '#10b981', color: 'white' }}
                          >
                            <CheckCircle className="w-4 h-4" />
                            Accept
                          </button>
                          <button
                            onClick={() => handleAcceptReject(form.id, false)}
                            className="px-4 py-2 rounded-lg font-semibold flex items-center gap-2"
                            style={{ backgroundColor: '#ef4444', color: 'white' }}
                          >
                            <XCircle className="w-4 h-4" />
                            Reject
                          </button>
                        </>
                      ) : null}
                    </div>
                  </div>
                </div>
              );
            })
          )}
        </div>
      )}

      {selectedForm && (
        <div className="fixed inset-0 flex items-center justify-center p-4" style={{ zIndex: 9999, backgroundColor: 'rgba(0, 0, 0, 0.5)' }}>
          <div className="bg-white rounded-2xl shadow-2xl w-full max-w-3xl max-h-[90vh] overflow-y-auto">
            <div className="bg-gradient-to-r from-blue-600 to-indigo-600 p-6 rounded-t-2xl sticky top-0">
              <h2 className="text-2xl font-bold" style={{ color: '#000000' }}>{selectedForm.title}</h2>
              <p className="text-blue-100 mt-1">Version {selectedForm.version}</p>
            </div>
            <div className="p-6" style={{ backgroundColor: '#ffffff' }}>
              <div className="prose max-w-none">
                <p className="text-gray-700 whitespace-pre-wrap" style={{ color: '#374151' }}>{selectedForm.content}</p>
              </div>
              {selectedForm.is_required && (
                <div className="mt-6 p-4 bg-yellow-50 border border-yellow-200 rounded-lg flex items-start gap-3">
                  <AlertCircle className="w-5 h-5 text-yellow-600 flex-shrink-0 mt-0.5" />
                  <p className="text-sm text-yellow-800">This consent is required. You must accept to continue using the platform.</p>
                </div>
              )}
            </div>
            <div className="flex gap-3 p-6 bg-gray-50 rounded-b-2xl">
              {!getConsentStatus(selectedForm.id) ? (
                <>
                  <button
                    onClick={() => handleAcceptReject(selectedForm.id, true)}
                    className="flex-1 px-6 py-3 rounded-lg font-semibold shadow-lg transition-all flex items-center justify-center gap-2"
                    style={{ backgroundColor: '#10b981', color: 'white' }}
                  >
                    <CheckCircle className="w-5 h-5" />
                    Accept
                  </button>
                  <button
                    onClick={() => handleAcceptReject(selectedForm.id, false)}
                    className="flex-1 px-6 py-3 rounded-lg font-semibold shadow-lg transition-all flex items-center justify-center gap-2"
                    style={{ backgroundColor: '#ef4444', color: 'white' }}
                  >
                    <XCircle className="w-5 h-5" />
                    Reject
                  </button>
                </>
              ) : null}
              <button
                onClick={() => setSelectedForm(null)}
                className="px-6 py-3 rounded-lg font-semibold transition-all"
                style={{ backgroundColor: 'white', border: '2px solid #d1d5db', color: '#374151' }}
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
