import React, { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Textarea } from '@/components/ui/textarea';
import { CheckCircle, XCircle, Clock, FileText } from 'lucide-react';

interface ContractActionsProps {
  contractId: string;
  currentStage: string;
  status: string;
  onAction: () => void;
}

export const ContractActions: React.FC<ContractActionsProps> = ({
  contractId,
  currentStage,
  status,
  onAction
}) => {
  const [comments, setComments] = useState('');
  const [loading, setLoading] = useState(false);

  const handleApprove = async () => {
    setLoading(true);
    try {
      const response = await fetch(`/api/contracts/${contractId}/approve`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('access_token')}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          stage: currentStage,
          comments: comments.trim() || undefined
        })
      });

      if (response.ok) {
        onAction();
        setComments('');
      } else {
        console.error('Failed to approve stage');
      }
    } catch (error) {
      console.error('Error approving stage:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleReject = async () => {
    if (!comments.trim()) {
      alert('Comments are required for rejection');
      return;
    }

    setLoading(true);
    try {
      const response = await fetch(`/api/contracts/${contractId}/reject`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('access_token')}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          stage: currentStage,
          comments: comments.trim()
        })
      });

      if (response.ok) {
        onAction();
        setComments('');
      } else {
        console.error('Failed to reject stage');
      }
    } catch (error) {
      console.error('Error rejecting stage:', error);
    } finally {
      setLoading(false);
    }
  };

  const getStageInfo = (stage: string) => {
    switch (stage) {
      case 'draft':
        return { icon: <FileText className="w-4 h-4" />, label: 'Draft Review', color: 'bg-gray-100 text-gray-800' };
      case 'review':
        return { icon: <Clock className="w-4 h-4" />, label: 'Owner Review', color: 'bg-yellow-100 text-yellow-800' };
      case 'approval':
        return { icon: <CheckCircle className="w-4 h-4" />, label: 'Admin Approval', color: 'bg-blue-100 text-blue-800' };
      case 'execution':
        return { icon: <CheckCircle className="w-4 h-4" />, label: 'Digital Signing', color: 'bg-green-100 text-green-800' };
      default:
        return { icon: <FileText className="w-4 h-4" />, label: stage, color: 'bg-gray-100 text-gray-800' };
    }
  };

  const stageInfo = getStageInfo(currentStage);

  if (status === 'completed' || status === 'active') {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <CheckCircle className="w-5 h-5 text-green-600" />
            Contract {status === 'active' ? 'Active' : 'Completed'}
          </CardTitle>
        </CardHeader>
        <CardContent>
          <p className="text-sm text-gray-600">
            This contract is {status === 'active' ? 'currently active' : 'completed'} and no further actions are required.
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          {stageInfo.icon}
          Pending Action
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        <div>
          <Badge className={stageInfo.color}>
            {stageInfo.label}
          </Badge>
          <p className="text-sm text-gray-600 mt-2">
            This contract is waiting for {stageInfo.label.toLowerCase()}.
          </p>
        </div>

        <div>
          <label className="text-sm font-medium text-gray-700 mb-2 block">
            Comments (Optional)
          </label>
          <Textarea
            value={comments}
            onChange={(e) => setComments(e.target.value)}
            placeholder="Add your comments here..."
            rows={3}
          />
        </div>

        <div className="flex gap-3">
          <Button
            onClick={handleApprove}
            disabled={loading}
            className="bg-green-600 hover:bg-green-700 flex-1"
          >
            <CheckCircle className="w-4 h-4 mr-2" />
            {loading ? 'Processing...' : 'Approve'}
          </Button>
          <Button
            onClick={handleReject}
            disabled={loading}
            variant="outline"
            className="border-red-200 text-red-600 hover:bg-red-50 flex-1"
          >
            <XCircle className="w-4 h-4 mr-2" />
            {loading ? 'Processing...' : 'Reject'}
          </Button>
        </div>
      </CardContent>
    </Card>
  );
};