import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { X, FileText, Calendar, DollarSign, Users, CheckCircle } from 'lucide-react';

interface Contract {
  id: string;
  title: string;
  type: string;
  status: string;
  property_id?: string;
  description?: string;
  terms?: string;
  amount?: number;
  start_date: string;
  end_date?: string;
  parties: string[];
  progress: number;
  created_at: string;
}

interface ContractDetailModalProps {
  contract: Contract;
  onClose: () => void;
}

export const ContractDetailModal: React.FC<ContractDetailModalProps> = ({ contract, onClose }) => {
  const getStatusColor = (status: string) => {
    switch (status) {
      case 'draft': return 'bg-gray-100 text-gray-800';
      case 'review': return 'bg-yellow-100 text-yellow-800';
      case 'approval': return 'bg-blue-100 text-blue-800';
      case 'active': return 'bg-green-100 text-green-800';
      case 'completed': return 'bg-purple-100 text-purple-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <Card className="w-full max-w-4xl max-h-[90vh] overflow-y-auto">
        <CardHeader className="flex flex-row items-center justify-between border-b">
          <div className="flex items-center gap-3">
            <FileText className="w-6 h-6 text-blue-600" />
            <div>
              <CardTitle>{contract.title}</CardTitle>
              <p className="text-sm text-gray-500 mt-1">Contract ID: {contract.id}</p>
            </div>
          </div>
          <Button variant="ghost" size="sm" onClick={onClose}>
            <X className="w-4 h-4" />
          </Button>
        </CardHeader>
        
        <CardContent className="p-6 space-y-6">
          {/* Status and Type */}
          <div className="flex gap-2">
            <Badge className={getStatusColor(contract.status)}>
              {contract.status.toUpperCase()}
            </Badge>
            <Badge variant="outline">{contract.type}</Badge>
          </div>

          {/* Progress Bar */}
          <div>
            <div className="flex justify-between mb-2">
              <span className="text-sm font-medium">Progress</span>
              <span className="text-sm text-gray-600">{contract.progress}%</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div 
                className="bg-blue-600 h-2 rounded-full transition-all"
                style={{ width: `${contract.progress}%` }}
              />
            </div>
          </div>

          {/* Details Grid */}
          <div className="grid grid-cols-2 gap-6">
            <div>
              <div className="flex items-center gap-2 mb-2">
                <Calendar className="w-4 h-4 text-gray-500" />
                <span className="text-sm font-medium">Start Date</span>
              </div>
              <p className="text-gray-900">{new Date(contract.start_date).toLocaleDateString()}</p>
            </div>

            {contract.end_date && (
              <div>
                <div className="flex items-center gap-2 mb-2">
                  <Calendar className="w-4 h-4 text-gray-500" />
                  <span className="text-sm font-medium">End Date</span>
                </div>
                <p className="text-gray-900">{new Date(contract.end_date).toLocaleDateString()}</p>
              </div>
            )}

            {contract.amount && (
              <div>
                <div className="flex items-center gap-2 mb-2">
                  <DollarSign className="w-4 h-4 text-gray-500" />
                  <span className="text-sm font-medium">Amount</span>
                </div>
                <p className="text-gray-900 font-semibold">₹{contract.amount.toLocaleString()}</p>
              </div>
            )}

            <div>
              <div className="flex items-center gap-2 mb-2">
                <FileText className="w-4 h-4 text-gray-500" />
                <span className="text-sm font-medium">Created</span>
              </div>
              <p className="text-gray-900">{new Date(contract.created_at).toLocaleDateString()}</p>
            </div>
          </div>

          {/* Parties */}
          <div>
            <div className="flex items-center gap-2 mb-3">
              <Users className="w-4 h-4 text-gray-500" />
              <span className="text-sm font-medium">Contract Parties</span>
            </div>
            <div className="flex flex-wrap gap-2">
              {(contract.parties || []).map((party, index) => (
                <Badge key={index} variant="secondary" className="px-3 py-1">
                  {party}
                </Badge>
              ))}
            </div>
          </div>

          {/* Description */}
          {contract.description && (
            <div>
              <h3 className="text-sm font-medium mb-2">Description</h3>
              <p className="text-gray-700 bg-gray-50 p-4 rounded-lg">{contract.description}</p>
            </div>
          )}

          {/* Terms */}
          {contract.terms && (
            <div>
              <h3 className="text-sm font-medium mb-2">Terms & Conditions</h3>
              <div className="text-gray-700 bg-gray-50 p-4 rounded-lg whitespace-pre-wrap">
                {contract.terms}
              </div>
            </div>
          )}

          {/* Property ID */}
          {contract.property_id && (
            <div>
              <h3 className="text-sm font-medium mb-2">Property Reference</h3>
              <p className="text-gray-700 bg-gray-50 p-3 rounded-lg font-mono">{contract.property_id}</p>
            </div>
          )}

          {/* Action Buttons */}
          <div className="flex justify-end gap-2 pt-4 border-t">
            <Button variant="outline" onClick={onClose}>
              Close
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};
