import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { X, Plus, Trash2, AlertCircle, CheckCircle, FileText } from 'lucide-react';

interface ContractFormProps {
  onClose: () => void;
  onSave: () => void;
}

export const ContractForm: React.FC<ContractFormProps> = ({ onClose, onSave }) => {
  const [formData, setFormData] = useState({
    title: '',
    type: '',
    amount: '',
    start_date: '',
    end_date: '',
    selectedProperty: null as any
  });
  const [parties, setParties] = useState([
    { name: 'Buyer', email: '' },
    { name: 'Seller', email: '' }
  ]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState(false);
  const [showPropertyModal, setShowPropertyModal] = useState(false);
  const [properties, setProperties] = useState<any[]>([]);

  const fetchProperties = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('http://localhost:8090/api/properties/', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      if (response.ok) {
        const data = await response.json();
        setProperties(data.properties || data || []);
      }
    } catch (error) {
      console.error('Error fetching properties:', error);
    }
  };

  const handlePropertySelect = (property: any) => {
    setFormData(prev => ({ ...prev, selectedProperty: property }));
    setShowPropertyModal(false);
    // Don't close the main contract form - only close property modal
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setError(null);
    setSuccess(false);

    try {
      // Get current user ID from token or localStorage
      const token = localStorage.getItem('access_token');
      let currentUserId = localStorage.getItem('user_id'); // Try localStorage first
      
      if (!currentUserId && token) {
        try {
          // Try to get user info from the seller profile API
          const userResponse = await fetch('http://localhost:8090/api/user-profile/me', {
            headers: {
              'Authorization': `Bearer ${token}`,
              'Content-Type': 'application/json'
            }
          });
          
          if (userResponse.ok) {
            const userData = await userResponse.json();
            currentUserId = userData.user_id;
          }
        } catch (error) {
          console.warn('Could not fetch user info from user profile :', error);
        }
      }
      
      // If still no user ID, try to decode from token
      if (!currentUserId && token) {
        try {
          const payload = JSON.parse(atob(token.split('.')[1]));
          currentUserId = payload.user_id || payload.sub;
        } catch (error) {
          console.warn('Could not decode token:', error);
        }
      }
      
      // If we still can't get the current user, show an error
      if (!currentUserId) {
        setError('Please log in to create a contract.');
        return;
      }

      const response = await fetch('http://localhost:8090/api/contracts/', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          property_id: formData.selectedProperty?.id || 4,
          buyer_id: currentUserId,
          // seller_id will be automatically fetched from property
          contract_type: formData.type === 'sale' ? 'purchase' : formData.type,
          contract_terms: 'Standard terms and conditions',
          contract_amount: parseFloat(formData.amount) || 0,
          metadata: {
            title: formData.title,
            description: formData.title,
            start_date: formData.start_date,
            end_date: formData.end_date || null,
            parties: ['Buyer', 'Seller']
          }
        })
      });

      if (response.ok) {
        setSuccess(true);
        setTimeout(() => {
          onSave();
          onClose();
        }, 1000);
      } else {
        const errorData = await response.text();
        console.error('Contract creation error:', errorData);
        setError(`Failed to create contract: ${response.status} - ${errorData}`);
      }
    } catch (error) {
      console.error('Error creating contract:', error);
      setError('Network error. Please check your connection and try again.');
    } finally {
      setLoading(false);
    }
  };

  const addParty = () => {
    setParties(prev => [...prev, { name: '', email: '' }]);
  };

  const removeParty = (index: number) => {
    if (parties.length > 2) {
      setParties(prev => prev.filter((_, i) => i !== index));
    }
  };

  const updateParty = (index: number, field: 'name', value: string) => {
    setParties(prev => prev.map((party, i) => 
      i === index ? { ...party, [field]: value } : party
    ));
  };

  return (
    <div 
      className="fixed inset-0 bg-black/60 flex items-center justify-center z-50 p-4 backdrop-blur-sm"
      onClick={onClose}
    >
      <Card 
        className="w-full max-w-3xl bg-white shadow-2xl border-0 rounded-3xl overflow-hidden"
        onClick={(e) => e.stopPropagation()}
      >
        {/* Header */}
        <div className="bg-gradient-to-br from-blue-600 via-blue-700 to-indigo-700 p-6 relative overflow-hidden">
          <div className="absolute top-0 right-0 w-32 h-32 bg-white/10 rounded-full -mr-16 -mt-16"></div>
          <div className="absolute bottom-0 left-0 w-24 h-24 bg-white/10 rounded-full -ml-12 -mb-12"></div>
          <div className="relative flex items-start justify-between">
            <div>
              <h2 className="text-2xl font-bold text-white mb-1">Create New Contract</h2>
              <p className="text-blue-100 text-sm">Fill in the details below</p>
            </div>
            <Button 
              variant="ghost" 
              size="sm" 
              onClick={onClose} 
              className="text-white hover:bg-white/20 rounded-full p-2 h-8 w-8"
            >
              <X className="w-4 h-4" />
            </Button>
          </div>
        </div>
        {/* Content */}
        <div className="p-6 max-h-[50vh] overflow-y-auto">
          {/* Messages */}
          {success && (
            <div className="mb-4 p-3 bg-emerald-50 border border-emerald-200 rounded-2xl flex items-center gap-3">
              <div className="w-6 h-6 bg-emerald-100 rounded-full flex items-center justify-center">
                <CheckCircle className="w-3 h-3 text-emerald-600" />
              </div>
              <p className="text-emerald-800 font-medium text-sm">Contract created successfully!</p>
            </div>
          )}

          {error && (
            <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-2xl flex items-start gap-3">
              <div className="w-6 h-6 bg-red-100 rounded-full flex items-center justify-center flex-shrink-0">
                <AlertCircle className="w-3 h-3 text-red-600" />
              </div>
              <div>
                <p className="text-red-800 font-medium text-sm">Something went wrong</p>
                <p className="text-red-700 text-xs mt-1">{error}</p>
              </div>
            </div>
          )}

          <form onSubmit={handleSubmit} className="space-y-4">
            {/* Basic Info */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-semibold text-gray-900 mb-2">Contract Title</label>
                <Input
                  value={formData.title}
                  onChange={(e) => setFormData(prev => ({ ...prev, title: e.target.value }))}
                  placeholder="e.g., Property Sale Agreement"
                  className="h-10 text-sm border-2 border-gray-200 focus:border-blue-500 rounded-xl transition-colors"
                  required
                />
              </div>
              <div>
                <label className="block text-sm font-semibold text-gray-900 mb-2">Contract Type</label>
                <Select value={formData.type} onValueChange={(value) => setFormData(prev => ({ ...prev, type: value }))}>
                  <SelectTrigger className="h-10 text-sm border-2 border-gray-200 focus:border-blue-500 rounded-xl">
                    <SelectValue placeholder="Choose contract type" />
                  </SelectTrigger>
                  <SelectContent className="rounded-xl border-2">
                    <SelectItem value="sale" className="text-sm py-2">🏠 Sale Agreement</SelectItem>
                    <SelectItem value="lease" className="text-sm py-2">📋 Lease Agreement</SelectItem>
                    <SelectItem value="joint_venture" className="text-sm py-2">🤝 Joint Venture</SelectItem>
                    <SelectItem value="development" className="text-sm py-2">🏗️ Development Agreement</SelectItem>
                    <SelectItem value="custom" className="text-sm py-2">⚙️ Custom Agreement</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            {/* Property Selection */}
            <div>
              <label className="block text-sm font-semibold text-gray-900 mb-3">Property</label>
              {formData.selectedProperty ? (
                <div className="border-2 border-emerald-200 rounded-2xl p-6 bg-gradient-to-r from-emerald-50 to-green-50">
                  <div className="flex justify-between items-start">
                    <div className="flex items-start gap-4">
                      <div className="w-12 h-12 bg-emerald-100 rounded-xl flex items-center justify-center">
                        <FileText className="w-6 h-6 text-emerald-600" />
                      </div>
                      <div>
                        <h4 className="font-bold text-lg text-gray-900">{formData.selectedProperty.title}</h4>
                        <p className="text-gray-600 mt-1">📍 {formData.selectedProperty.city}, {formData.selectedProperty.locality}</p>
                        <p className="text-emerald-600 font-bold text-xl mt-2">₹{formData.selectedProperty.price?.toLocaleString()}</p>
                      </div>
                    </div>
                    <Button 
                      type="button" 
                      variant="outline" 
                      onClick={() => setFormData(prev => ({ ...prev, selectedProperty: null }))}
                      className="border-2 border-gray-300 hover:border-red-400 hover:text-red-600 rounded-xl px-4 py-2"
                    >
                      Change
                    </Button>
                  </div>
                </div>
              ) : (
                <div 
                  className="border-2 border-dashed border-blue-300 rounded-2xl p-8 hover:border-blue-500 hover:bg-blue-50/50 transition-all cursor-pointer group"
                  onClick={() => {
                    fetchProperties();
                    setShowPropertyModal(true);
                  }}
                >
                  <div className="text-center">
                    <div className="w-16 h-16 bg-blue-100 rounded-2xl flex items-center justify-center mx-auto mb-4 group-hover:bg-blue-200 transition-colors">
                      <Plus className="w-8 h-8 text-blue-600" />
                    </div>
                    <h4 className="font-semibold text-gray-900 mb-2">Select Property</h4>
                    <p className="text-gray-500">Choose a property for this contract</p>
                  </div>
                </div>
              )}
            </div>

            {/* Amount and Dates */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <div>
                <label className="block text-sm font-semibold text-gray-900 mb-3">Contract Amount</label>
                <div className="relative">
                  <div className="absolute left-6 top-1/2 transform -translate-y-1/2 text-gray-500 font-semibold text-lg">₹</div>
                  <Input
                    type="number"
                    value={formData.amount}
                    onChange={(e) => setFormData(prev => ({ ...prev, amount: e.target.value }))}
                    placeholder="1,00,000"
                    className="h-12 pl-14 text-base border-2 border-gray-200 focus:border-blue-500 rounded-xl transition-colors"
                  />
                </div>
              </div>
              <div>
                <label className="block text-sm font-semibold text-gray-900 mb-3">Start Date</label>
                <Input
                  type="date"
                  value={formData.start_date}
                  onChange={(e) => setFormData(prev => ({ ...prev, start_date: e.target.value }))}
                  className="h-12 text-base border-2 border-gray-200 focus:border-blue-500 rounded-xl transition-colors"
                  required
                />
              </div>
              <div>
                <label className="block text-sm font-semibold text-gray-900 mb-3">End Date <span className="text-gray-400 font-normal">(Optional)</span></label>
                <Input
                  type="date"
                  value={formData.end_date}
                  onChange={(e) => setFormData(prev => ({ ...prev, end_date: e.target.value }))}
                  className="h-12 text-base border-2 border-gray-200 focus:border-blue-500 rounded-xl transition-colors"
                />
              </div>
            </div>

            {/* Actions */}
            <div className="flex flex-col sm:flex-row gap-4 pt-6">
              <Button 
                type="button" 
                variant="outline" 
                onClick={onClose} 
                disabled={loading}
                className="flex-1 h-12 border-2 border-gray-300 hover:border-gray-400 rounded-xl font-semibold text-base transition-colors"
              >
                Cancel
              </Button>
              <Button 
                type="submit" 
                disabled={loading || !formData.title || !formData.type || !formData.start_date}
                className="flex-1 h-12 bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white rounded-xl font-semibold text-base shadow-lg hover:shadow-xl transition-all"
              >
                {loading ? (
                  <div className="flex items-center justify-center gap-3">
                    <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                    Creating...
                  </div>
                ) : (
                  <div className="flex items-center justify-center gap-2">
                    <CheckCircle className="w-5 h-5" />
                    Create Contract
                  </div>
                )}
              </Button>
            </div>
          </form>
        </div>
      </Card>
      
      {/* Property Selection Modal */}
      {showPropertyModal && (
        <div 
          className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-60 p-4"
          onClick={(e) => {
            e.stopPropagation();
            setShowPropertyModal(false);
          }}
        >
          <div 
            className="w-full max-w-4xl h-[80vh] flex flex-col bg-white overflow-hidden"
            onClick={(e) => e.stopPropagation()}
          >
            <div className="flex flex-row items-center justify-between border-b flex-shrink-0 p-6">
              <h3 className="text-lg font-semibold">Select Property</h3>
              <Button 
                variant="ghost" 
                size="sm" 
                onClick={() => setShowPropertyModal(false)}
              >
                <X className="w-5 h-5" />
              </Button>
            </div>
            <div className="flex-1 overflow-auto" style={{maxHeight: '60vh'}}>
              <div className="p-6">
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {properties.map((property) => (
                  <Card 
                    key={property.id} 
                    className="cursor-pointer hover:shadow-lg transition-shadow border-2 hover:border-blue-500"
                    onClick={(e) => {
                      e.stopPropagation();
                      handlePropertySelect(property);
                    }}
                  >
                    <CardContent className="p-4">
                      <h4 className="font-semibold text-lg mb-2 line-clamp-2">{property.title}</h4>
                      <p className="text-gray-600 mb-2">📍 {property.city}, {property.locality}</p>
                      <p className="text-blue-600 font-bold text-xl mb-3">₹{property.price?.toLocaleString('en-IN') || 'Price on request'}</p>
                      
                      {/* Additional Information */}
                      <div className="space-y-1 mb-3 text-sm">
                        <div className="flex justify-between">
                          <span className="text-gray-600">Type:</span>
                          <span className="font-medium capitalize">{property.property_type || property.type || 'N/A'}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-gray-600">Area:</span>
                          <span className="font-medium">{property.area || property.built_up_area || property.carpet_area || 'N/A'} {(property.area || property.built_up_area || property.carpet_area) ? 'sq ft' : ''}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-gray-600">Bedrooms:</span>
                          <span className="font-medium">{property.bedrooms || property.bedroom_count || 'N/A'}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-gray-600">Bathrooms:</span>
                          <span className="font-medium">{property.bathrooms || property.bathroom_count || 'N/A'}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-gray-600">Floors:</span>
                          <span className="font-medium">{property.floors || property.total_floors || property.floor_number || 'N/A'}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-gray-600">Furnished:</span>
                          <span className="font-medium capitalize">{property.furnished_status || property.furnishing || property.furnished || 'N/A'}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-gray-600">Parking:</span>
                          <span className="font-medium">{(property.parking || property.parking_spaces || property.car_parking) ? 'Available' : 'Not Available'}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-gray-600">Age:</span>
                          <span className="font-medium">{property.property_age || property.age || property.construction_year || 'N/A'}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-gray-600">Facing:</span>
                          <span className="font-medium capitalize">{property.facing || property.direction || 'N/A'}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-gray-600">Status:</span>
                          <span className={`font-medium capitalize ${
                            property.status === 'available' ? 'text-green-600' : 
                            property.status === 'sold' ? 'text-red-600' : 'text-orange-600'
                          }`}>
                            {property.status || property.availability_status || 'N/A'}
                          </span>
                        </div>
                      </div>
                      
                      <div className="flex gap-1 flex-wrap">
                        <Badge variant="outline" className="text-xs">{property.property_type || property.type}</Badge>
                        <Badge variant="outline" className="text-xs">{property.bedrooms || property.bedroom_count}BHK</Badge>
                        {(property.bathrooms || property.bathroom_count) && <Badge variant="outline" className="text-xs">{property.bathrooms || property.bathroom_count} Bath</Badge>}
                        {(property.balconies || property.balcony_count) && <Badge variant="outline" className="text-xs">{property.balconies || property.balcony_count} Balcony</Badge>}
                      </div>
                    </CardContent>
                  </Card>
                ))}
                </div>
              </div>
              {properties.length === 0 && (
                <div className="text-center py-12">
                  <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <FileText className="w-8 h-8 text-gray-400" />
                  </div>
                  <p className="text-gray-500 text-lg">No properties found</p>
                  <p className="text-gray-400 text-sm mt-1">Try adjusting your search criteria</p>
                </div>
              )}
            </div>
          </div>
        </div>
      )}
    </div>
  );
};