import React, { useState } from 'react';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent } from '@/components/ui/card';
import { Eye, Edit, Trash2, CheckCircle, Clock, AlertCircle, MapPin, Mail, DollarSign, Calendar, Home, User, PenTool, Download, FileText } from 'lucide-react';
import { ContractDetailModal } from './ContractDetailModal';
import { ContractSigningModal } from './ContractSigningModal';
import { StampPaperViewer } from './StampPaperViewer';

interface Contract {
  id: string;
  title: string;
  type: string;
  status: 'draft' | 'review' | 'approval' | 'active' | 'completed' | 'pending_signatures' | 'signed';
  property?: string;
  property_location?: string;
  property_type?: string;
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
  sale_price?: number;
  monthly_rent?: number;
  property_email?: string;
  property_owner?: string;
  parties: string[];
  party_emails?: string[];
  amount?: number;
  start_date?: string;
  end_date?: string;
  created_at: string;
  next_milestone?: string;
  progress: number;
  buyer_id?: string;
  seller_id?: string;
}

interface ContractListProps {
  contracts: Contract[];
  onRefresh: () => void;
}

export const ContractList: React.FC<ContractListProps> = ({ contracts, onRefresh }) => {
  const [selectedContract, setSelectedContract] = useState<Contract | null>(null);
  const [showDetailModal, setShowDetailModal] = useState(false);
  const [showSigningModal, setShowSigningModal] = useState(false);
  const [showStampPaperViewer, setShowStampPaperViewer] = useState(false);
  const [contractToSign, setContractToSign] = useState<string | null>(null);
  const [stampPaperContractId, setStampPaperContractId] = useState<string | null>(null);

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'draft': return 'bg-gray-100 text-gray-800';
      case 'review': return 'bg-yellow-100 text-yellow-800';
      case 'approval': return 'bg-blue-100 text-blue-800';
      case 'pending_signatures': return 'bg-orange-100 text-orange-800';
      case 'signed': return 'bg-green-100 text-green-800';
      case 'active': return 'bg-green-100 text-green-800';
      case 'completed': return 'bg-purple-100 text-purple-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'draft': return <Edit className="w-4 h-4" />;
      case 'review': return <Clock className="w-4 h-4" />;
      case 'approval': return <AlertCircle className="w-4 h-4" />;
      case 'pending_signatures': return <PenTool className="w-4 h-4" />;
      case 'signed': return <CheckCircle className="w-4 h-4" />;
      case 'active': return <CheckCircle className="w-4 h-4" />;
      case 'completed': return <CheckCircle className="w-4 h-4" />;
      default: return <Edit className="w-4 h-4" />;
    }
  };

  const getStatusMessage = (status: string) => {
    switch (status) {
      case 'draft': return 'Waiting for seller approval';
      case 'review': return 'Contract under review';
      case 'pending_signatures': return 'Waiting for digital signatures';
      case 'signed': return 'Contract signed by all parties';
      case 'active': return 'Contract is active and in effect';
      case 'completed': return 'Contract completed successfully';
      default: return status;
    }
  };

  const handleViewContract = (contract: Contract) => {
    setSelectedContract(contract);
    setShowDetailModal(true);
  };

  const handleSignContract = (contractId: string) => {
    setContractToSign(contractId);
    setShowSigningModal(true);
  };

  const canUserApprove = (contract: Contract) => {
    // Get current user ID from localStorage
    let currentUserId = null;
    try {
      // Try seller-specific storage first
      currentUserId = localStorage.getItem('sellerId') || localStorage.getItem('user_id');
      
      // Try to get from sellerToken
      if (!currentUserId) {
        const sellerToken = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
        if (sellerToken && sellerToken.includes('.')) {
          try {
            const payload = JSON.parse(atob(sellerToken.split('.')[1]));
            currentUserId = payload.user_id || payload.sub || payload.id;
          } catch (e) {
            console.warn('Failed to decode token');
          }
        }
      }
      
      // Try user object
      if (!currentUserId) {
        const user = JSON.parse(localStorage.getItem('user') || '{}');
        currentUserId = user.user_id || user.id;
      }
      
      // Try sellerData object
      if (!currentUserId) {
        const sellerData = JSON.parse(localStorage.getItem('sellerData') || '{}');
        currentUserId = sellerData.user_id || sellerData.id;
      }
    } catch (e) {
      console.error('Error getting user ID:', e);
      return false;
    }
    
    console.log('canUserApprove check:', { currentUserId, seller_id: contract.seller_id, status: contract.status });
    
    // Show approve button for draft contracts where user is seller
    return contract.status === 'draft' && contract.seller_id === currentUserId;
  };

  const canUserSign = (contract: Contract) => {
    // Get current user ID from localStorage
    let currentUserId = null;
    try {
      // Try seller-specific storage first
      currentUserId = localStorage.getItem('sellerId') || localStorage.getItem('user_id');
      
      // Try to get from sellerToken
      if (!currentUserId) {
        const sellerToken = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
        if (sellerToken && sellerToken.includes('.')) {
          try {
            const payload = JSON.parse(atob(sellerToken.split('.')[1]));
            currentUserId = payload.user_id || payload.sub || payload.id;
          } catch (e) {
            console.warn('Failed to decode token');
          }
        }
      }
      
      // Try user object
      if (!currentUserId) {
        const user = JSON.parse(localStorage.getItem('user') || '{}');
        currentUserId = user.user_id || user.id;
      }
      
      // Try sellerData object
      if (!currentUserId) {
        const sellerData = JSON.parse(localStorage.getItem('sellerData') || '{}');
        currentUserId = sellerData.user_id || sellerData.id;
      }
    } catch (e) {
      console.error('Error getting user ID:', e);
      return false;
    }
    
    // Show sign button only for pending_signatures contracts where user is buyer or seller
    return contract.status === 'pending_signatures' && 
           (contract.buyer_id === currentUserId || contract.seller_id === currentUserId);
  };

  const handleApproveContract = async (contractId: string) => {
    try {
      // Get current user ID from localStorage
      let currentUserId = null;
      try {
        // Try seller-specific storage first
        currentUserId = localStorage.getItem('sellerId') || localStorage.getItem('user_id');
        
        // Try to get from sellerToken
        if (!currentUserId) {
          const sellerToken = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
          if (sellerToken && sellerToken.includes('.')) {
            try {
              const payload = JSON.parse(atob(sellerToken.split('.')[1]));
              currentUserId = payload.user_id || payload.sub || payload.id;
            } catch (e) {
              console.warn('Failed to decode token');
            }
          }
        }
        
        // Try user object
        if (!currentUserId) {
          const user = JSON.parse(localStorage.getItem('user') || '{}');
          currentUserId = user.user_id || user.id;
        }
        
        // Try sellerData object
        if (!currentUserId) {
          const sellerData = JSON.parse(localStorage.getItem('sellerData') || '{}');
          currentUserId = sellerData.user_id || sellerData.id;
        }
        
        if (!currentUserId) {
          console.error('No user ID found');
          alert('Please log in to approve contracts');
          return;
        }
      } catch (e) {
        console.error('Error getting user ID:', e);
        alert('Failed to get user information');
        return;
      }
      
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/approve?user_id=${currentUserId}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        onRefresh();
        // After approval, show signing modal
        setContractToSign(contractId);
        setShowSigningModal(true);
      } else {
        const errorData = await response.json();
        console.error('Failed to approve contract:', errorData);
      }
    } catch (error) {
      console.error('Error approving contract:', error);
    }
  };

  const handleRejectContract = async (contractId: string) => {
    const reason = prompt('Please provide a reason for rejection:');
    if (!reason) return;

    try {
      // Get current user ID
      let currentUserId = null;
      try {
        currentUserId = localStorage.getItem('sellerId') || localStorage.getItem('user_id');
        
        if (!currentUserId) {
          const sellerToken = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
          if (sellerToken && sellerToken.includes('.')) {
            try {
              const payload = JSON.parse(atob(sellerToken.split('.')[1]));
              currentUserId = payload.user_id || payload.sub || payload.id;
            } catch (e) {
              console.warn('Failed to decode token');
            }
          }
        }
        
        if (!currentUserId) {
          const user = JSON.parse(localStorage.getItem('user') || '{}');
          currentUserId = user.user_id || user.id;
        }
        
        if (!currentUserId) {
          const sellerData = JSON.parse(localStorage.getItem('sellerData') || '{}');
          currentUserId = sellerData.user_id || sellerData.id;
        }
        
        if (!currentUserId) {
          alert('Please log in to reject contracts');
          return;
        }
      } catch (e) {
        console.error('Error getting user ID:', e);
        alert('Failed to get user information');
        return;
      }
      
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/reject?user_id=${currentUserId}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ reason })
      });

      if (response.ok) {
        alert('Contract rejected successfully');
        onRefresh();
      } else {
        const errorData = await response.json();
        alert(`Failed to reject: ${errorData.detail || 'Unknown error'}`);
      }
    } catch (error) {
      console.error('Error rejecting contract:', error);
      alert('Failed to reject contract');
    }
  };

  const handleDeleteContract = async (contractId: string) => {
    if (!confirm('Are you sure you want to delete this contract?')) return;

    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}`, {
        method: 'DELETE',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        onRefresh();
      } else {
        console.error('Failed to delete contract');
      }
    } catch (error) {
      console.error('Error deleting contract:', error);
    }
  };

  const handleCreateEscrow = async (contractId: string) => {
    try {
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/create-escrow`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' }
      });

      if (response.ok) {
        const data = await response.json();
        alert(`Escrow created successfully!\nEscrow Number: ${data.escrow_number}\nAmount: ₹${data.escrow_amount?.toLocaleString()}`);
        onRefresh();
      } else {
        const error = await response.json();
        alert(`Failed to create escrow: ${error.detail}`);
      }
    } catch (error) {
      console.error('Error creating escrow:', error);
      alert('Failed to create escrow');
    }
  };

  const handleDownloadStampPaper = async (contractId: string) => {
    try {
      const token = localStorage.getItem('access_token') || localStorage.getItem('sellerToken');
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/stamp-paper`, {
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });

      if (response.ok) {
        const blob = await response.blob();
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `stamp_paper_contract_${contractId}.pdf`;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
      } else if (response.status === 404) {
        alert('Stamp paper is being generated. Please wait a moment and try again.');
      } else {
        console.error('Failed to download stamp paper');
        alert('Failed to download stamp paper. Please try again later.');
      }
    } catch (error) {
      console.error('Error downloading stamp paper:', error);
      alert('Failed to download stamp paper. Please check your connection.');
    }
  };

  const handleViewStampPaper = async (contractId: string) => {
    setStampPaperContractId(contractId);
    setShowStampPaperViewer(true);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  return (
    <>
      <div className="space-y-4">
        {contracts && contracts.length > 0 ? contracts.map((contract) => (
          <Card key={contract.id} className="hover:shadow-md transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-start justify-between">
                <div className="flex-1">
                  <div className="flex items-center gap-3 mb-2">
                    <h3 className="text-lg font-semibold text-gray-900">{contract.title}</h3>
                    <Badge className={getStatusColor(contract.status)}>
                      <div className="flex items-center gap-1">
                        {getStatusIcon(contract.status)}
                        {contract.status.charAt(0).toUpperCase() + contract.status.slice(1)}
                      </div>
                    </Badge>
                    <Badge variant="outline">{contract.type}</Badge>
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                    <div>
                      <p className="text-sm text-gray-500">Contract ID</p>
                      <p className="font-medium">{contract.id}</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-500">Created</p>
                      <p className="font-medium">{formatDate(contract.created_at)}</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-500">Status</p>
                      <p className="font-medium text-sm">{getStatusMessage(contract.status)}</p>
                    </div>
                  </div>

                  {contract.property && (
                    <div className="mb-3">
                      <div className="flex items-start gap-2">
                        <Home className="w-4 h-4 text-blue-500 mt-0.5" />
                        <div className="flex-1">
                          <p className="text-sm text-gray-500">Property Details</p>
                          <p className="font-medium">{contract.property}</p>
                          {contract.property_location && (
                            <p className="text-sm text-gray-600 mt-1 flex items-center gap-1">
                              <MapPin className="w-3 h-3" />
                              {contract.property_location}
                            </p>
                          )}
                          <div className="flex flex-wrap gap-2 mt-2">
                            {contract.property_type && (
                              <Badge variant="outline" className="text-xs capitalize">{contract.property_type}</Badge>
                            )}
                            {contract.bedrooms && (
                              <Badge variant="outline" className="text-xs">{contract.bedrooms}BHK</Badge>
                            )}
                            {contract.bathrooms && (
                              <Badge variant="outline" className="text-xs">{contract.bathrooms} Bath</Badge>
                            )}
                            {contract.built_up_area && (
                              <Badge variant="outline" className="text-xs">{contract.built_up_area} sq ft</Badge>
                            )}
                          </div>
                          {(contract.sale_price || contract.monthly_rent) && (
                            <p className="text-green-600 font-semibold mt-2 flex items-center gap-1">
                              <DollarSign className="w-4 h-4" />
                              ₹{(contract.sale_price || contract.monthly_rent)?.toLocaleString()}
                              {contract.monthly_rent ? '/month' : ''}
                            </p>
                          )}
                          {contract.property_owner && (
                            <div className="flex items-center gap-2 text-sm text-gray-600 mt-2">
                              <User className="w-3 h-3" />
                              <span>{contract.property_owner}</span>
                              {contract.property_email && (
                                <span className="flex items-center gap-1 ml-2">
                                  <Mail className="w-3 h-3" />
                                  {contract.property_email}
                                </span>
                              )}
                            </div>
                          )}
                        </div>
                      </div>
                    </div>
                  )}

                  <div className="mb-3">
                    <p className="text-sm text-gray-500">Parties</p>
                    <div className="flex flex-wrap gap-1">
                      {(contract.parties || []).map((party, index) => (
                        <Badge key={index} variant="secondary">{party}</Badge>
                      ))}
                    </div>
                  </div>

                  {contract.next_milestone && (
                    <div className="mb-3">
                      <p className="text-sm text-gray-500">Next Milestone</p>
                      <p className="font-medium text-orange-600">{contract.next_milestone}</p>
                    </div>
                  )}
                </div>

                <div className="flex flex-col gap-2 ml-4">
                  <Button 
                    variant="outline" 
                    size="sm"
                    onClick={() => handleViewContract(contract)}
                  >
                    <Eye className="w-4 h-4 mr-1" />
                    View
                  </Button>
                  
                  {/* Show approve button for sellers on draft contracts */}
                  {canUserApprove(contract) && (
                    <>
                      <Button 
                        size="sm"
                        onClick={() => handleApproveContract(contract.id)}
                        style={{ backgroundColor: '#2563eb', color: 'white', border: 'none' }}
                      >
                        ✓ Approve
                      </Button>
                      <Button 
                        size="sm"
                        onClick={() => handleRejectContract(contract.id)}
                        style={{ backgroundColor: '#dc2626', color: 'white', border: 'none' }}
                      >
                        ✕ Reject
                      </Button>
                    </>
                  )}
                  
                  {/* Show sign button for both parties on pending signatures */}
                  {canUserSign(contract) && (
                    <Button 
                      size="sm"
                      onClick={() => handleSignContract(contract.id)}
                      style={{ backgroundColor: '#16a34a', color: 'white', border: 'none' }}
                    >
                      ✍️ Sign
                    </Button>
                  )}
                  
                  {/* View/Download Stamp Paper buttons for signed contracts */}
                  {contract.status === 'signed' && (
                    <>
                      <Button 
                        size="sm"
                        onClick={() => handleViewStampPaper(contract.id)}
                        className="bg-blue-500 hover:bg-blue-600 text-white"
                      >
                        <FileText className="w-4 h-4 mr-1" />
                        View Stamp Paper
                      </Button>
                      <Button 
                        size="sm"
                        onClick={() => handleDownloadStampPaper(contract.id)}
                        className="bg-blue-500 hover:bg-blue-600 text-white"
                      >
                        <Download className="w-4 h-4 mr-1" />
                        Download PDF
                      </Button>
                      <Button 
                        size="sm"
                        onClick={() => handleCreateEscrow(contract.id)}
                        className="bg-green-500 hover:bg-green-600 text-white"
                      >
                         Create Escrow
                      </Button>
                    </>
                  )}
                  
                  {/* Test button - shows status */}
                  <Button 
                    variant="outline" 
                    size="sm"
                    style={{ backgroundColor: '#fef3c7', color: '#92400e', border: '1px solid #f59e0b' }}
                  >
                    Status: {contract.status}
                  </Button>
                  
                  {/* Delete button for draft contracts */}
                  {contract.status === 'draft' && (
                    <Button 
                      variant="outline" 
                      size="sm"
                      onClick={() => handleDeleteContract(contract.id)}
                      style={{ color: '#dc2626', borderColor: '#dc2626' }}
                    >
                      <Trash2 className="w-4 h-4 mr-1" />
                      Delete
                    </Button>
                  )}
                </div>
              </div>
            </CardContent>
          </Card>
        )) : (
          <div className="text-center py-8">
            <p className="text-gray-500">No contracts available</p>
          </div>
        )}
      </div>
      
      {/* Contract Detail Modal */}
      {showDetailModal && selectedContract && (
        <ContractDetailModal 
          contract={selectedContract}
          onClose={() => {
            setShowDetailModal(false);
            setSelectedContract(null);
          }}
        />
      )}
      
      {/* Stamp Paper Viewer Modal */}
      {showStampPaperViewer && stampPaperContractId && (
        <StampPaperViewer 
          isOpen={showStampPaperViewer}
          contractId={stampPaperContractId}
          onClose={() => {
            setShowStampPaperViewer(false);
            setStampPaperContractId(null);
          }}
        />
      )}
      
      {/* Contract Signing Modal */}
      <ContractSigningModal 
        isOpen={showSigningModal}
        contractId={contractToSign || ''}
        onClose={() => {
          setShowSigningModal(false);
          setContractToSign(null);
        }}
        onSigningComplete={() => {
          setShowSigningModal(false);
          setContractToSign(null);
          onRefresh();
        }}
      />
    </>
  );
};